package io.privy.wallet.webview

import kotlinx.serialization.KSerializer
import kotlinx.serialization.descriptors.PrimitiveKind
import kotlinx.serialization.descriptors.PrimitiveSerialDescriptor
import kotlinx.serialization.descriptors.SerialDescriptor
import kotlinx.serialization.encoding.Decoder
import kotlinx.serialization.encoding.Encoder
import kotlin.io.encoding.Base64
import kotlin.io.encoding.ExperimentalEncodingApi

/**
 * A [KSerializer] for [ByteArray] that encodes to and decodes from a Base64 [String].
 * This is used to ensure byte arrays are consistently represented as Base64 strings in JSON.
 */
@OptIn(ExperimentalEncodingApi::class)
internal object Base64ByteArraySerializer : KSerializer<ByteArray> {
    override val descriptor: SerialDescriptor =
        PrimitiveSerialDescriptor("Base64ByteArray", PrimitiveKind.STRING)

    /**
     * Serializes the given [ByteArray] into a Base64 encoded [String].
     */
    override fun serialize(encoder: Encoder, value: ByteArray) {
        val encoded = Base64.encode(value)
        encoder.encodeString(encoded)
    }

    /**
     * Deserializes a Base64 encoded [String] back into a [ByteArray].
     */
    override fun deserialize(decoder: Decoder): ByteArray {
        val decoded = Base64.decode(decoder.decodeString())
        return decoded
    }
}
