package io.privy.network

import io.ktor.client.HttpClient
import io.ktor.client.plugins.HttpTimeout
import io.ktor.client.plugins.api.ClientPlugin
import io.ktor.client.plugins.api.createClientPlugin
import io.ktor.client.plugins.contentnegotiation.ContentNegotiation
import io.ktor.client.plugins.defaultRequest
import io.ktor.client.plugins.resources.Resources
import io.ktor.client.request.accept
import io.ktor.client.request.headers
import io.ktor.client.statement.request
import io.ktor.http.ContentType
import io.ktor.http.contentType
import io.ktor.serialization.kotlinx.json.json
import io.privy.di.AppBundleIdentifier
import io.privy.di.KmpAppScope
import io.privy.di.PrivyAppClientId
import io.privy.di.PrivyAppId
import io.privy.logging.PrivyLogger
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.json.Json
import me.tatarka.inject.annotations.Provides

public interface NetworkBindings {
  @KmpAppScope
  @Provides
  public fun provideKtorProvider(
      privyEnvironment: PrivyEnvironment,
      logger: PrivyLogger,
      privyAppId: PrivyAppId,
      privyAppClientId: PrivyAppClientId,
      appBundleIdentifier: AppBundleIdentifier,
  ): KtorProvider {
    return KtorProvider(
        httpClient =
            createKtorHttpClient(
                privyEnvironment = privyEnvironment,
                customLoggerPlugin = createLoggerPlugin(logger),
                privyAppId = privyAppId,
                privyAppClientId = privyAppClientId,
                appBundleIdentifier = appBundleIdentifier,
            ))
  }

  // By using a custom plugin for logging, we can use PrivyLogger to adhere
  // to specified log levels
  public fun createLoggerPlugin(logger: PrivyLogger): ClientPlugin<Unit> {
    return createClientPlugin("CustomLoggerPlugin") {
      onRequest { request, content ->
        val message =
            """
          ============= API REQUEST ==============
          ${request.method.value} => ${request.url}
          HEADERS => ${request.headers.build()}
          BODY => ${request.body}
          =============END API REQUEST============
        """
                .trimIndent()

        logger.debug(message)
      }

      onResponse { response ->
        val message =
            """
          ============= API RESPONSE ==============
          [${response.request.method.value}] ${response.request.url}
          STATUS => ${response.status}
          BODY => $response
          ============= END API RESPONSE===========
        """
                .trimIndent()

        logger.debug(message)
      }
    }
  }

  @OptIn(ExperimentalSerializationApi::class)
  private fun createKtorHttpClient(
      privyEnvironment: PrivyEnvironment,
      customLoggerPlugin: ClientPlugin<Unit>,
      privyAppId: PrivyAppId,
      privyAppClientId: PrivyAppClientId,
      appBundleIdentifier: AppBundleIdentifier,
  ): HttpClient {
    val httpClient = HttpClient {
      // Throws an error on HTTP responses if not 2XX
      expectSuccess = true

      // Set timeouts to 60 seconds
      install(HttpTimeout) {
        socketTimeoutMillis = 60_000
        requestTimeoutMillis = 60_000
      }

      // Add support for type safe requests!
      install(Resources)

      // Add custom logger that adheres to PrivyLogLevel
      install(customLoggerPlugin)

      defaultRequest {
        // Specify base url for all requests
        url(privyEnvironment.baseUrl())

        // Specify Json for "Content-Type" and "Accept" headers
        contentType(ContentType.Application.Json)
        accept(ContentType.Application.Json)

        // Add privy app id as header to all requests
        headers {
          append(HEADER_PRIVY_APP_ID, privyAppId)

          append(HEADER_PRIVY_CLIENT_ID, privyAppClientId)

          append(HEADER_NATIVE_APP_IDENTIFIER, appBundleIdentifier)
        }
      }

      // Add support for type safe response handling
      install(ContentNegotiation) {
        json(
            Json {
              prettyPrint = true
              isLenient = true
              ignoreUnknownKeys = true
              explicitNulls = false
            },
        )
      }
    }

    return httpClient
  }
}

private fun PrivyEnvironment.baseUrl(): String {
  return when (this) {
    PrivyEnvironment.Staging -> STAGING_URL
    PrivyEnvironment.Production -> PRODUCTION_URL
  }
}

private const val PRODUCTION_URL = "https://auth.privy.io/api/v1/"
private const val STAGING_URL = "https://auth.staging.privy.io/api/v1/"
private const val HEADER_PRIVY_APP_ID = "privy-app-id"
private const val HEADER_PRIVY_CLIENT_ID = "privy-client-id"
private const val HEADER_NATIVE_APP_IDENTIFIER = "x-native-app-identifier"
