package io.privy.network

import io.ktor.client.HttpClient
import io.ktor.client.call.body
import io.ktor.client.plugins.ClientRequestException
import io.ktor.client.plugins.resources.delete
import io.ktor.client.plugins.resources.get
import io.ktor.client.plugins.resources.post
import io.ktor.client.request.HttpRequestBuilder
import io.ktor.client.request.post
import io.ktor.client.request.setBody
import io.ktor.client.statement.HttpResponse
import kotlinx.serialization.SerializationException

public suspend inline fun <reified T : Any, reified R : Any> HttpClient.getResult(
    resource: T,
    builder: HttpRequestBuilder.() -> Unit = {}
): ApiResult<R> = requestCatching { get(resource = resource, builder = builder) }

public suspend inline fun <reified T : Any, reified B, reified R : Any> HttpClient.postResult(
    resource: T,
    body: B,
    builder: HttpRequestBuilder.() -> Unit = {}
): ApiResult<R> = requestCatching {
  post(resource) {
    builder()
    setBody(body)
  }
}

public suspend inline fun <reified B, reified R : Any> HttpClient.postResult(
    url: String,
    body: B,
    builder: HttpRequestBuilder.() -> Unit = {}
): ApiResult<R> = requestCatching {
  post(url) {
    builder()
    setBody(body)
  }
}

public suspend inline fun <reified T : Any, reified R : Any> HttpClient.deleteResult(
    resource: T,
    builder: HttpRequestBuilder.() -> Unit = {},
): ApiResult<R> {
  return requestCatching { delete(resource) { builder() } }
}

/**
 * Takes in a `block` lambda that returns an HttpResponse from Ktor Internally, tries to convert the
 * response into a response body defined by the generic R If success (2XX response), return
 * ApiResult.Success If error (3XX - 5XX response), try to figure out error type and return
 * ApiResult.Failure
 */
public suspend inline fun <reified R : Any> requestCatching(
    block: () -> HttpResponse
): ApiResult<R> {
  return try {
    val response = block()

    response
        .safeBody<R>()
        .fold(
            onSuccess = { responseBody -> ApiResult.Success(responseBody) },
            onFailure = { throwable ->
                ApiResult.Error.GenericError(
                    exception = throwable,
                    humanReadableMessage = "The was an issue decoding the network response.",
                )
            })
  } catch (exception: ClientRequestException) {
    val responseBody = exception.response.safeBody<PrivyHttpErrorResponse?>().getOrNull()

      ApiResult.Error.HttpError(
          code = exception.response.status.value,
          responseBody = responseBody,
          exception = exception,
          humanReadableMessage = responseBody?.error ?: GENERIC_ERROR_MESSAGE,
      )
  } catch (exception: HttpExceptions) {
    val responseBody = exception.response.safeBody<PrivyHttpErrorResponse?>().getOrNull()

      ApiResult.Error.HttpError(
          code = exception.response.status.value,
          responseBody = responseBody,
          exception = exception,
          humanReadableMessage = responseBody?.error ?: GENERIC_ERROR_MESSAGE,
      )
  } catch (exception: SerializationException) {
      ApiResult.Error.SerializationError(
          exception = exception,
          humanReadableMessage = "Something went wrong while serializing the response",
      )
  } catch (exception: Exception) {
      ApiResult.Error.GenericError(
          exception = exception,
          humanReadableMessage = GENERIC_ERROR_MESSAGE,
      )
  }
}

// Helper method to safely convert response to body type with ContentNegotiation, since it
// can throw NoTransformationFoundException if the specified body type doesn't match response
// structure
public suspend inline fun <reified T> HttpResponse.safeBody(): Result<T> {
  return runCatching { body<T>() }
}

public const val GENERIC_ERROR_MESSAGE: String = "Something went wrong"
