package io.privy.network

import kotlinx.coroutines.flow.Flow

public interface NetworkStateManager {
  /**
   * Returns user's network connectivity status
   *
   * @return [NetworkState] indicating device's network state.
   * Will return [NetworkState.Unknown] if proper permissions were not added to Android manifest.
   */
  public val current: NetworkState

  /**
   * Exposes a flow that allows the caller to subscribe to NetworkState updates
   */
  public fun observeNetworkChanges(): Flow<NetworkState>

  /**
   * Exposes a flow that allows the caller to subscribe to NetworkState updates
   */
  public fun observeNetworkRestored(): Flow<Unit>
}

public inline fun <reified R : Any> NetworkStateManager.failIfNetworkConfirmedDisconnectedElse(
  block: () -> Result<R>
): Result<R> {
  // isConfirmedDisconnected only returns true if we can confirm network state is disconnected
  return if (current.isConfirmedDisconnected()) {
    Result.failure(NoNetworkException)
  } else {
    block()
  }
}

public inline fun <reified R : Any> NetworkState.failIfNetworkConfirmedDisconnectedElse(
  block: () -> Result<R>
): Result<R> {
  // isConfirmedDisconnected only returns true if we can confirm network state is disconnected
  return if (isConfirmedDisconnected()) {
    Result.failure(NoNetworkException)
  } else {
    block()
  }
}