package io.privy.network

import io.ktor.client.request.HttpRequestBuilder
import io.ktor.client.request.headers
import io.privy.BuildKonfig
import io.privy.analytics.ClientAnalyticsIdRepository
import kotlin.jvm.JvmInline
import me.tatarka.inject.annotations.Inject

@Inject
public class KtorWrapper(
  private val privyInternalSettings: PrivyInternalSettings?,
  public val ktorProvider: KtorProvider,
  private val clientAnalyticsIdRepository: ClientAnalyticsIdRepository,
  public val networkStateManager: NetworkStateManager,
) {

  public suspend inline fun <reified T : Any, reified R : Any> getResult(
      resource: T,
      authType: AuthType,
      builder: HttpRequestBuilder.() -> Unit = {}
  ): ApiResult<R> {
    return ensureDeviceConnectedToInternet {
      ktorProvider.httpClient.getResult(resource) {
        builder()
        addAdditionalHeaders(authType)
      }
    }
  }

  public suspend inline fun <reified T : Any, reified B, reified R : Any> postResult(
      resource: T,
      body: B,
      authType: AuthType,
      builder: HttpRequestBuilder.() -> Unit = {}
  ): ApiResult<R> {
    return ensureDeviceConnectedToInternet {
      ktorProvider.httpClient.postResult(resource, body) {
        builder()
        addAdditionalHeaders(authType)
      }
    }
  }

  public suspend inline fun <reified B, reified R : Any> postResult(
      url: String,
      body: B,
      authType: AuthType,
  ): ApiResult<R> {
    return ensureDeviceConnectedToInternet {
      ktorProvider.httpClient.postResult(url, body) { addAdditionalHeaders(authType) }
    }
  }

  public suspend inline fun <reified T : Any, reified R : Any> deleteResult(
      resource: T,
      authType: AuthType,
  ): ApiResult<R> {
    return ensureDeviceConnectedToInternet {
      ktorProvider.httpClient.deleteResult(resource) { addAdditionalHeaders(authType) }
    }
  }

  public suspend fun HttpRequestBuilder.addAdditionalHeaders(authType: AuthType) {
    // analytics id header
    addClientAnalyticsIdHeader()

    // privy client + sdk version header
    addPrivyClientHeader()

    // Auth header
    when (authType) {
      AuthType.None -> {}
      is AuthType.Authorization -> addAuthorizationHeader(authType.accessToken)
    }
  }

  private suspend fun HttpRequestBuilder.addClientAnalyticsIdHeader() {
    val clientAnalyticsId = clientAnalyticsIdRepository.loadClientAnalyticsId()
    headers { append(HEADER_PRIVY_CLIENT_ANALYTICS_ID, clientAnalyticsId) }
  }

  private fun HttpRequestBuilder.addPrivyClientHeader() {
    val androidSdkVersion = BuildKonfig.sdkVersion

    val sdkVersionString =
        if (privyInternalSettings == null) {

          "android:$androidSdkVersion"
        } else {
          val platform = privyInternalSettings.platform
          val platformSdkVersion = privyInternalSettings.sdkVersion
          "$platform-android:$platformSdkVersion"
        }

    headers { append(HEADER_PRIVY_CLIENT, sdkVersionString) }
  }

  private fun HttpRequestBuilder.addAuthorizationHeader(accessToken: String) {
    headers { append(HEADER_AUTHORIZATION, "Bearer $accessToken") }
  }

  public inline fun <reified R : Any> ensureDeviceConnectedToInternet(
    block: () -> ApiResult<R>
  ): ApiResult<R> {
    // Before executing any HTTP request, confirm device is connected to the internet
    return if (networkStateManager.current.isConfirmedDisconnected()) {
      ApiResult.Error.NoNetworkError
    } else {
      block()
    }
  }
}

public sealed interface AuthType {
  public data object None : AuthType

  // Ideally 1 day this module can grab the access token itself
  // rather than having it passed into method calls
  // difficult to do this today because a dependency on auth module results in
  // circular dependency
  @JvmInline public value class Authorization(public val accessToken: String) : AuthType

}
// Define all header constants here or in a shared constants file
private const val HEADER_PRIVY_CLIENT_ANALYTICS_ID = "privy-ca-id"
private const val HEADER_PRIVY_CLIENT = "privy-client"
private const val HEADER_AUTHORIZATION = "Authorization"

