package io.privy.network

import io.ktor.client.plugins.ResponseException
import io.ktor.client.statement.HttpResponse
import kotlinx.serialization.Serializable

// Wrapper class to help Swift interop.
// Issue: Kotlin Result<T> is exposed with type Any? to swift.
public sealed class ApiResult<out T : Any> {
  public data class Success<out T : Any>(val value: T) : ApiResult<T>()

  public sealed class Error : ApiResult<Nothing>() {
    public abstract val exception: Throwable
    public abstract val humanReadableMessage: String

    /**
     * Represents server errors.
     *
     * @param code HTTP Status code
     * @param responseBody Response body
     */
    public data class HttpError(
      val code: Int,
      val responseBody: PrivyHttpErrorResponse?,
      override val exception: Throwable,
      override val humanReadableMessage: String,
    ) : Error()

    /**
     * Represent SerializationExceptions.
     *
     * @param message Detail exception message
     * @param exceptionMessage Formatted error message
     */
    public data class SerializationError(
        override val exception: Throwable,
        override val humanReadableMessage: String,
    ) : Error()

    /**
     * Represent other exceptions.
     *
     * @param message Detail exception message
     * @param exceptionMessage Formatted error message
     */
    public data class GenericError(
        override val exception: Throwable,
        override val humanReadableMessage: String,
    ) : Error()
  }
}

@Serializable
public data class PrivyHttpErrorResponse(
  val code: String,
  val error: String,
)

public class HttpExceptions(
    response: HttpResponse,
    failureReason: String?,
    cachedResponseText: String,
) : ResponseException(response, cachedResponseText) {
  override val message: String = "Status: ${response.status}" + " Failure: $failureReason"
}

// Returns the encapsulated result of the given transform function applied to the encapsulated value
// if this instance represents success or the original encapsulated Throwable exception
// if it is failure.
public inline fun <reified R : Any, reified T : Any> ApiResult<T>.map(
    transform: (value: T) -> R?
): ApiResult<R> {
  return when (this) {
    is ApiResult.Success -> {
      val transformResult = transform(value)
      if (transformResult == null) {
        val message = "Failed to map ${T::class.simpleName} to ${R::class.simpleName}"
        return ApiResult.Error.SerializationError(
            exception = Throwable(message),
            humanReadableMessage = message
        )
      }
        ApiResult.Success(transformResult)
    }
    is ApiResult.Error -> this
  }
}

public fun <T: Any> ApiResult<T>.toResult(): Result<T> {
  return when (this) {
    is ApiResult.Success -> Result.success(this.value)
    is ApiResult.Error -> Result.failure(
      exception = PrivyApiException(
        statusCode = (this as? ApiResult.Error.HttpError)?.code,
        responseBody = (this as? ApiResult.Error.HttpError)?.responseBody,
        message = this.humanReadableMessage,
        cause = this.exception,
      )
    )
  }
}