package io.privy.sdk

import android.content.Context
import io.privy.analytics.AnalyticsEvent
import io.privy.auth.AuthState
import io.privy.auth.PrivyUser
import io.privy.auth.customAuth.LoginWithCustomAuth
import io.privy.auth.email.LoginWithEmail
import io.privy.auth.internal.InternalAuthState
import io.privy.auth.sms.LoginWithSms
import io.privy.network.PrivyEnvironment
import io.privy.sdk.di.PrivyCoreComponent
import io.privy.sdk.webview.RealWebViewHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch

internal class PrivyImpl(
  context: Context,
  config: PrivyConfig,
) : Privy {
  private val scope = CoroutineScope(SupervisorJob() + Dispatchers.Main.immediate)

  private val privyEnvironment: PrivyEnvironment = if (config.appId in internalAppIds) {
    PrivyEnvironment.Staging
  } else {
    PrivyEnvironment.Production
  }

  private val webViewHandler = RealWebViewHandler(
    context = context,
    privyEnvironment = privyEnvironment,
    appId = config.appId,
    appClientId = config.appClientId,
  )

  // Initialize DI graph
  private val coreComponent: PrivyCoreComponent =
    PrivyCoreComponent.create(
      context = context,
      privyAppId = config.appId,
      privyAppClientId = config.appClientId,
      privyLogLevel = config.logLevel,
      appBundleIdentifier = context.packageName,
      webViewHandler = webViewHandler,
      privyEnvironment = privyEnvironment,
    )

  private val authManager = coreComponent.kmpComponent.authManager
  private val authStateRepository = coreComponent.kmpComponent.authStateRepository

  // We determine that Privy is ready for use after auth state has been set
  // IMPORTANT - the execution happens in the getter "get() = " so that it's executed
  // each time the isReady variable is read
  override val isReady: Boolean
    get() = authState.value.hasBeenSet()

  // Only expose user object is there is an authenticated session
  override val user: PrivyUser?
    get() {
      val currentAuthState = authState.value
      return if (currentAuthState is AuthState.Authenticated) {
        // if there is an authenticated session return a user
        currentAuthState.user
      } else {
        // if no session, return null
        null
      }
    }

  // Automatically update external auth state on internal auth state updates
  override val authState: StateFlow<AuthState> = authStateRepository.internalAuthState.map { internalAuthState ->
      when(internalAuthState) {
        InternalAuthState.NotReady -> AuthState.NotReady
        // If authenticated, expose user object
        is InternalAuthState.Authenticated -> AuthState.Authenticated(
          user = authManager.privyUser
        )
        InternalAuthState.Unauthenticated -> {
          AuthState.Unauthenticated
        }
      }
  }.stateIn(
    scope = scope,
    started = SharingStarted.Eagerly,
    initialValue = AuthState.NotReady
  )

  override val sms: LoginWithSms = coreComponent.kmpComponent.loginWithSms

  override val email: LoginWithEmail = coreComponent.kmpComponent.loginWithEmail

  override val customAuth: LoginWithCustomAuth = coreComponent.kmpComponent.loginWithCustomAuth

  init {
    // If tokenProvider is provided, pass it to custom auth flow
    if (config.customAuthConfig != null) {
      customAuth.setTokenProvider(
        tokenProvider = config.customAuthConfig.tokenProvider
      )
    }

    logSdkInitializeEvent()
  }

  override suspend fun awaitReady() {
    this.authState.awaitReady()
  }

  override suspend fun logout() {
    coreComponent.kmpComponent.clientAnalyticsIdRepository.clearClientAnalyticsId()
    coreComponent.kmpComponent.authManager.logout()
  }

  private fun logSdkInitializeEvent() {
    scope.launch {
      coreComponent.kmpComponent.analyticsManager.logEvent(AnalyticsEvent.SdkInitialize)
    }
  }
}

private fun AuthState.hasBeenSet(): Boolean {
  // when auth state has been updated from "NotReady", auth state was set
  return this !is AuthState.NotReady
}

private suspend fun StateFlow<AuthState>.awaitReady() {
  // Check if the auth state has already been set (non-suspending)
  if (this.value.hasBeenSet()) return

  // Suspend execution until auth state has been set
  this.filter { it.hasBeenSet() }.first()
}

private val internalAppIds = listOf(
  "clpijy3tw0001kz0g6ixs9z15"
)