package io.privy.sdk.webview

import android.annotation.SuppressLint
import android.content.Context
import android.util.Log
import android.webkit.WebView
import android.webkit.WebViewClient
import io.privy.auth.EmbeddedWalletException
import io.privy.network.PrivyEnvironment
import io.privy.wallet.EmbeddedWalletDetails
import io.privy.wallet.ethereum.EthereumRpcRequest
import io.privy.wallet.solana.SolanaRpcRequest
import io.privy.wallet.solana.SolanaSignMessageResponse
import io.privy.wallet.webview.IFrameResponse
import io.privy.wallet.webview.PrivyEventType
import io.privy.wallet.webview.PrivyIFrameErrorResponse
import io.privy.wallet.webview.PrivyIFrameRequest
import io.privy.wallet.webview.PrivyIFrameSuccessResponse
import io.privy.wallet.webview.SolanaSignMessageResponseData
import io.privy.wallet.webview.WebViewHandler
import java.util.UUID
import kotlin.coroutines.cancellation.CancellationException
import kotlin.time.Duration
import kotlin.time.Duration.Companion.milliseconds
import kotlin.time.Duration.Companion.seconds
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.channels.Channel
import kotlinx.coroutines.launch
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlinx.coroutines.withContext
import kotlinx.coroutines.withTimeout
import kotlinx.serialization.ExperimentalSerializationApi
import kotlinx.serialization.Serializable
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.contentOrNull
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive

// TODO: list of things to do before finishing
// Make this class injectable?
// wallet needs to be recovered every time, do we need to enable storage or something?

@SuppressLint("SetJavaScriptEnabled")
internal class RealWebViewHandler(
    context: Context,
    privyEnvironment: PrivyEnvironment,
    appId: String,
    appClientId: String,
) : WebViewHandler {
  private val webView: WebView = WebView(context)
  private val baseUrl =
      if (privyEnvironment == PrivyEnvironment.Staging) {
        "https://auth.staging.privy.io"
      } else {
        "https://auth.privy.io"
      }
  private val webViewUrl = "$baseUrl/apps/$appId/embedded-wallets?client_id=$appClientId"

  // Create a coroutine scope for Privy to use.
  // SupervisorJob offers robustness so that if a child coroutine fails, the parent doesn't
  // Dispatchers.IO offloads execution to a background thread
  private val scope = CoroutineScope(SupervisorJob() + Dispatchers.IO)

  private val webViewState = WebViewState()

  private val pingReadyMutex = Mutex()

  private val requestToStringResponseChannelMap: MutableMap<String, Channel<String>> =
      mutableMapOf()

  // Only print logs for internal builds
  private val printLogs = privyEnvironment == PrivyEnvironment.Staging

  @OptIn(ExperimentalSerializationApi::class)
  private val json = Json {
    explicitNulls = false
    ignoreUnknownKeys = true
  }

  init {
    // TODO: REMOVE THIS!!
    // WebView.setWebContentsDebuggingEnabled(true);

    webView.settings.javaScriptEnabled = true
    webView.settings.domStorageEnabled = true

    // this allows javascript to call a native method. in this case, when JS calls
    // JS_INTERFACE_NAME.postMessage, it would trigger PrivyJavascriptInterface.postMessage
    webView.addJavascriptInterface(
        PrivyJavascriptInterface { message -> this.onWebViewMessageReceived(message = message) },
        JS_INTERFACE_NAME)

    webView.webViewClient =
        object : WebViewClient() {
          override fun onPageFinished(view: WebView?, url: String?) {
            // Callback triggered when the page finishes loading
            printLog("Page finished loading: $url")

            // When page is finish loading, set window.PRIVY_NATIVE_ANDROID to true
            webView.evaluateJavascript(scriptSource, null)

            // Attempt ready ping until successful
            pingReadyUntilSuccessful()
          }
        }

    subscribeToWebViewReady()

    // Load URL, should trigger "onPageFinished" on complete.
    webView.loadUrl(webViewUrl)
  }

  private fun subscribeToWebViewReady() {
    // Just for debugging purposes, ensures web view ready works.
    scope.launch {
      // simulate ready ping delay
      webViewState.awaitReady()
      printLog("Webview is ready!")
    }
  }

  // TODO, do we want to add a timeout to this attempt instead of each individual ping?
  private fun pingReadyUntilSuccessful() {
    scope.launch {
      // Lock this portion of code with a mutex so it's not executed multiple times
      pingReadyMutex.withLock {
        // when mutex is unlocked, we want to double check if it's ready before attempting
        if (webViewState.isReady()) {
          // already ready, return
          return@launch
        }

        // Keep pinging ready until we get a response!
        // Internally, pingReady will return an error if we don't get a response within 150MS
        while (!webViewState.isReady()) {
          val result = pingReady()

          if (result is IFrameResponse.Success) {
            webViewState.setReady(true)
          }
        }
      }
    }
  }

  private suspend fun pingReady(): IFrameResponse<PrivyIFrameSuccessResponse.IFrameReadyResponse> {
    // technically this request doesn't have an associated type for PrivyIFrameRequest
    // so setting it to String?. Can't set to Any since it doesn't have a serializer
    val readyRequest =
        PrivyIFrameRequest<String?>(
            id = UUID.randomUUID().toString().lowercase(),
            event = PrivyEventType.Ready,
            data = null)

    // We can expect to receive ready ping response within 100 milliseconds,
    // so setting timeout to 150 milliseconds is plenty.
    return performRequest(request = readyRequest, timeout = 150.milliseconds)
  }

  override suspend fun createEthereumWallet(
      accessToken: String,
      existingSolanaWalletAddress: String?,
  ): IFrameResponse<PrivyIFrameSuccessResponse.CreateEthereumWalletResponse> {
    val createEthereumWalletWalletRequest =
        PrivyIFrameRequest(
            id = UUID.randomUUID().toString().lowercase(),
            event = PrivyEventType.CreateEthereumWallet,
            data =
                CreateEthereumWalletRequestData(
                    accessToken = accessToken, solanaAddress = existingSolanaWalletAddress))

    return performRequest(createEthereumWalletWalletRequest, timeout = 60.seconds)
  }

  override suspend fun createAdditionalWallet(
      accessToken: String,
      primaryWalletAddress: String,
      hdWalletIndex: Int,
  ): IFrameResponse<PrivyIFrameSuccessResponse.CreateAdditionalWalletResponse> {
    val createAdditionalWalletRequest =
        PrivyIFrameRequest(
            id = UUID.randomUUID().toString().lowercase(),
            event = PrivyEventType.CreateAdditional,
            data =
                CreateAdditionalWalletRequestData(
                    accessToken = accessToken,
                    primaryWalletAddress = primaryWalletAddress,
                    hdWalletIndex = hdWalletIndex))

    return performRequest(createAdditionalWalletRequest, timeout = 60.seconds)
  }

  override suspend fun createSolanaWallet(
      accessToken: String,
      existingEthereumWalletAddress: String?,
  ): IFrameResponse<PrivyIFrameSuccessResponse.CreateSolanaWalletResponse> {
    val createSolanaWalletRequest =
        PrivyIFrameRequest(
            id = UUID.randomUUID().toString().lowercase(),
            event = PrivyEventType.CreateSolanaWallet,
            data =
                CreateSolanaWalletRequestData(
                    accessToken = accessToken, ethereumAddress = existingEthereumWalletAddress))

    return performRequest(createSolanaWalletRequest, timeout = 60.seconds)
  }

  override suspend fun connectWallet(
      accessToken: String,
      embeddedWalletDetails: EmbeddedWalletDetails
  ): IFrameResponse<PrivyIFrameSuccessResponse.ConnectWalletResponse> {
    val connectWalletRequest =
        PrivyIFrameRequest(
            id = UUID.randomUUID().toString().lowercase(),
            event = PrivyEventType.Connect,
            data =
                ConnectWalletRequestData(
                    accessToken = accessToken,
                    embeddedWalletDetails = embeddedWalletDetails,
                ))

    printLog("Attempting to connect to wallet!!")

    return performRequest(connectWalletRequest)
  }

  override suspend fun recoverWallet(
      accessToken: String,
      embeddedWalletDetails: EmbeddedWalletDetails
  ): IFrameResponse<PrivyIFrameSuccessResponse.RecoverWalletResponse> {
    val recoverWalletRequest =
        PrivyIFrameRequest(
            id = UUID.randomUUID().toString().lowercase(),
            event = PrivyEventType.Recover,
            data =
                RecoverWalletRequestData(
                    accessToken = accessToken,
                    entropyId = embeddedWalletDetails.primaryWalletDetails.entropyId,
                    entropyIdVerifier =
                        embeddedWalletDetails.primaryWalletDetails.entropyIdVerifier,
                ))

    printLog("Attempting to recover wallet")

    return performRequest(recoverWalletRequest)
  }

  override suspend fun ethereumRpc(
      accessToken: String,
      embeddedWalletDetails: EmbeddedWalletDetails,
      request: EthereumRpcRequest
  ): IFrameResponse<PrivyIFrameSuccessResponse.WalletRpcResponse<String>> {
    val rpcRequest =
        PrivyIFrameRequest(
            id = UUID.randomUUID().toString().lowercase(),
            event = PrivyEventType.Rpc,
            data =
                RpcRequestData(
                    accessToken = accessToken,
                    embeddedWalletDetails = embeddedWalletDetails,
                    request =
                        EthereumRpcRequestBody(method = request.method, params = request.params)))

    return performRequest(rpcRequest)
  }

  override suspend fun solanaRpc(
      accessToken: String,
      embeddedWalletDetails: EmbeddedWalletDetails,
      request: SolanaRpcRequest
  ): Result<SolanaSignMessageResponse> {
    val rpcRequest =
        PrivyIFrameRequest(
            id = UUID.randomUUID().toString().lowercase(),
            event = PrivyEventType.Rpc,
            data =
                RpcRequestData(
                    accessToken = accessToken,
                    embeddedWalletDetails = embeddedWalletDetails,
                    request = request.rpcRequest()))

    return when (request) {
      is SolanaRpcRequest.SignMessageParams -> {

        when (val response:
            IFrameResponse<
                PrivyIFrameSuccessResponse.WalletRpcResponse<SolanaSignMessageResponseData>> =
            performRequest(rpcRequest)) {
          is IFrameResponse.Success -> {
            Result.success(
                SolanaSignMessageResponse(signature = response.value.data.response.data.signature))
          }
          is IFrameResponse.Error -> {
            // Some other error - surface it as failure
            val errorMessage = response.message
            Result.failure(EmbeddedWalletException(errorMessage))
          }
        }
      }
    }
  }

  private suspend inline fun <reified T, reified R : PrivyIFrameSuccessResponse> performRequest(
      request: PrivyIFrameRequest<T>,
      timeout: Duration = 30.seconds
  ): IFrameResponse<R> {
    if (request.event != PrivyEventType.Ready) {
      // for all requests except for "ping ready", we need to await until web view is ready
      // todo: right now we only attempt calling pingReadyUntilSuccessful, is that enough?
      webViewState.awaitReady()
    }

    val serializedRequest = request.stringData()

    // Should never happen, but in case request didn't serialize properly
    if (serializedRequest == null) {
      // Need to find better return type here
      printErrorLog("Couldn't serialize iframe request.")

      return IFrameResponse.Error(
          type = "request_not_serializable",
          message = "The privy IFrame request type is not serializable.")
    }

    // Specify timeout for webview request, defaults to 15 seconds
    return withTimeout(timeout) {
      try {
        sendRequestToWebView(
            requestId = request.id,
            serializedRequest = serializedRequest,
        )
      } catch (e: CancellationException) {
        IFrameResponse.Error(
            type = "request_timed_out", message = "The privy IFrame request timed out.")
      }
    }
  }

  private suspend inline fun <reified T : PrivyIFrameSuccessResponse> sendRequestToWebView(
      requestId: String,
      serializedRequest: String,
  ): IFrameResponse<T> {
    // Behavior: No buffer. The sender suspends until a receiver is ready, and the receiver suspends
    // until a sender is ready. It's a direct hand-off.
    val responseChannel = Channel<String>()
    requestToStringResponseChannelMap[requestId] = responseChannel

    // VERY IMPORTANT. SINCE THIS MAY BE LAUNCHED IN COROUTINE SCOPE THAT USES DEFAULT DISPATCHER
    // NEED TO SWITCH CONTEXT TO MAIN THREAD FOR WEBVIEW INTERACTIONS
    withContext(Dispatchers.Main) {
      val jsDispatchEvent =
          "window.dispatchEvent(new MessageEvent('message', { data: $serializedRequest}));"
      printLog("Dispatching event: $jsDispatchEvent")

      webView.evaluateJavascript(jsDispatchEvent) { receivedValue ->
        printLog("Performed privy i frame request received value: $receivedValue")
      }
    }

    // Suspends until channel receives a result
    val result = responseChannel.receive()

    printLog("Channel received result! $result")

    return decodeWebViewMessage<T>(result)
  }

  private fun onWebViewMessageReceived(message: String) {
    scope.launch {
      try {
        val responseJson = Json.parseToJsonElement(message)
        val id = responseJson.jsonObject["id"]?.jsonPrimitive?.contentOrNull

        val channel = requestToStringResponseChannelMap[id]

        if (channel == null) {
          printErrorLog("There is no associated request to the id specified in response.")
          return@launch
        }

        channel.send(message)
      } catch (e: Exception) {
        printLog("response decoding error. $e")
      }
    }
  }

  private inline fun <reified T : PrivyIFrameSuccessResponse> decodeWebViewMessage(
      message: String
  ): IFrameResponse<T> {
    return try {
      if (message.contains("error")) {
        val error = json.decodeFromString<PrivyIFrameErrorResponse>(message)
        printErrorLog("Error received in post message! $error")
        IFrameResponse.Error(
            type = error.error.type,
            message = error.error.message,
        )
      } else {
        val success = json.decodeFromString<T>(message)
        printLog("Success received in post message! $success")
        IFrameResponse.Success(success)
      }
    } catch (e: Exception) {
      printErrorLog(
          "We don't support the response type structure in the received message! Message: $message.\nError: ${e.message}")

      return IFrameResponse.Error(
          type = "response_not_deserializable",
          message = "The iFrame response couldn't be deserialized",
      )
    }
  }

  private inline fun <reified T> PrivyIFrameRequest<T>.stringData(): String? {
    return try {
      json.encodeToString(this)
    } catch (e: Exception) {
      printErrorLog("Error encoding iframe request: $e")
      null // Handle encoding errors gracefully
    }
  }

  private fun printLog(message: String) {
    if (printLogs) {
      Log.i("PrivyWebView", message)
    }
  }

  private fun printErrorLog(message: String) {
    if (printLogs) {
      Log.e("PrivyWebView", message)
    }
  }

  private companion object {
    const val JS_INTERFACE_NAME = "AndroidProxy"
    const val scriptSource = "window.PRIVY_NATIVE_ANDROID = true;"
  }
}

@Serializable
private data class CreateEthereumWalletRequestData(
    val accessToken: String,

    /**
     * The address of the user's existing Solana embedded wallet.
     *
     * If an existing address is passed, the iframe will re-assemble the entropy for the existing
     * address to derive the Ethereum wallet. If no existing address is passed, the iframe will
     * generate new entropy for the Ethereum wallet.
     */
    val solanaAddress: String?
)

@Serializable
private data class CreateSolanaWalletRequestData(
    val accessToken: String,

    /**
     * The address of the user's existing Ethereum embedded wallet.
     *
     * If an existing address is passed, the iframe will re-assemble the entropy for the existing
     * address to derive the Ethereum wallet. If no existing address is passed, the iframe will
     * generate new entropy for the Ethereum wallet.
     */
    val ethereumAddress: String?
)

@Serializable
private data class CreateAdditionalWalletRequestData(
    val accessToken: String,
    val primaryWalletAddress: String,
    val hdWalletIndex: Int
)

@Serializable
private data class ConnectWalletRequestData(
    val accessToken: String,
    val chainType: String,
    val entropyId: String,
    val entropyIdVerifier: String,
) {
  constructor(
      accessToken: String,
      embeddedWalletDetails: EmbeddedWalletDetails
  ) : this(
      accessToken = accessToken,
      chainType = embeddedWalletDetails.chainType,
      entropyId = embeddedWalletDetails.primaryWalletDetails.entropyId,
      entropyIdVerifier = embeddedWalletDetails.primaryWalletDetails.entropyIdVerifier)
}

@Serializable
private data class RecoverWalletRequestData(
    val accessToken: String,
    val entropyId: String,
    val entropyIdVerifier: String,
)

@Serializable
private data class RpcRequestData(
    val accessToken: String,
    val entropyId: String,
    val entropyIdVerifier: String,
    val chainType: String,
    val hdWalletIndex: Int,
    val request: RpcRequestInfo
) {
  constructor(
      accessToken: String,
      embeddedWalletDetails: EmbeddedWalletDetails,
      request: RpcRequestInfo
  ) : this(
      accessToken = accessToken,
      entropyId = embeddedWalletDetails.primaryWalletDetails.entropyId,
      entropyIdVerifier = embeddedWalletDetails.primaryWalletDetails.entropyIdVerifier,
      chainType = embeddedWalletDetails.chainType,
      hdWalletIndex = embeddedWalletDetails.hdWalletIndex,
      request = request,
  )
}

@Serializable private sealed interface RpcRequestInfo

@Serializable
private data class EthereumRpcRequestBody(val method: String, val params: List<String>) :
    RpcRequestInfo

@Serializable
private sealed interface SolanaRpcRequestBody : RpcRequestInfo {
  val method: String

  @Serializable
  data class SignMessageRequest(override val method: String, val params: SolanaSignMessageParams) :
      SolanaRpcRequestBody
}

@Serializable private data class SolanaSignMessageParams(val message: String)

// SolanaRpcRequestParams extensions to convert to SolanaRpcRequest
private fun SolanaRpcRequest.rpcRequest() =
    when (this) {
      is SolanaRpcRequest.SignMessageParams ->
          SolanaRpcRequestBody.SignMessageRequest(
              method = "signMessage", params = SolanaSignMessageParams(message = this.message))
    }
