package io.privy.sdk

import android.content.Context
import io.privy.auth.AuthState
import io.privy.auth.PrivyUser
import io.privy.auth.customAuth.LoginWithCustomAuth
import io.privy.auth.email.LoginWithEmail
import io.privy.auth.oAuth.LoginWithOAuth
import io.privy.auth.siwe.LoginWithSiwe
import io.privy.auth.sms.LoginWithSms
import kotlinx.coroutines.flow.StateFlow

public interface Privy {
  /**
   * Check if Privy is ready to be used. Check this boolean or call [Privy.awaitReady] before taking
   * any action with Privy.
   */
  public val isReady: Boolean

  /**
   * When the PrivySDK is initialized, Privy attempts to asynchronously restore the user's previous
   * session if they were previously logged in. During this time, the user's authentication state
   * will be not ready. Call this function to await Privy ready state. It is **highly** recommended
   * to call this function before executing any calls to Privy
   */
  public suspend fun awaitReady()

  /**
   * The privy user object. Will be non null if the user is authenticated. Be sure to check
   * [Privy.isReady] or call [Privy.awaitReady] before accessing the user.
   */
  public val user: PrivyUser?

  /**
   * A state flow that can be used to check current auth state or can be subscribed to for auth
   * state updates
   */
  public val authState: StateFlow<AuthState>

  // Entry point for SMS auth
  public val sms: LoginWithSms

  // Entry point for Email auth
  public val email: LoginWithEmail

  // Entry point for third-party auth sign in
  public val customAuth: LoginWithCustomAuth

  public val siwe: LoginWithSiwe

  public val oAuth: LoginWithOAuth

  /**
   * Logs the user out. The PrivyUser will be set to null and [authState] will be updated to
   * [AuthState.Unauthenticated]
   */
  public suspend fun logout()

  public companion object {
    /**
     * Privy's initializer. This will return an instance of Privy, which should be used across the
     * lifetime of the application.
     *
     * @param context The app's ApplicationContext
     * @param config Configuration for initialization
     */
    public fun init(context: Context, config: PrivyConfig): Privy {
      return PrivyImpl(
          context = context,
          config = config,
      )
    }
  }
}
