package io.privy.sdk.network

import android.annotation.SuppressLint
import android.content.Context
import android.content.pm.PackageManager
import android.net.ConnectivityManager
import android.net.Network
import android.net.NetworkCapabilities
import android.util.Log
import io.privy.logging.PrivyLogLevel
import io.privy.network.NetworkState
import io.privy.network.NetworkStateManager

// Adding this because registerDefaultNetworkCallback requires the ACCESS_NETWORK_STATE to be present
// in the AndroidManifest. However, adding it is up to the developer, and our SDK gracefully fails
// if it's not present.
@SuppressLint("MissingPermission")
public class RealNetworkStateManager(
  private val context: Context,
  private val logLevel: PrivyLogLevel,
): NetworkStateManager {
  override val current: NetworkState
    get() = _currentState

  private var _currentState: NetworkState = NetworkState.Unknown

  // Connectivity manager will only be set if permissions are accepted
  private var connectivityManager: ConnectivityManager = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager

  private val networkCallback = object : ConnectivityManager.NetworkCallback() {
    override fun onAvailable(network: Network) {
      super.onAvailable(network)
      updateState(NetworkState.Connected)
    }

    override fun onLost(network: Network) {
      super.onLost(network)
      updateState(NetworkState.Disconnected)
    }

    override fun onUnavailable() {
      super.onUnavailable()
      updateState(NetworkState.Disconnected)
    }

    // Important - this allows us to check if wifi / internet is avialable, but there's no data
    // being passed through the network connection.
    override fun onCapabilitiesChanged(network: Network, networkCapabilities: NetworkCapabilities) {
      super.onCapabilitiesChanged(network, networkCapabilities)
      networkCapabilities.updateState()
    }
  }

  init {
    try {
      if (hasAccessNetworkStatePermission()) {
        setupConnectivity()
      } else {
        printLog {
          Log.i("PrivyNetwork", "Please add the android.permission.ACCESS_NETWORK_STATE permission to your AndroidManifest to allow Privy to determine the device's network state.")
        }
      }
    } catch (e: SecurityException) {
      // SecurityException thrown if permissions not granted.
      // This should never happen since we check permissions.
      printLog {
        Log.e("PrivyNetwork", "NetworkStateManager threw Security exception. ${e.message}")
      }
    }
  }

  private fun setupConnectivity() {
    // Determine initial network state
    connectivityManager.activeNetwork?.let { activeNetwork ->
      val networkCapabilities = connectivityManager.getNetworkCapabilities(activeNetwork)
      networkCapabilities?.updateState()
    } ?: kotlin.run {
      // no active network
      updateState(NetworkState.Disconnected)
    }

    // Register to network changes
    connectivityManager.registerDefaultNetworkCallback(networkCallback)
  }

  private fun NetworkCapabilities?.updateState() {
    if (this != null) {
      // https://www.youtube.com/watch?v=wvDPG2iQ-OE
      val isConnected = hasCapability(NetworkCapabilities.NET_CAPABILITY_VALIDATED)

      if (isConnected) {
        updateState(NetworkState.Connected)
      } else {
        updateState(NetworkState.Disconnected)
      }
    } else {
      // null network capabilities indicate network is disconnected
      updateState(NetworkState.Disconnected)
    }
  }

  private fun updateState(newState: NetworkState) {
    if (newState != _currentState) {
      printLog {
        Log.v("PrivyNetwork", "Device network has changed from: $_currentState to $newState.")
      }

      // Update current state if it's different than the newly determined one
      _currentState = newState
    }
  }

  private fun printLog(log: () -> Unit) {
    if (logLevel != PrivyLogLevel.NONE) {
      // for connectivity, print log levels as long as log level is not NONE
      log()
    }
  }

  private fun hasAccessNetworkStatePermission(): Boolean {
    return context.checkSelfPermission(android.Manifest.permission.ACCESS_NETWORK_STATE) == PackageManager.PERMISSION_GRANTED
  }
}