package io.privy.sdk.oAuth

import io.privy.auth.oAuth.PKCEHelper
import java.security.MessageDigest
import java.security.SecureRandom
import java.util.Base64

internal class RealPKCEHelper : PKCEHelper {
    private companion object {
        const val CODE_VERIFIER_BYTE_LENGTH = 32
    }

    /**
     * As specified here: https://datatracker.ietf.org/doc/html/rfc7636#section-4.1
     * code_verifier = high-entropy cryptographic random STRING using the unreserved characters [A-Z] / [a-z] / [0-9] / "-" / "." / "_" / "~"
     * with a minimum length of 43 characters and a maximum length of 128 characters.
     *
     * 32 bytes gets us 256 bits, which results in 43 characters (6 bits per base64 character)
     * This is what is recommended in the spec.
     */
    override fun generateCodeVerifier(): String {
        val randomBytes = ByteArray(CODE_VERIFIER_BYTE_LENGTH)
        SecureRandom().nextBytes(randomBytes)
        // Use URL-safe Base64 encoding without padding
        return Base64.getUrlEncoder().withoutPadding().encodeToString(randomBytes)
    }

    override fun generateStateCode(): String {
        // For simplicity, reuse the code verifier generation for the state code
        return generateCodeVerifier()
    }

    override fun deriveCodeChallenge(codeVerifier: String): String {
        val digest = MessageDigest.getInstance("SHA-256")
            .digest(codeVerifier.toByteArray(Charsets.UTF_8))
        // Use URL-safe Base64 encoding without padding
        return Base64.getUrlEncoder().withoutPadding().encodeToString(digest)
    }
}
