package io.privy.sdk

import android.content.Context
import io.privy.analytics.AnalyticsEvent
import io.privy.auth.AuthState
import io.privy.auth.PrivyUser
import io.privy.auth.customAuth.LoginWithCustomAuth
import io.privy.auth.email.LoginWithEmail
import io.privy.auth.internal.InternalAuthState
import io.privy.auth.oAuth.LoginWithOAuth
import io.privy.auth.siwe.LoginWithSiwe
import io.privy.auth.sms.LoginWithSms
import io.privy.network.PrivyEnvironment
import io.privy.network.isConfirmedDisconnected
import io.privy.sdk.di.PrivyCoreComponent
import io.privy.sdk.internal.PrivyInternal
import io.privy.sdk.network.RealNetworkStateManager
import io.privy.sdk.oAuth.RealOAuthHandler
import io.privy.sdk.oAuth.RealPKCEHelper
import io.privy.sdk.webview.RealWebViewHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.flow.SharingStarted
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.filter
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.flow.map
import kotlinx.coroutines.flow.stateIn
import kotlinx.coroutines.launch
import kotlin.math.log

internal class PrivyImpl(
    context: Context,
    config: PrivyConfig,
) : Privy {
  private val scope = CoroutineScope(SupervisorJob() + Dispatchers.Main.immediate)

  private val privyEnvironment: PrivyEnvironment =
      if (config.appId in stagingAppIds) {
        PrivyEnvironment.Staging
      } else {
        PrivyEnvironment.Production
      }

  override val networkStateManager = RealNetworkStateManager(
    context = context,
    logLevel = config.logLevel,
  )

  private val webViewHandler =
      RealWebViewHandler(
          context = context,
          privyEnvironment = privyEnvironment,
          appId = config.appId,
          appClientId = config.appClientId,
          logLevel = config.logLevel,
          networkStateManager = networkStateManager,
      )

  private val pkceHelper = RealPKCEHelper()

  private val oAuthHandler = RealOAuthHandler(context = context)

  private val privyInternalSettings = PrivyInternal.internalSettings

  // Initialize DI graph
  private val coreComponent: PrivyCoreComponent =
      PrivyCoreComponent.create(
          context = context,
          privyAppId = config.appId,
          privyAppClientId = config.appClientId,
          privyLogLevel = config.logLevel,
          appBundleIdentifier = context.packageName,
          webViewHandler = webViewHandler,
          privyEnvironment = privyEnvironment,
          privyInternalSettings = privyInternalSettings,
          networkStateManager = networkStateManager,
          pkceHelper = pkceHelper,
          oAuthHandler = oAuthHandler
      )

  private val authManager = coreComponent.kmpComponent.authManager
  private val authStateRepository = coreComponent.kmpComponent.authStateRepository

  // We determine that Privy is ready for use after auth state has been set
  // IMPORTANT - the execution happens in the getter "get() = " so that it's executed
  // each time the isReady variable is read
  override val isReady: Boolean
    get() = authState.value.hasBeenDetermined()

  // Only expose user object is there is an authenticated session
  override val user: PrivyUser?
    get() {
      val currentAuthState = authState.value
      return if (currentAuthState is AuthState.Authenticated) {
        // if there is an authenticated session return a user
        currentAuthState.user
      } else {
        // if no session, return null
        null
      }
    }

  // Automatically update external auth state on internal auth state updates
  override val authState: StateFlow<AuthState> =
      authStateRepository.internalAuthState
          .map { internalAuthState ->
            when (internalAuthState) {
              InternalAuthState.NotReady -> AuthState.NotReady
              // If authenticated, expose user object
              is InternalAuthState.Authenticated ->
                  AuthState.Authenticated(user = authManager.privyUser)
              InternalAuthState.Unauthenticated -> {
                AuthState.Unauthenticated
              }
            }
          }
          .stateIn(
              scope = scope, started = SharingStarted.Eagerly, initialValue = AuthState.NotReady)

  override val sms: LoginWithSms = coreComponent.kmpComponent.loginWithSms

  override val email: LoginWithEmail = coreComponent.kmpComponent.loginWithEmail

  override val customAuth: LoginWithCustomAuth = coreComponent.kmpComponent.loginWithCustomAuth

  override val siwe: LoginWithSiwe = coreComponent.kmpComponent.loginWithSiwe

  override val oAuth: LoginWithOAuth = coreComponent.kmpComponent.loginWithOAuth

  init {
    // If tokenProvider is provided, pass it to custom auth flow
    if (config.customAuthConfig != null) {
      customAuth.setTokenProvider(tokenProvider = config.customAuthConfig.tokenProvider)
    }

    logSdkInitializeEvent()
  }

  override suspend fun awaitReady() {
    // For now, we consider auth manager initialization completion as the indicator
    // that devs can access privy auth state and receive the most up to date state.
    this.authManager.awaitInitializationComplete()
  }

  override suspend fun hasPersistedAuthCredentials(): Boolean {
    return this.authManager.hasPersistedAuthCredentials()
  }

  override suspend fun onNetworkRestored() {
    // We only care about network restoration if we never determined auth state at SDK init
    if (authState.value.hasBeenDetermined()) {
      coreComponent.kmpComponent.privyLogger.debug(
        "Received network restored event, but auth state has already been determined. No op."
      )

      return
    }

    // By this point, we know auth state has not been determined, so we attempt to restore
    // the prior session
    authManager.restorePriorSessionIfNeeded()
  }

  override suspend fun logout() {
    coreComponent.kmpComponent.clientAnalyticsIdRepository.clearClientAnalyticsId()
    coreComponent.kmpComponent.authManager.logout()
  }

  private fun logSdkInitializeEvent() {
    scope.launch {
      coreComponent.kmpComponent.analyticsManager.logEvent(AnalyticsEvent.SdkInitialize)
    }
  }
}

private suspend fun StateFlow<AuthState>.awaitReady() {
  // Check if the auth state has already been set (non-suspending)
  if (this.value.hasBeenDetermined()) return

  // Suspend execution until auth state has been set
  this.filter { it.hasBeenDetermined() }.first()
}

private val stagingAppIds = listOf("clpijy3tw0001kz0g6ixs9z15")
