package io.privy.sdk.network

import android.annotation.SuppressLint
import android.content.Context
import android.content.pm.PackageManager
import android.net.ConnectivityManager
import android.net.Network
import android.net.NetworkCapabilities
import android.util.Log
import io.privy.logging.PrivyLogLevel
import io.privy.network.NetworkState
import io.privy.network.NetworkStateManager
import io.privy.network.isConfirmedConnected
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.asSharedFlow
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.reduce
import kotlinx.coroutines.flow.update
import kotlinx.coroutines.withContext

// Adding this because registerDefaultNetworkCallback requires the ACCESS_NETWORK_STATE to be present
// in the AndroidManifest. However, adding it is up to the developer, and our SDK gracefully fails
// if it's not present.
@SuppressLint("MissingPermission")
public class RealNetworkStateManager(
  private val context: Context,
  private val logLevel: PrivyLogLevel,
): NetworkStateManager {
  private var _networkState: MutableStateFlow<NetworkState> = MutableStateFlow(NetworkState.Unknown)
  override fun observeNetworkChanges(): Flow<NetworkState> = _networkState.asSharedFlow()

  override fun observeNetworkRestored(): Flow<Unit> = flow {
    _networkState.reduce { oldState, newState ->
        if (!oldState.isConfirmedConnected() && newState.isConfirmedConnected()) {
          // Whenever network is connected after not being connected, emit an event to the flow
          emit(Unit)
        }

        newState
      }
  }

  private var _current: NetworkState = NetworkState.Unknown
  override val current: NetworkState
    get() = _current

  // Connectivity manager will only be set if permissions are accepted
  private var connectivityManager: ConnectivityManager = context.getSystemService(Context.CONNECTIVITY_SERVICE) as ConnectivityManager

  private val networkCallback = object : ConnectivityManager.NetworkCallback() {
    override fun onAvailable(network: Network) {
      super.onAvailable(network)
      printLog {
        Log.v("Privy", "Network onAvailable")
      }
      updateState(NetworkState.Connected)
    }

    override fun onLost(network: Network) {
      super.onLost(network)
      printLog {
        Log.v("Privy", "Network onLost")
      }
      updateState(NetworkState.Disconnected)
    }

    override fun onUnavailable() {
      super.onUnavailable()
      printLog {
        Log.v("Privy", "Network onUnavailable")
      }
      updateState(NetworkState.Disconnected)
    }
  }

  init {
    try {
      if (hasAccessNetworkStatePermission()) {
        setupConnectivity()
      } else {
        printLog {
          Log.i("Privy", "Please add the android.permission.ACCESS_NETWORK_STATE permission to your AndroidManifest to allow Privy to determine the device's network state.")
        }
      }
    } catch (e: SecurityException) {
      // SecurityException thrown if permissions not granted.
      // This should never happen since we check permissions.
      printLog {
        Log.e("Privy", "NetworkStateManager threw Security exception. ${e.message}")
      }
    }
  }

  private fun setupConnectivity() {
    // Determine initial network state
    connectivityManager.activeNetwork?.let { activeNetwork ->
      val networkCapabilities = connectivityManager.getNetworkCapabilities(activeNetwork)
      networkCapabilities?.updateState()
    } ?: kotlin.run {
      // no active network
      updateState(NetworkState.Disconnected)
    }

    printLog {
      Log.v("Privy", "Initial network state: ${_networkState.value}")
    }

    // Register to network changes
    connectivityManager.registerDefaultNetworkCallback(networkCallback)
  }

  private fun NetworkCapabilities?.updateState() {
    if (this != null) {
      // https://www.youtube.com/watch?v=wvDPG2iQ-OE
      val isConnected = hasCapability(NetworkCapabilities.NET_CAPABILITY_VALIDATED)

      if (isConnected) {
        updateState(NetworkState.Connected)
      } else {
        updateState(NetworkState.Disconnected)
      }
    } else {
      // null network capabilities indicate network is disconnected
      updateState(NetworkState.Disconnected)
    }
  }

  private fun updateState(newState: NetworkState) {
    // Update current state if it's different than the newly determined one
    if (newState != _networkState.value) {
      printLog {
        Log.v("Privy", "Device network has changed from: ${_networkState.value} to $newState.")
      }

      _current = newState

      printLog {
        Log.v("Privy", "Updating current network state: $_current.")
      }

      _networkState.value = newState
    }
  }

  private fun printLog(log: () -> Unit) {
    if (logLevel != PrivyLogLevel.NONE) {
      // for connectivity, print log levels as long as log level is not NONE
      log()
    }
  }

  private fun hasAccessNetworkStatePermission(): Boolean {
    return context.checkSelfPermission(android.Manifest.permission.ACCESS_NETWORK_STATE) == PackageManager.PERMISSION_GRANTED
  }
}