package io.privy.sdk

import android.content.Context
import io.privy.auth.AuthState
import io.privy.auth.PrivyUser
import io.privy.auth.customAuth.LoginWithCustomAuth
import io.privy.auth.email.LoginWithEmail
import io.privy.auth.oAuth.LoginWithOAuth
import io.privy.auth.siwe.LoginWithSiwe
import io.privy.auth.sms.LoginWithSms
import io.privy.network.NetworkStateManager
import kotlinx.coroutines.flow.StateFlow

public interface Privy {
  /**
   * Check if Privy is ready to be used. Check this boolean or call [Privy.awaitReady] before taking
   * any action with Privy.
   */
  public val isReady: Boolean

  /**
   * The network state manager, which can be used to determine the device's network state
   * or observe network state changes.
   */
  public val networkStateManager: NetworkStateManager

  /**
   * When the PrivySDK is initialized, the user's authentication state will be [AuthState.NotReady].
   * Privy attempts to asynchronously determine the user's authenticated state. Call [awaitReady] to
   * suspend execution until Privy determines the user's auth state, at which point you can access
   * [Privy.authState] to get the most up to date auth state.
   *
   * Note: If the user's device doesn't have network connectivity, Privy can't restore the prior user's
   * session, if it exists, and [Privy.authState] will remain [AuthState.NotReady]. See [onNetworkRestored]
   * to see how to better manage network connectivity.
   *
   * It is **highly** recommended to call this function when device network is available and
   * before executing any calls to Privy
   */
  public suspend fun awaitReady()

  /**
   * Checks if there are persisted credentials. This does not mean the user is authenticated or that
   * there is a valid user session. It simply suggests there was an authenticated user in a prior
   * session, and this session could potentially be used to re-authenticate the user.
   *
   * When there is no network connectivity, [Privy.authState] could be [AuthState.NotReady],
   * at which point this method could be used to determine if there is potentially an authenticated user.
   */
  public suspend fun hasPersistedAuthCredentials(): Boolean

  /**
   * If the PrivySDK was initialized while the device had no network connectivity, the [authState]
   * might remain as [AuthState.NotReady]. Whenever you determine the device regains connectivity,
   * call [onNetworkRestored], which will attempt to restore the prior user session. After calling
   * this, check [Privy.authState] to check the user's authenticated state.
   *
   * A call to [onNetworkRestored] after the user's auth state had already been determined will
   * result in a no op, so you can safely call this whenever.
   */
  public suspend fun onNetworkRestored()

  /**
   * The privy user object. Will be non null if the user is authenticated. Be sure to check
   * [Privy.isReady] or call [Privy.awaitReady] before accessing the user.
   */
  public val user: PrivyUser?

  /**
   * A state flow that can be used to check current auth state or can be subscribed to for auth
   * state updates
   */
  public val authState: StateFlow<AuthState>

  // Entry point for SMS auth
  public val sms: LoginWithSms

  // Entry point for Email auth
  public val email: LoginWithEmail

  // Entry point for third-party auth sign in
  public val customAuth: LoginWithCustomAuth

  public val siwe: LoginWithSiwe

  public val oAuth: LoginWithOAuth

  /**
   * Logs the user out. The PrivyUser will be set to null and [authState] will be updated to
   * [AuthState.Unauthenticated]
   */
  public suspend fun logout()

  public companion object {
    /**
     * Privy's initializer. This will return an instance of Privy, which should be used across the
     * lifetime of the application.
     *
     * @param context The app's ApplicationContext
     * @param config Configuration for initialization
     */
    public fun init(context: Context, config: PrivyConfig): Privy {
      return PrivyImpl(
          context = context,
          config = config,
      )
    }
  }
}
