package io.privy.sdk.oAuth
import android.content.Context
import android.content.Intent
import androidx.browser.customtabs.CustomTabsIntent
import io.privy.auth.oAuth.OAuthHandler
import io.privy.auth.oAuth.OAuthRedirectQueryParams
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.withContext
import me.tatarka.inject.annotations.Inject
import androidx.core.net.toUri

/**
 * Implementation of the OAuthHandler interface that manages OAuth authentication flows.
 * This class handles the OAuth authorization process, including generating authorization URLs,
 * launching browser tabs for authentication, and processing redirect URLs.
 */
@Inject
public class RealOAuthHandler(
    private val context: Context
) : OAuthHandler {
    /**
     * Opens the authentication URL in a Chrome Custom Tab.
     * Falls back to the default browser if Custom Tabs are not available.
     *
     * @param authUrl The authentication URL to open
     * @return Result indicating success or failure of launching the browser
     */
    public override suspend fun launchCustomTab(authUrl: String): Result<Unit> {
        return try {
            withContext(Dispatchers.Main) {
                val customTabsIntent = CustomTabsIntent.Builder().setShowTitle(true).build()
                val intent = customTabsIntent.intent.apply {
                    addFlags( Intent.FLAG_ACTIVITY_NEW_TASK or Intent.FLAG_ACTIVITY_SINGLE_TOP)
                    data = authUrl.toUri()
                }
                context.startActivity(intent)
            }
            Result.success(Unit)
        } catch (e: android.content.ActivityNotFoundException) {
            Result.failure(RuntimeException("No browser available to handle OAuth redirect", e))
        } catch (e: Exception) {
            Result.failure(e)
        }
    }
}