package io.privy.sdk.oAuth

import android.content.Intent
import android.os.Bundle
import android.util.Log
import androidx.activity.ComponentActivity
import androidx.activity.result.ActivityResultLauncher
import androidx.activity.result.contract.ActivityResultContracts
import androidx.browser.customtabs.CustomTabsIntent
import androidx.core.net.toUri
import io.privy.auth.AuthenticationException
import io.privy.auth.oAuth.OAuthDataManager

/**
 * Activity that handles OAuth flows: both launching Custom Tabs and handling redirects. This
 * activity is automatically included in the app's manifest through manifest merging.
 *
 * The activity detects when users cancel the OAuth flow by closing the Custom Tab
 * without completing authorization.
 */
public class PrivyRedirectActivity : ComponentActivity() {

  private lateinit var customTabLauncher: ActivityResultLauncher<Intent>
  private var isOAuthCompleted = false

  override fun onCreate(savedInstanceState: Bundle?) {
    super.onCreate(savedInstanceState)

    // Register for activity result
    customTabLauncher =
        registerForActivityResult(ActivityResultContracts.StartActivityForResult()) {
          // Custom Tab closed - delay check to handle race condition with redirect
          window.decorView.post {
            if (!isOAuthCompleted && !isFinishing) {
              Log.d("PrivyRedirectActivity", "Chrome tab closed by user without completing OAuth")
              handleCustomTabResult(
                  Result.failure(AuthenticationException("OAuth flow cancelled by user")))
            }
          }
        }

    // Get OAuth URL and launch Custom Tab
    val oauthUrl = intent.getStringExtra("url")
    if (oauthUrl == null) {
      handleCustomTabResult(Result.failure(AuthenticationException("OAuth URL not provided")))
      return
    }
    launchCustomTab(oauthUrl)
  }

  override fun onNewIntent(intent: Intent) {
    super.onNewIntent(intent)
    
    // Handle OAuth redirect
    intent.data?.let { redirectUri ->
      Log.d("PrivyRedirectActivity", "Received OAuth redirect: $redirectUri")
      handleCustomTabResult(Result.success(redirectUri.toString()))
    }
  }

  private fun handleCustomTabResult(result: Result<String>) {
    if (!isOAuthCompleted) {
      isOAuthCompleted = true
      OAuthDataManager.handleCustomTabResult(result)
      finish()
    }
  }

  private fun launchCustomTab(oauthUrl: String) {
    try {
      val customTabsIntent = CustomTabsIntent.Builder().setShowTitle(true).build()
      val intent = customTabsIntent.intent.apply { data = oauthUrl.toUri() }

      customTabLauncher.launch(intent)
      Log.d("PrivyRedirectActivity", "Chrome Custom Tab launched with registerForActivityResult")
    } catch (e: Exception) {
      // Failed to launch browser
      Log.e("PrivyRedirectActivity", "Failed to launch Chrome Custom Tab", e)
      handleCustomTabResult(Result.failure(e))
    }
  }
}