package io.privy.sdk

import android.content.Context
import io.privy.analytics.AnalyticsEvent
import io.privy.auth.AuthState
import io.privy.auth.PrivyAuth
import io.privy.auth.PrivyUser
import io.privy.auth.customAuth.LoginWithCustomAuth
import io.privy.auth.email.LoginWithEmail
import io.privy.auth.oAuth.LoginWithOAuth
import io.privy.auth.siwe.LoginWithSiwe
import io.privy.auth.sms.LoginWithSms
import io.privy.network.PrivyEnvironment
import io.privy.sdk.di.PrivyCoreComponent
import io.privy.sdk.internal.PrivyInternal
import io.privy.sdk.network.RealNetworkStateManager
import io.privy.sdk.oAuth.RealOAuthHandler
import io.privy.sdk.oAuth.RealPKCEHelper
import io.privy.sdk.webview.RealWebViewHandler
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.launch

internal class PrivyImpl(
    context: Context,
    config: PrivyConfig,
) : Privy {
  private val scope = CoroutineScope(SupervisorJob() + Dispatchers.Main.immediate)

  private val privyEnvironment: PrivyEnvironment =
      if (config.appId in stagingAppIds) {
        PrivyEnvironment.Staging
      } else {
        PrivyEnvironment.Production
      }

  override val networkStateManager =
      RealNetworkStateManager(
          context = context,
          logLevel = config.logLevel,
      )

  private val webViewHandler =
      RealWebViewHandler(
          context = context,
          privyEnvironment = privyEnvironment,
          appId = config.appId,
          appClientId = config.appClientId,
          logLevel = config.logLevel,
          networkStateManager = networkStateManager,
      )

  private val pkceHelper = RealPKCEHelper()

  private val oAuthHandler = RealOAuthHandler(context = context)

  private val privyInternalSettings = PrivyInternal.internalSettings

  // Initialize DI graph
  private val coreComponent: PrivyCoreComponent =
      PrivyCoreComponent.create(
          context = context,
          privyAppId = config.appId,
          privyAppClientId = config.appClientId,
          privyLogLevel = config.logLevel,
          appBundleIdentifier = context.packageName,
          webViewHandler = webViewHandler,
          privyEnvironment = privyEnvironment,
          privyInternalSettings = privyInternalSettings,
          networkStateManager = networkStateManager,
          pkceHelper = pkceHelper,
          oAuthHandler = oAuthHandler,
        )

  // Main entrypoint to all auth operations
  private val auth: PrivyAuth = coreComponent.kmpComponent.auth

  // We determine that Privy is ready for use after auth state has been set
  // IMPORTANT - the execution happens in the getter "get() = " so that it's executed
  // each time the isReady variable is read
  @Deprecated("It's no longer required to check ready state prior to checking the user's AuthState anymore. Instead, users should use \"suspend fun getAuthState(): AuthState\"")
  override val isReady: Boolean
    get() = this.auth.isReady

  // Only expose user object is there is an authenticated session
  override val user: PrivyUser?
    get() = auth.user

  override val authState: StateFlow<AuthState> = auth.authStateUpdates

  override val sms: LoginWithSms = auth.sms

  override val email: LoginWithEmail = auth.email

  override val customAuth: LoginWithCustomAuth = auth.customAuth

  override val siwe: LoginWithSiwe = auth.siwe

  override val oAuth: LoginWithOAuth = auth.oAuth

  init {
    // TODO: Could just inject "customAuthConfig" into the graph, but would have to move it into
    // another module that KMP can depend on
    // If tokenProvider is provided, pass it to custom auth flow
    if (config.customAuthConfig != null) {
      customAuth.setTokenProvider(tokenProvider = config.customAuthConfig.tokenProvider)
    }

    logSdkInitializeEvent()
  }

  override suspend fun getAuthState(): AuthState {
    return this.auth.getAuthState()
  }

  override suspend fun awaitReady() {
    // For now, we consider auth manager initialization completion as the indicator
    // that devs can access privy auth state and receive the most up to date state.
    this.auth.awaitInitializationComplete()
  }

  override suspend fun hasPersistedAuthCredentials(): Boolean {
    return this.auth.hasPersistedAuthCredentials()
  }

  override suspend fun onNetworkRestored() {
    this.auth.onNetworkRestored()
  }

  override suspend fun logout() {
    this.auth.logout()
  }

  private fun logSdkInitializeEvent() {
    scope.launch {
      coreComponent.kmpComponent.analyticsManager.logEvent(AnalyticsEvent.SdkInitialize)
    }
  }
}

private val stagingAppIds = listOf("clpijy3tw0001kz0g6ixs9z15", "cla06f34x0001mh08l8nsr496")
