package io.privy.sdk.webview

import android.annotation.SuppressLint
import android.content.Context
import android.util.Log
import android.webkit.WebView
import android.webkit.WebViewClient
import io.privy.wallet.EmbeddedWalletException
import io.privy.logging.PrivyLogLevel
import io.privy.network.NetworkStateManager
import io.privy.network.PrivyEnvironment
import io.privy.network.isConfirmedDisconnected
import io.privy.wallet.EmbeddedWalletDetails
import io.privy.wallet.ethereum.EthereumRpcRequest
import io.privy.wallet.rpc.EthereumRpcRequestBody
import io.privy.wallet.rpc.RpcRequestInfo
import io.privy.wallet.rpc.rpcRequest
import io.privy.wallet.solana.SolanaRpcRequest
import io.privy.wallet.solana.SolanaSignMessageResponse
import io.privy.wallet.webview.IFrameResponse
import io.privy.wallet.webview.PrivyEventType
import io.privy.wallet.webview.PrivyIFrameErrorResponse
import io.privy.wallet.webview.PrivyIFrameRequest
import io.privy.wallet.webview.PrivyIFrameSuccessResponse
import io.privy.wallet.webview.UserSignerSignRequestData
import io.privy.wallet.webview.SolanaSignMessageResponseData
import io.privy.wallet.webview.WebViewHandler
import java.util.UUID
import kotlin.coroutines.cancellation.CancellationException
import kotlin.time.Duration
import kotlin.time.Duration.Companion.milliseconds
import kotlin.time.Duration.Companion.seconds
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.Dispatchers
import kotlinx.coroutines.SupervisorJob
import kotlinx.coroutines.channels.Channel
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.isActive
import kotlinx.coroutines.launch
import kotlinx.coroutines.sync.Mutex
import kotlinx.coroutines.sync.withLock
import kotlinx.coroutines.withContext
import kotlinx.coroutines.withTimeout
import kotlinx.serialization.Serializable
import kotlinx.serialization.encodeToString
import kotlinx.serialization.json.Json
import kotlinx.serialization.json.contentOrNull
import kotlinx.serialization.json.jsonObject
import kotlinx.serialization.json.jsonPrimitive

// TODO: list of things to do before finishing
// Make this class injectable?
// wallet needs to be recovered every time, do we need to enable storage or something?

@SuppressLint("SetJavaScriptEnabled")
internal class RealWebViewHandler(
  private val context: Context,
  privyEnvironment: PrivyEnvironment,
  appId: String,
  appClientId: String,
  logLevel: PrivyLogLevel,
  private val networkStateManager: NetworkStateManager,
) : WebViewHandler {
  private var webView: WebView

  private val baseUrl =
    if (privyEnvironment == PrivyEnvironment.Staging) {
      "https://auth.staging.privy.io"
    } else {
      "https://auth.privy.io"
    }

  private val webViewUrl = "$baseUrl/apps/$appId/embedded-wallets?client_id=$appClientId"

  // Create a coroutine scope for Privy to use.
  // SupervisorJob offers robustness so that if a child coroutine fails, the parent doesn't
  // Dispatchers.IO offloads execution to a background thread
  private val scope = CoroutineScope(SupervisorJob() + Dispatchers.IO)

  private val webViewState = WebViewState()

  private val pingReadyMutex = Mutex()

  private val requestToStringResponseChannelMap: MutableMap<String, Channel<String>> =
    mutableMapOf()

  private val internalAppIds = listOf("clpijy3tw0001kz0g6ixs9z15")
  private val printLogs = logLevel != PrivyLogLevel.NONE
  private val isInternalAppId = appId in internalAppIds

  private val json = Json {
    explicitNulls = false
    ignoreUnknownKeys = true
  }

  init {
    if (isInternalAppId) {
      WebView.setWebContentsDebuggingEnabled(true);
    }

    subscribeToWebViewReady()

    subscribeToNetworkUpdates()

    this.webView = createAndConfigureWebView()

    if (networkStateManager.current.isConfirmedDisconnected()) {
      printInternalLog("No network, not loading webview until network is restored.")
    } else {
      // Load URL, should trigger "onPageFinished" on complete.
      webView.loadUrl(webViewUrl)
    }
  }

  private fun subscribeToNetworkUpdates() {
    scope.launch {
      networkStateManager
        .observeNetworkRestored()
        .collectLatest {
          printLog("Network restored, resetting webview.")

          // If network is restored, reload webview and reset ready state
          webViewState.setReady(false)

          // create new one to override old, stale one
          withContext(Dispatchers.Main.immediate) {
            webView = createAndConfigureWebView()
            webView.loadUrl(webViewUrl)
          }
        }
    }
  }

  private fun createAndConfigureWebView(): WebView {
    // Webview not yet initialized, create and return
    val webView = WebView(context)

    webView.settings.javaScriptEnabled = true
    webView.settings.domStorageEnabled = true

    // this allows javascript to call a native method. in this case, when JS calls
    // JS_INTERFACE_NAME.postMessage, it would trigger PrivyJavascriptInterface.postMessage
    webView.addJavascriptInterface(
      PrivyJavascriptInterface { message -> this.onWebViewMessageReceived(message = message) },
      JS_INTERFACE_NAME
    )

    webView.webViewClient =
      object : WebViewClient() {
        override fun onPageFinished(view: WebView?, url: String?) {
          // Callback triggered when the page finishes loading
          printLog("Page finished loading: $url")

          // When page is finish loading, set window.PRIVY_NATIVE_ANDROID to true
          webView.evaluateJavascript(scriptSource, null)

          // Attempt ready ping until successful
          awaitReadyInBackground()
        }
      }

    return webView
  }

  private fun subscribeToWebViewReady() {
    // Just for debugging purposes, ensures web view ready works.
    scope.launch {
      // simulate ready ping delay
      webViewState.awaitReady()
      printLog("Webview is ready!")
    }
  }

  // Kick off ready check
  private fun awaitReadyInBackground() {
    printLog("Preemptively checking webview ready state")

    if (networkStateManager.current.isConfirmedDisconnected()) {
      printLog("No network, cancelling webview ready state check.")
      return
    }

    scope.launch {
      awaitReady()
    }
  }

  private suspend fun awaitReady(): Result<Unit> {
    pingReadyMutex.withLock {
      // when mutex is unlocked, we want to double check if it's ready before attempting
      if (webViewState.isReady()) {
        // already ready, return
        return Result.success(Unit)
      }

      return pingReadyUntilSuccessfulOrTimedOut()
        .onSuccess {
          printLog("Webview is ready.")
        }
        .onFailure {
          printLog("Webview ready check failed: ${it.message}")
        }
    }
  }

  private suspend fun pingReadyUntilSuccessfulOrTimedOut(timeout: Duration = 15.seconds): Result<Unit> {
    return try {
      withTimeout(timeout) {
        while (!webViewState.isReady()) {
          val result = pingReady()

          if (result is IFrameResponse.Success) {
            webViewState.setReady(true)
            return@withTimeout Result.success(Unit)
          }

          if (!isActive) {
            // Coroutine has been cancelled, likely due to timeout
            printInternalLog("Coroutine cancelled -- breaking from ping ready loop.")
            break
          }
        }

        // should never reach here - either ping ready should succeed and we return above, or
        // it will time out and error will be thrown. adding return here to suffice compiler
        return@withTimeout Result.failure(EmbeddedWalletException("Unexpected error while pinging ready."))
      }
    } catch (e: CancellationException) {
      printLog("Ping ready cancelled with exception: ${e.message}")
      return Result.failure(e)
    }
  }

  private suspend fun pingReady(): IFrameResponse<PrivyIFrameSuccessResponse.IFrameReadyResponse> {
    // technically this request doesn't have an associated type for PrivyIFrameRequest
    // so setting it to String?. Can't set to Any since it doesn't have a serializer
    val readyRequest =
      PrivyIFrameRequest<String?>(
        id = UUID.randomUUID().toString().lowercase(),
        event = PrivyEventType.Ready,
        data = null)

    // We can expect to receive ready ping response within 100 milliseconds,
    // so setting timeout to 150 milliseconds is plenty.
    return performRequest(request = readyRequest, timeout = 150.milliseconds)
  }

  override suspend fun createEthereumWallet(
    accessToken: String,
    existingSolanaWalletAddress: String?,
  ): IFrameResponse<PrivyIFrameSuccessResponse.CreateEthereumWalletResponse> {
    val createEthereumWalletWalletRequest =
      PrivyIFrameRequest(
        id = UUID.randomUUID().toString().lowercase(),
        event = PrivyEventType.CreateEthereumWallet,
        data =
        CreateEthereumWalletRequestData(
          accessToken = accessToken, solanaAddress = existingSolanaWalletAddress))

    return performRequest(createEthereumWalletWalletRequest, timeout = 60.seconds)
  }

  override suspend fun createAdditionalWallet(
    accessToken: String,
    chainType: String,
    entropyId: String,
    entropyIdVerifier: String,
    hdWalletIndex: Int,
  ): IFrameResponse<PrivyIFrameSuccessResponse.CreateAdditionalWalletResponse> {
    val createAdditionalWalletRequest =
      PrivyIFrameRequest(
        id = UUID.randomUUID().toString().lowercase(),
        event = PrivyEventType.CreateAdditional,
        data =
        CreateAdditionalWalletRequestData(
          accessToken = accessToken,
          chainType = chainType,
          entropyId = entropyId,
          entropyIdVerifier = entropyIdVerifier,
          hdWalletIndex = hdWalletIndex))

    return performRequest(createAdditionalWalletRequest, timeout = 60.seconds)
  }

  override suspend fun createSolanaWallet(
    accessToken: String,
    existingEthereumWalletAddress: String?,
  ): IFrameResponse<PrivyIFrameSuccessResponse.CreateSolanaWalletResponse> {
    val createSolanaWalletRequest =
      PrivyIFrameRequest(
        id = UUID.randomUUID().toString().lowercase(),
        event = PrivyEventType.CreateSolanaWallet,
        data =
        CreateSolanaWalletRequestData(
          accessToken = accessToken, ethereumAddress = existingEthereumWalletAddress))

    return performRequest(createSolanaWalletRequest, timeout = 60.seconds)
  }

  override suspend fun connectWallet(
    accessToken: String,
    embeddedWalletDetails: EmbeddedWalletDetails
  ): IFrameResponse<PrivyIFrameSuccessResponse.ConnectWalletResponse> {
    val connectWalletRequest =
      PrivyIFrameRequest(
        id = UUID.randomUUID().toString().lowercase(),
        event = PrivyEventType.Connect,
        data =
        ConnectWalletRequestData(
          accessToken = accessToken,
          embeddedWalletDetails = embeddedWalletDetails,
        ))

    printLog("Attempting to connect to wallet")

    return performRequest(connectWalletRequest)
  }

  override suspend fun recoverWallet(
    accessToken: String,
    embeddedWalletDetails: EmbeddedWalletDetails
  ): IFrameResponse<PrivyIFrameSuccessResponse.RecoverWalletResponse> {
    val recoverWalletRequest =
      PrivyIFrameRequest(
        id = UUID.randomUUID().toString().lowercase(),
        event = PrivyEventType.Recover,
        data =
        RecoverWalletRequestData(
          accessToken = accessToken,
          entropyId = embeddedWalletDetails.entropyWalletDetails.entropyId,
          entropyIdVerifier =
          embeddedWalletDetails.entropyWalletDetails.entropyIdVerifier,
        ))

    printLog("Attempting to recover wallet")

    return performRequest(recoverWalletRequest)
  }

  override suspend fun ethereumRpc(
    accessToken: String,
    embeddedWalletDetails: EmbeddedWalletDetails,
    request: EthereumRpcRequest
  ): IFrameResponse<PrivyIFrameSuccessResponse.WalletRpcResponse<String>> {
    val rpcRequest =
        PrivyIFrameRequest(
            id = UUID.randomUUID().toString().lowercase(),
            event = PrivyEventType.Rpc,
            data =
                RpcRequestData(
                    accessToken = accessToken,
                    embeddedWalletDetails = embeddedWalletDetails,
                    request =
                        EthereumRpcRequestBody(method = request.method, params = request.params)
                ))

    return performRequest(rpcRequest)
  }

  override suspend fun solanaRpc(
    accessToken: String,
    embeddedWalletDetails: EmbeddedWalletDetails,
    request: SolanaRpcRequest
  ): Result<SolanaSignMessageResponse> {
    val rpcRequest =
      PrivyIFrameRequest(
        id = UUID.randomUUID().toString().lowercase(),
        event = PrivyEventType.Rpc,
        data =
        RpcRequestData(
          accessToken = accessToken,
          embeddedWalletDetails = embeddedWalletDetails,
          request = request.rpcRequest()))

    return when (request) {
      is SolanaRpcRequest.SignMessageParams -> {

        when (val response:
                IFrameResponse<
                        PrivyIFrameSuccessResponse.WalletRpcResponse<SolanaSignMessageResponseData>> =
          performRequest(rpcRequest)) {
          is IFrameResponse.Success -> {
            Result.success(
              SolanaSignMessageResponse(signature = response.value.data.response.data.signature))
          }
          is IFrameResponse.Error -> {
            // Some other error - surface it as failure
            val errorMessage = response.message
            Result.failure(EmbeddedWalletException(errorMessage))
          }
        }
      }
    }
  }

  override suspend fun signWithUserSigner(
    accessToken: String,
    message: ByteArray
): IFrameResponse<PrivyIFrameSuccessResponse.UserSignerSignResponse> {
    
    val request = PrivyIFrameRequest(
        id = UUID.randomUUID().toString().lowercase(),
        event = PrivyEventType.UserSign,
        data = UserSignerSignRequestData(
            accessToken = accessToken,
            message = message
        )
    )

    return performRequest<UserSignerSignRequestData, PrivyIFrameSuccessResponse.UserSignerSignResponse>(
        request = request,
        timeout = 60.seconds
    )
}

  private suspend inline fun <reified T, reified R : PrivyIFrameSuccessResponse> performRequest(
    request: PrivyIFrameRequest<T>,
    timeout: Duration = 30.seconds
  ): IFrameResponse<R> {
    // ensure internet is connected, else throw an error
    if (networkStateManager.current.isConfirmedDisconnected()) {
      return IFrameResponse.Error(
        type = "device_not_connected",
        message = "The device is not connected to the internet."
      )
    }

    if (request.event != PrivyEventType.Ready) {
      awaitReady().onFailure {
        printErrorLog("Webview ready state failure while performing ${request.event} request.")

        // If ready state fails, we should consider this call failed too
        return IFrameResponse.Error(
          type = "ready_request_timed_out", message = "The privy ready request timed out."
        )
      }
    }

    val serializedRequest = request.stringData()

    // Should never happen, but in case request didn't serialize properly
    if (serializedRequest == null) {
      // Need to find better return type here
      printErrorLog("Couldn't serialize iframe request for ${request.event} request.")

      return IFrameResponse.Error(
        type = "request_not_serializable",
        message = "The privy IFrame request type is not serializable."
      )
    }

    // Specify timeout for webview request, defaults to 15 seconds
    return try {
      withTimeout(timeout) {
        // Even though this block can be cancelled, don't need to handle cancellation bc it's one shot
        sendRequestToWebView(
          requestId = request.id,
          serializedRequest = serializedRequest,
        )
      }
    } catch (e: CancellationException) {
      IFrameResponse.Error(
        type = "request_timed_out",
        message = "The privy IFrame request timed out.",
      )
    }
  }

  private suspend inline fun <reified T : PrivyIFrameSuccessResponse> sendRequestToWebView(
    requestId: String,
    serializedRequest: String,
  ): IFrameResponse<T> {
    // Behavior: No buffer. The sender suspends until a receiver is ready, and the receiver suspends
    // until a sender is ready. It's a direct hand-off.
    val responseChannel = Channel<String>()
    requestToStringResponseChannelMap[requestId] = responseChannel

    // VERY IMPORTANT. SINCE THIS MAY BE LAUNCHED IN COROUTINE SCOPE THAT USES DEFAULT DISPATCHER
    // NEED TO SWITCH CONTEXT TO MAIN THREAD FOR WEBVIEW INTERACTIONS
    withContext(Dispatchers.Main) {
      val jsDispatchEvent =
        "window.dispatchEvent(new MessageEvent('message', { data: $serializedRequest}));"
      printInternalLog("Dispatching event: $jsDispatchEvent")

      webView.evaluateJavascript(jsDispatchEvent) { receivedValue ->
        printLog("Performed privy i frame request received value: $receivedValue")
      }
    }

    // Suspends until channel receives a result
    val result = responseChannel.receive()

    printLog("Received webview result! $result")

    return decodeWebViewMessage<T>(result)
  }

  private fun onWebViewMessageReceived(message: String) {
    scope.launch {
      try {
        printLog("Message from JS: $message")

        val responseJson = Json.parseToJsonElement(message)
        val id = responseJson.jsonObject["id"]?.jsonPrimitive?.contentOrNull

        val channel = requestToStringResponseChannelMap[id]

        if (channel == null) {
          printErrorLog("There is no associated request to the id specified in response.")
          return@launch
        }

        channel.send(message)
      } catch (e: Exception) {
        printErrorLog("Response decoding error. $e")
      }
    }
  }

  private inline fun <reified T : PrivyIFrameSuccessResponse> decodeWebViewMessage(
    message: String
  ): IFrameResponse<T> {
    return try {
      if (message.contains("error")) {
        val error = json.decodeFromString<PrivyIFrameErrorResponse>(message)
        printErrorLog("Error received in post message! $error")
        IFrameResponse.Error(
          type = error.error.type,
          message = error.error.message,
        )
      } else {
        val success = json.decodeFromString<T>(message)
        printLog("Success received in post message! $success")
        IFrameResponse.Success(success)
      }
    } catch (e: Exception) {
      printErrorLog(
        "We don't support the response type structure in the received message! Message: $message.\nError: ${e.message}")

      return IFrameResponse.Error(
        type = "response_not_deserializable",
        message = "The iFrame response couldn't be deserialized",
      )
    }
  }

  private inline fun <reified T> PrivyIFrameRequest<T>.stringData(): String? {
    return try {
      json.encodeToString(this)
    } catch (e: Exception) {
      printErrorLog("Error encoding iframe request: $e")
      null // Handle encoding errors gracefully
    }
  }

  private fun printLog(message: String) {
    if (printLogs) {
      Log.i("PrivyWebView", message)
    }
  }

  private fun printErrorLog(message: String) {
    if (printLogs) {
      Log.e("PrivyWebView", message)
    }
  }

  private fun printInternalLog(message: String) {
    if (isInternalAppId && printLogs) {
      Log.d("PrivyWebView", "(Internal) $message")
    }
  }

  private companion object {
    const val JS_INTERFACE_NAME = "AndroidProxy"
    const val scriptSource = "window.PRIVY_NATIVE_ANDROID = true;"
  }
}

@Serializable
private data class CreateEthereumWalletRequestData(
  val accessToken: String,

  /**
   * The address of the user's existing Solana embedded wallet.
   *
   * If an existing address is passed, the iframe will re-assemble the entropy for the existing
   * address to derive the Ethereum wallet. If no existing address is passed, the iframe will
   * generate new entropy for the Ethereum wallet.
   */
  val solanaAddress: String?
)

@Serializable
private data class CreateSolanaWalletRequestData(
  val accessToken: String,

  /**
   * The address of the user's existing Ethereum embedded wallet.
   *
   * If an existing address is passed, the iframe will re-assemble the entropy for the existing
   * address to derive the Ethereum wallet. If no existing address is passed, the iframe will
   * generate new entropy for the Ethereum wallet.
   */
  val ethereumAddress: String?
)

@Serializable
private data class CreateAdditionalWalletRequestData(
  val accessToken: String,
  val chainType: String,
  val entropyId: String,
  val entropyIdVerifier: String,
  val hdWalletIndex: Int
)

@Serializable
private data class ConnectWalletRequestData(
  val accessToken: String,
  val chainType: String,
  val entropyId: String,
  val entropyIdVerifier: String,
) {
  constructor(
    accessToken: String,
    embeddedWalletDetails: EmbeddedWalletDetails
  ) : this(
    accessToken = accessToken,
    chainType = embeddedWalletDetails.chainType,
    entropyId = embeddedWalletDetails.entropyWalletDetails.entropyId,
    entropyIdVerifier = embeddedWalletDetails.entropyWalletDetails.entropyIdVerifier)
}

@Serializable
private data class RecoverWalletRequestData(
  val accessToken: String,
  val entropyId: String,
  val entropyIdVerifier: String,
)

@Serializable
private data class RpcRequestData(
  val accessToken: String,
  val entropyId: String,
  val entropyIdVerifier: String,
  val chainType: String,
  val hdWalletIndex: Int,
  val request: RpcRequestInfo
) {
  constructor(
    accessToken: String,
    embeddedWalletDetails: EmbeddedWalletDetails,
    request: RpcRequestInfo
  ) : this(
    accessToken = accessToken,
    entropyId = embeddedWalletDetails.entropyWalletDetails.entropyId,
    entropyIdVerifier = embeddedWalletDetails.entropyWalletDetails.entropyIdVerifier,
    chainType = embeddedWalletDetails.chainType,
    hdWalletIndex = embeddedWalletDetails.hdWalletIndex,
    request = request,
  )
}