/*
 * Decompiled with CFR 0.152.
 */
package reactor.pool;

import java.io.Closeable;
import java.io.IOException;
import java.time.Clock;
import java.time.Duration;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicIntegerFieldUpdater;
import java.util.function.Function;
import org.reactivestreams.Publisher;
import org.reactivestreams.Subscription;
import reactor.core.CoreSubscriber;
import reactor.core.Disposable;
import reactor.core.Disposables;
import reactor.core.Scannable;
import reactor.core.publisher.Mono;
import reactor.core.publisher.Operators;
import reactor.core.scheduler.Schedulers;
import reactor.pool.InstrumentedPool;
import reactor.pool.PoolAcquireTimeoutException;
import reactor.pool.PoolBuilder;
import reactor.pool.PoolConfig;
import reactor.pool.PoolMetricsRecorder;
import reactor.pool.PooledRef;
import reactor.pool.PooledRefMetadata;
import reactor.util.Logger;
import reactor.util.annotation.Nullable;
import reactor.util.context.Context;

abstract class AbstractPool<POOLABLE>
implements InstrumentedPool<POOLABLE>,
InstrumentedPool.PoolMetrics {
    final Logger logger;
    final PoolConfig<POOLABLE> poolConfig;
    final PoolMetricsRecorder metricsRecorder;
    final Clock clock;
    volatile int pendingCount;
    static final AtomicIntegerFieldUpdater<AbstractPool> PENDING_COUNT = AtomicIntegerFieldUpdater.newUpdater(AbstractPool.class, "pendingCount");

    AbstractPool(PoolConfig<POOLABLE> poolConfig, Logger logger) {
        this.poolConfig = poolConfig;
        this.logger = logger;
        this.metricsRecorder = poolConfig.metricsRecorder();
        this.clock = poolConfig.clock();
    }

    @Override
    public InstrumentedPool.PoolMetrics metrics() {
        return this;
    }

    @Override
    public int pendingAcquireSize() {
        return PENDING_COUNT.get(this);
    }

    @Override
    public int allocatedSize() {
        return this.poolConfig.allocationStrategy().permitGranted();
    }

    @Override
    public abstract int idleSize();

    @Override
    public int acquiredSize() {
        return this.allocatedSize() - this.idleSize();
    }

    @Override
    public int getMaxAllocatedSize() {
        return this.poolConfig.allocationStrategy().permitMaximum();
    }

    @Override
    public int getMaxPendingAcquireSize() {
        return this.poolConfig.maxPending() < 0 ? Integer.MAX_VALUE : this.poolConfig.maxPending();
    }

    abstract boolean elementOffer(POOLABLE var1);

    abstract void doAcquire(Borrower<POOLABLE> var1);

    abstract void cancelAcquire(Borrower<POOLABLE> var1);

    private void defaultDestroy(@Nullable POOLABLE poolable) {
        if (poolable instanceof Disposable) {
            ((Disposable)poolable).dispose();
        } else if (poolable instanceof Closeable) {
            try {
                ((Closeable)poolable).close();
            }
            catch (IOException e) {
                this.logger.trace("Failure while discarding a released Poolable that is Closeable, could not close", (Throwable)e);
            }
        }
    }

    Mono<Void> destroyPoolable(AbstractPooledRef<POOLABLE> ref) {
        if (ref.state != 3) {
            throw new IllegalStateException("destroying non invalidated ref " + ref);
        }
        POOLABLE poolable = ref.poolable();
        this.poolConfig.allocationStrategy().returnPermits(1);
        long start = this.clock.millis();
        this.metricsRecorder.recordLifetimeDuration(ref.lifeTime());
        Function<POOLABLE, Publisher<Void>> factory = this.poolConfig.destroyHandler();
        if (factory == PoolBuilder.NOOP_HANDLER) {
            return Mono.fromRunnable(() -> {
                this.defaultDestroy(poolable);
                this.metricsRecorder.recordDestroyLatency(this.clock.millis() - start);
            });
        }
        return Mono.from(factory.apply(poolable)).doFinally(fin -> this.metricsRecorder.recordDestroyLatency(this.clock.millis() - start));
    }

    static final class Borrower<POOLABLE>
    extends AtomicBoolean
    implements Scannable,
    Subscription,
    Runnable {
        static final Disposable TIMEOUT_DISPOSED = Disposables.disposed();
        final CoreSubscriber<? super AbstractPooledRef<POOLABLE>> actual;
        final AbstractPool<POOLABLE> pool;
        final Duration acquireTimeout;
        Disposable timeoutTask;

        Borrower(CoreSubscriber<? super AbstractPooledRef<POOLABLE>> actual, AbstractPool<POOLABLE> pool, Duration acquireTimeout) {
            this.actual = actual;
            this.pool = pool;
            this.acquireTimeout = acquireTimeout;
            this.timeoutTask = TIMEOUT_DISPOSED;
        }

        @Override
        public void run() {
            if (this.compareAndSet(false, true)) {
                this.pool.cancelAcquire(this);
                this.actual.onError((Throwable)new PoolAcquireTimeoutException(this.acquireTimeout));
            }
        }

        public void request(long n) {
            if (Operators.validate((long)n)) {
                boolean noPermits;
                boolean noIdle = this.pool.idleSize() == 0;
                boolean bl = noPermits = this.pool.poolConfig.allocationStrategy().estimatePermitCount() == 0;
                if (!this.acquireTimeout.isZero() && noIdle && noPermits) {
                    this.timeoutTask = Schedulers.parallel().schedule((Runnable)this, this.acquireTimeout.toMillis(), TimeUnit.MILLISECONDS);
                }
                this.pool.doAcquire(this);
            }
        }

        void stopPendingCountdown() {
            this.timeoutTask.dispose();
        }

        public void cancel() {
            this.set(true);
            this.pool.cancelAcquire(this);
            this.stopPendingCountdown();
        }

        @Nullable
        public Object scanUnsafe(Scannable.Attr key) {
            if (key == Scannable.Attr.CANCELLED) {
                return this.get();
            }
            if (key == Scannable.Attr.REQUESTED_FROM_DOWNSTREAM) {
                return 1;
            }
            if (key == Scannable.Attr.ACTUAL) {
                return this.actual;
            }
            return null;
        }

        void deliver(AbstractPooledRef<POOLABLE> poolSlot) {
            this.stopPendingCountdown();
            if (this.get()) {
                poolSlot.release().subscribe(aVoid -> {}, e -> Operators.onErrorDropped((Throwable)e, (Context)Context.empty()));
            } else {
                poolSlot.markAcquired();
                this.actual.onNext(poolSlot);
                this.actual.onComplete();
            }
        }

        void fail(Throwable error) {
            this.stopPendingCountdown();
            if (!this.get()) {
                this.actual.onError(error);
            }
        }

        @Override
        public String toString() {
            return this.get() ? "Borrower(cancelled)" : "Borrower";
        }
    }

    static abstract class AbstractPooledRef<T>
    implements PooledRef<T>,
    PooledRefMetadata {
        final long creationTimestamp;
        final PoolMetricsRecorder metricsRecorder;
        final Clock clock;
        final T poolable;
        final int acquireCount;
        long timeSinceRelease;
        volatile int state;
        static final AtomicIntegerFieldUpdater<AbstractPooledRef> STATE = AtomicIntegerFieldUpdater.newUpdater(AbstractPooledRef.class, "state");
        static final int STATE_IDLE = 0;
        static final int STATE_ACQUIRED = 1;
        static final int STATE_RELEASED = 2;
        static final int STATE_INVALIDATED = 3;

        AbstractPooledRef(T poolable, PoolMetricsRecorder metricsRecorder, Clock clock) {
            this.poolable = poolable;
            this.metricsRecorder = metricsRecorder;
            this.clock = clock;
            this.creationTimestamp = clock.millis();
            this.acquireCount = 0;
            this.timeSinceRelease = -2L;
            this.state = 0;
        }

        AbstractPooledRef(AbstractPooledRef<T> oldRef) {
            this.poolable = oldRef.poolable;
            this.metricsRecorder = oldRef.metricsRecorder;
            this.clock = oldRef.clock;
            this.creationTimestamp = oldRef.creationTimestamp;
            this.acquireCount = oldRef.acquireCount();
            this.timeSinceRelease = oldRef.timeSinceRelease;
            this.state = oldRef.state == 3 ? 3 : 0;
        }

        @Override
        public T poolable() {
            return this.poolable;
        }

        @Override
        public PooledRefMetadata metadata() {
            return this;
        }

        void markAcquired() {
            if (STATE.compareAndSet(this, 0, 1)) {
                long tsr = this.timeSinceRelease;
                if (tsr > 0L) {
                    this.metricsRecorder.recordIdleTime(this.clock.millis() - tsr);
                } else {
                    this.metricsRecorder.recordIdleTime(this.clock.millis() - this.creationTimestamp);
                }
            }
        }

        boolean markReleased() {
            int s;
            do {
                if ((s = this.state) != 2 && s != 3) continue;
                return false;
            } while (!STATE.compareAndSet(this, s, 2));
            this.timeSinceRelease = this.clock.millis();
            return true;
        }

        boolean markInvalidate() {
            int s;
            do {
                if ((s = this.state) != 3) continue;
                return false;
            } while (!STATE.compareAndSet(this, s, 3));
            return true;
        }

        @Override
        public int acquireCount() {
            if (STATE.get(this) == 0) {
                return this.acquireCount;
            }
            return this.acquireCount + 1;
        }

        @Override
        public long lifeTime() {
            return this.clock.millis() - this.creationTimestamp;
        }

        @Override
        public long idleTime() {
            if (STATE.get(this) == 1) {
                return 0L;
            }
            long tsr = this.timeSinceRelease;
            if (tsr < 0L) {
                tsr = this.creationTimestamp;
            }
            return this.clock.millis() - tsr;
        }

        @Override
        public abstract Mono<Void> release();

        @Override
        public abstract Mono<Void> invalidate();

        public String toString() {
            return "PooledRef{poolable=" + this.poolable + ", lifeTime=" + this.lifeTime() + "ms, idleTime=" + this.idleTime() + "ms, acquireCount=" + this.acquireCount + '}';
        }
    }
}

