import { pages } from 'build-time-data';
import { LitElement, css, html } from 'lit';
import 'qwc/qwc-extension-link.js';

const NAME = "Zebra Crossing";
export class QwcZXingCard extends LitElement {

  static styles = css`
      .identity {
        display: flex;
        justify-content: flex-start;
      }

      .description {
        padding-bottom: 10px;
      }

      .logo {
        padding-bottom: 10px;
        margin-right: 5px;
      }

      .card-content {
        color: var(--lumo-contrast-90pct);
        display: flex;
        flex-direction: column;
        justify-content: flex-start;
        padding: 2px 2px;
        height: 100%;
      }

      .card-content slot {
        display: flex;
        flex-flow: column wrap;
        padding-top: 5px;
      }
    `;

  static properties = {
    description: { type: String }
  };

  constructor() {
    super();
  }

  connectedCallback() {
    super.connectedCallback();
  }

  render() {
    return html`<div class="card-content" slot="content">
            <div class="identity">
                <div class="logo">
                    <img src="data:image/png;base64,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"
                                       alt="${NAME}" 
                                       title="${NAME}"
                                       width="32" 
                                       height="32">
                </div>
                <div class="description">${this.description}</div>
            </div>
            ${this._renderCardLinks()}
        </div>
        `;
  }

  _renderCardLinks() {
    return html`${pages.map(page => html`
                            <qwc-extension-link slot="link"
                                extensionName="${NAME}"
                                iconName="${page.icon}"
                                displayName="${page.title}"
                                staticLabel="${page.staticLabel}"
                                dynamicLabel="${page.dynamicLabel}"
                                streamingLabel="${page.streamingLabel}"
                                path="${page.id}"
                                ?embed=${page.embed}
                                externalUrl="${page.metadata.externalUrl}"
                                webcomponent="${page.componentLink}" >
                            </qwc-extension-link>
                        `)}`;
  }

}
customElements.define('qwc-zxing-card', QwcZXingCard);