package io.quarkiverse.cxf;

import java.util.Map;
import java.util.Optional;

import io.quarkus.runtime.annotations.ConfigGroup;
import io.quarkus.runtime.annotations.ConfigItem;
import io.quarkus.runtime.annotations.ConfigPhase;
import io.quarkus.runtime.annotations.ConfigRoot;

/**
 * Quarkus CXF build time configuration options that are also available at runtime but only in read-only mode.
 */
@ConfigRoot(name = "cxf", phase = ConfigPhase.BUILD_AND_RUN_TIME_FIXED)
public class CxfFixedConfig {

    /**
     * The size in bytes of the chunks of memory allocated when writing data.
     * <p>
     * This is a very advanced setting that should only be set if you understand exactly how it affects the output IO operations
     * of the application.
     */
    @ConfigItem(defaultValue = "128")
    public int minChunkSize;

    /**
     * The size of the output stream response buffer in bytes. If a response is larger than this and no content-length
     * is provided then the response will be chunked.
     * <p>
     * Larger values may give slight performance increases for large responses, at the expense of more memory usage.
     */
    @ConfigItem(defaultValue = "8191")
    public int outputBufferSize;

    /**
     * The build time part of the client configuration.
     */
    @ConfigItem(name = "client")
    public Map<String, ClientFixedConfig> clients;

    @ConfigGroup
    public static class ClientFixedConfig {

        /**
         * The client service interface class name
         */
        @ConfigItem
        public Optional<String> serviceInterface;

        /**
         * Indicates whether this is an alternative proxy client configuration. If
         * true, then this configuration is ignored when configuring a client without
         * annotation `@CXFClient`.
         */
        @ConfigItem(defaultValue = "false")
        public boolean alternative;

        /** Configuration options related to native mode */
        @ConfigItem(name = "native")
        public NativeClientFixedConfig native_;

        public static ClientFixedConfig createDefault() {
            ClientFixedConfig result = new ClientFixedConfig();
            result.serviceInterface = Optional.empty();
            return result;
        }
    }

    @ConfigGroup
    public static class NativeClientFixedConfig {

        /**
         * If {@code true}, the client dynamic proxy class generated by native compiler will be initialized at runtime;
         * otherwise the proxy class will be initialized at build time.
         * <p>
         * Setting this to {@code true} makes sense if your service endpoint interface references some class initialized
         * at runtime in its method signatures. E.g. Say, your service interface has method {@code int add(Operands o)}
         * and the {@code Operands} class was requested to be initialized at runtime. Then, without setting this
         * configuration parameter to {@code true}, the native compiler will throw an exception saying something like
         * {@code Classes that should be initialized at run time got initialized during image building: org.acme.Operands ... jdk.proxy<some-number>.$Proxy<some-number> caused initialization of this class}.
         * {@code jdk.proxy<some-number>.$Proxy<some-number>} is the proxy class generated by the native compiler.
         */
        @ConfigItem(defaultValue = "false")
        public boolean runtimeInitialized;

    }

}
