package io.quarkiverse.cxf;

import java.util.Map;
import java.util.Optional;

import io.quarkus.runtime.annotations.ConfigGroup;
import io.quarkus.runtime.annotations.ConfigPhase;
import io.quarkus.runtime.annotations.ConfigRoot;
import io.smallrye.config.ConfigMapping;
import io.smallrye.config.WithDefault;
import io.smallrye.config.WithName;

/**
 * Quarkus CXF build time configuration options that are also available at runtime but only in read-only mode.
 */
@ConfigMapping(prefix = "quarkus.cxf")
@ConfigRoot(phase = ConfigPhase.BUILD_AND_RUN_TIME_FIXED)
public interface CxfFixedConfig {

    /**
     * The build time part of the client configuration.
     */
    @WithName("client")
    public Map<String, ClientFixedConfig> clients();

    @ConfigGroup
    public interface ClientFixedConfig {

        /**
         * The client service interface class name
         */
        public Optional<String> serviceInterface();

        /**
         * Indicates whether this is an alternative proxy client configuration. If
         * true, then this configuration is ignored when configuring a client without
         * annotation `@CXFClient`.
         */
        @WithDefault("false")
        public boolean alternative();

        /** Configuration options related to native mode */
        @WithName("native")
        public NativeClientFixedConfig native_();

        public static ClientFixedConfig createDefault() {
            return new ClientFixedConfig() {

                @Override
                public Optional<String> serviceInterface() {
                    return Optional.empty();
                }

                @Override
                public boolean alternative() {
                    return false;
                }

                @Override
                public NativeClientFixedConfig native_() {
                    return null;
                }

            };
        }
    }

    @ConfigGroup
    public interface NativeClientFixedConfig {

        /**
         * If {@code true}, the client dynamic proxy class generated by native compiler will be initialized at runtime;
         * otherwise the proxy class will be initialized at build time.
         * <p>
         * Setting this to {@code true} makes sense if your service endpoint interface references some class initialized
         * at runtime in its method signatures. E.g. Say, your service interface has method {@code int add(Operands o)}
         * and the {@code Operands} class was requested to be initialized at runtime. Then, without setting this
         * configuration parameter to {@code true}, the native compiler will throw an exception saying something like
         * {@code Classes that should be initialized at run time got initialized during image building: org.acme.Operands ... jdk.proxy<some-number>.$Proxy<some-number> caused initialization of this class}.
         * {@code jdk.proxy<some-number>.$Proxy<some-number>} is the proxy class generated by the native compiler.
         */
        @WithDefault("false")
        public boolean runtimeInitialized();

    }

}
