package io.quarkiverse.cxf;

import java.util.Map;
import java.util.Optional;

import io.quarkus.runtime.annotations.ConfigGroup;
import io.quarkus.runtime.annotations.ConfigPhase;
import io.quarkus.runtime.annotations.ConfigRoot;
import io.smallrye.config.ConfigMapping;
import io.smallrye.config.WithDefault;
import io.smallrye.config.WithName;

/**
 * Quarkus CXF build time configuration options that are also available at runtime but only in read-only mode.
 */
@ConfigMapping(prefix = "quarkus.cxf")
@ConfigRoot(phase = ConfigPhase.BUILD_AND_RUN_TIME_FIXED)
public interface CxfFixedConfig {

    /**
     * The default path for CXF resources.
     *
     * [NOTE]
     * .Earlier versions
     * ====
     * The default value before Quarkus CXF version 2.0.0 was `/`.
     * ====
     *
     * @asciidoclet
     */
    @WithDefault("/services")
    String path();

    /**
     * The size in bytes of the chunks of memory allocated when writing data.
     *
     * This is a very advanced setting that should only be set if you understand exactly how it affects the output IO operations
     * of the application.
     *
     * @asciidoclet
     */
    @WithDefault("128")
    int minChunkSize();

    /**
     * The size of the output stream response buffer in bytes. If a response is larger than this and no content-length is
     * provided then the response will be chunked.
     *
     * Larger values may give slight performance increases for large responses, at the expense of more memory usage.
     *
     * @asciidoclet
     */
    @WithDefault("8191")
    int outputBufferSize();

    /**
     * The build time part of the client configuration.
     *
     * @asciidoclet
     */
    @WithName("client")
    public Map<String, ClientFixedConfig> clients();

    @ConfigGroup
    public interface ClientFixedConfig {

        /**
         * The client service interface class name
         *
         * @asciidoclet
         */
        public Optional<String> serviceInterface();

        /**
         * Indicates whether this is an alternative proxy client configuration. If true, then this configuration is ignored when
         * configuring a client without annotation `@CXFClient`.
         *
         * @asciidoclet
         */
        @WithDefault("false")
        public boolean alternative();

        /**
         * Configuration options related to native mode
         *
         * @asciidoclet
         */
        @WithName("native")
        public NativeClientFixedConfig native_();

        public static ClientFixedConfig createDefault() {
            return new ClientFixedConfig() {

                @Override
                public Optional<String> serviceInterface() {
                    return Optional.empty();
                }

                @Override
                public boolean alternative() {
                    return false;
                }

                @Override
                public NativeClientFixedConfig native_() {
                    return null;
                }
            };
        }
    }

    @ConfigGroup
    public interface NativeClientFixedConfig {

        /**
         * If `true`, the client dynamic proxy class generated by native compiler will be initialized at runtime; otherwise the
         * proxy class will be initialized at build time.
         *
         * Setting this to `true` makes sense if your service endpoint interface references some class initialized at runtime in
         * its method signatures. E.g. Say, your service interface has method `int add(Operands o)` and the `Operands` class was
         * requested to be initialized at runtime. Then, without setting this configuration parameter to `true`, the native
         * compiler will throw an exception saying something like `Classes that should be initialized at run time got
         * initialized during image building: org.acme.Operands ... jdk.proxy<some-number>.$Proxy<some-number> caused
         * initialization of this class`. `jdk.proxy<some-number>.$Proxy<some-number>` is the proxy class generated by the
         * native compiler.
         *
         * @asciidoclet
         */
        @WithDefault("false")
        public boolean runtimeInitialized();
    }
}
