import { LitElement, html, css} from 'lit';
import { pages } from 'build-time-data';
import 'qwc/qwc-extension-link.js';

const NAME = "Morphia";
export class QwcMorphiaCard extends LitElement {

    static styles = css`
      .identity {
        display: flex;
        justify-content: flex-start;
      }

      .description {
        padding-bottom: 10px;
      }

      .logo {
        padding-bottom: 10px;
        margin-right: 5px;
      }

      .card-content {
        color: var(--lumo-contrast-90pct);
        display: flex;
        flex-direction: column;
        justify-content: flex-start;
        padding: 2px 2px;
        height: 100%;
      }

      .card-content slot {
        display: flex;
        flex-flow: column wrap;
        padding-top: 5px;
      }
    `;

    static properties = {
        description: {type: String}
    };

    constructor() {
        super();
    }

    connectedCallback() {
        super.connectedCallback();
    }

    render() {
        return html`<div class="card-content" slot="content">
            <div class="identity">
                <div class="logo">
                    <img src="data:image/png;base64,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"
                                       alt="${NAME}" 
                                       title="${NAME}"
                                       width="32" 
                                       height="32">
                </div>
                <div class="description">${this.description}</div>
            </div>
            ${this._renderCardLinks()}
        </div>
        `;
    }

    _renderCardLinks(){
        return html`${pages.map(page => html`
                            <qwc-extension-link slot="link"
                                extensionName="${NAME}"
                                iconName="${page.icon}"
                                displayName="${page.title}"
                                staticLabel="${page.staticLabel}"
                                dynamicLabel="${page.dynamicLabel}"
                                streamingLabel="${page.streamingLabel}"
                                path="${page.id}"
                                ?embed=${page.embed}
                                externalUrl="${page.metadata.externalUrl}"
                                webcomponent="${page.componentLink}" >
                            </qwc-extension-link>
                        `)}`;
    }

}
customElements.define('qwc-morphia-card', QwcMorphiaCard);