import { LitElement, html, css} from 'lit';
import { pages } from 'build-time-data';
import 'qwc/qwc-extension-link.js';

export class QwcPoiCard extends LitElement {

    static styles = css`
      .identity {
        display: flex;
        justify-content: flex-start;
      }

      .description {
        padding-bottom: 10px;
      }

      .logo {
        padding-bottom: 10px;
        margin-right: 5px;
      }

      .card-content {
        color: var(--lumo-contrast-90pct);
        display: flex;
        flex-direction: column;
        justify-content: flex-start;
        padding: 2px 2px;
        height: 100%;
      }

      .card-content slot {
        display: flex;
        flex-flow: column wrap;
        padding-top: 5px;
      }
    `;

    static properties = {
        extensionName: {attribute: true},
        description: {attribute: true},
        guide: {attribute: true},
        namespace: {attribute: true},
    };


    constructor() {
        super();
    }

    connectedCallback() {
        super.connectedCallback();
    }

    render() {
        return html`<div class="card-content" slot="content">
            <div class="identity">
                <div class="logo">
                    <img src="data:image/svg+xml;base64,PHN2ZyB4bWxucz0iaHR0cDovL3d3dy53My5vcmcvMjAwMC9zdmciIGhlaWdodD0iODAwIiB3aWR0aD0iMTIwMCIgdmlld0JveD0iLTE5LjI4Njc5IC05LjE2MjQ3NSAxNjcuMTUyMTggNTQuOTc0ODUiPjxkZWZzPjxsaW5lYXJHcmFkaWVudCBncmFkaWVudFVuaXRzPSJ1c2VyU3BhY2VPblVzZSIgZ3JhZGllbnRUcmFuc2Zvcm09InRyYW5zbGF0ZSgwIDM1LjIxKSBzY2FsZSguODgxNDUpIiB5Mj0iMjcyLjgxIiB5MT0iMjk2LjIiIHgyPSIxMDEuNDMiIHgxPSIxMDEuNDMiIGlkPSJhIj48c3RvcCBvZmZzZXQ9IjAiIHN0b3AtY29sb3I9IiMwMDAwN2YiLz48c3RvcCBvZmZzZXQ9IjEiIHN0b3AtY29sb3I9IiM1OTkzZGMiLz48L2xpbmVhckdyYWRpZW50PjwvZGVmcz48cGF0aCBmaWxsPSJ1cmwoI2EpIiBkPSJNMzAuODY0IDI5MC4xNXYtNi44M2gzNC4yN2wtLjA1OS0xLjg4Ni0uMDU4LTEuODg3LTE3LjA4My0uMDUyLTE3LjA4My0uMDUyLjA1Ny0yLjM5NC4wNTgtMi4zOTVINDguNWMxNy4zOTkgMCAxNy41NDIuMDAzIDE4LjQ2My40MyAxLjI4OS41OTYgMi4yNTYgMS41IDIuODUgMi42Ni40NjQuOTEuNTAzIDEuMjA2LjQ5NiAzLjc0LS4wMDggMi45ODYtLjE2NCAzLjYxMS0xLjIxMiA0Ljg1My0uMzExLjM3LTEuMDM2LjkyLTEuNjEgMS4yMjRsLTEuMDQ1LjU1LTE1LjAzNy4wNTYtMTUuMDM3LjA1N3Y4Ljc1N2gtNS41MDV6bTQ5LjQwMiA2LjUzNWMtMS42OC0uNTI1LTMuMzIxLTIuMDk1LTMuNzMtMy41Ny0uMTE0LS40MDYtLjIwMi0zLjY3Mi0uMjAzLTcuNDYydi02Ljc0bC41MzYtMS4wNzVjLjU5My0xLjE5IDIuMDM3LTIuNDcxIDMuMjg0LTIuOTEyLjYxMy0uMjE3IDMuODYtLjI3MiAxNi4xNjItLjI3MiAxNS4xOSAwIDE1LjQwNi4wMDYgMTYuMzEyLjQyNyAxLjI1Ny41ODQgMi4xNDcgMS4zNzcgMi44MDcgMi41bC41NTcgMS41NjZ2Ni42MmMwIDcuMDc1LS4wMSA3LjI1Ny0uNDUgOC4wOC0uNjM4IDEuMTkxLTIuMTYzIDIuNDY1LTMuMzg5IDIuODI5LTEuNDI4LjQyNC0zMC41My40MzItMzEuODg1LjAxem0zMC4yMTktMTAuODE3di02LjMybC0xNC4zMjQtLjA1My0xNC4zMjQtLjA1MnY2LjI4OGMwIDMuNDU4LjA2MiA2LjM1LjEzNyA2LjQyNS4wNzYuMDc2IDYuNTIyLjExNCAxNC4zMjQuMDg1bDE0LjE4Ni0uMDUyem0xMi43NDMtLjEwMnYtMTEuMjE0aDUuNTA1djIyLjQyOGgtNS41MDV6IiB0cmFuc2Zvcm09InRyYW5zbGF0ZSgtLjE1NDQgLTI2MC4zNDg0KSIvPjxnIHN0cm9rZS13aWR0aD0iLjI2NSI+PHBhdGggZmlsbD0iIzZkNmU3MSIgZD0iTTg0LjYwNzYgMi4wOTM2Yy4xNS4xNDkuMjI0LjMyOC4yMjQuNTM4IDAgLjIxLS4wNzQuMzg5LS4yMjQuNTQtLjE0OS4xNS0uMzMuMjI1LS41NDIuMjI1LS4yMTIgMC0uMzkyLS4wNzQtLjU0LS4yMjMtLjE0Ny0uMTUtLjIyMS0uMzI5LS4yMjEtLjUzOCAwLS4yMDkuMDc0LS4zOS4yMjMtLjU0YS43MzIuNzMyIDAgMDEuNTQtLjIyNWMuMjEgMCAuMzkxLjA3NC41NC4yMjN6bS0uMDY0IDEuMDE1YS42NTUuNjU1IDAgMDAuMTk1LS40NzUuNjUxLjY1MSAwIDAwLS4xOTMtLjQ3M2MtLjEyOS0uMTMtLjI4Ny0uMTk2LS40NzQtLjE5NnMtLjM0NS4wNjUtLjQ3NS4xOTctLjE5NC4yOS0uMTk0LjQ3NS4wNjUuMzQzLjE5Mi40NzRjLjEyOC4xMy4yODYuMTk2LjQ3My4xOTZzLjM0NS0uMDY3LjQ3Ni0uMTk4em0tLjEwNy0uNjIyYzAgLjEzNi0uMDYuMjIyLS4xOC4yNTdsLjIyLjI5N2gtLjE3OGwtLjE5OC0uMjc2aC0uMTh2LjI3NmgtLjE0NHYtLjgyNGguMzFjLjEyNSAwIC4yMTQuMDIxLjI2OS4wNjQuMDU0LjA0My4wOC4xMTEuMDguMjA2em0tLjE4NC4xMTZjLjAyOS0uMDI0LjA0Mi0uMDY0LjA0Mi0uMTE3IDAtLjA1My0uMDE0LS4wOTItLjA0My0uMTEzLS4wMy0uMDIyLS4wODItLjAzMi0uMTU2LS4wMzJoLS4xNzZ2LjI5OGguMTdjLjA4LS4wMDEuMTM0LS4wMTMuMTYzLS4wMzZ6Ii8+PHBhdGggZmlsbD0iI2QyMjEyOCIgZD0iTTgyLjA3MjYgMTEuMDM2NnYuODVoLTYuNTM0di05LjcyNWg2LjQxMXYuODVoLTUuNDY1djMuNTA2aDQuNzY3di44MDhoLTQuNzY3djMuNzEyaDUuNTg4em0tMjEuNTE4LTguOTMxYzEuNzI1IDAgMi45ODYuODc3IDMuNTg4IDIuMTI0bC0uNzY3LjQ2NmMtLjYxNi0xLjI4OC0xLjgyMi0xLjcyNi0yLjg3Ny0xLjcyNi0yLjMyOSAwLTMuNjAyIDIuMDgzLTMuNjAyIDQuMDQxIDAgMi4xNSAxLjU3NiA0LjA5NSAzLjY0MyA0LjA5NSAxLjA5NiAwIDIuMzg0LS41NDggMy0xLjgwOGwuNzk0LjQxYy0uNjMgMS40NTItMi4zNTYgMi4yNi0zLjg0OSAyLjI2LTIuNjcgMC00LjU0Ny0yLjU0Ny00LjU0Ny01LjAxMyAwLTIuMzQxIDEuNjg1LTQuODQ4IDQuNjE3LTQuODQ4em0xMi43ODguMDU2djkuNzI1aC0uOTZ2LTQuNTQ3aC01Ljc2NnY0LjU0N2gtLjk0NHYtOS43MjVoLjk0NHY0LjMyOWg1Ljc2N3YtNC4zMjl6bS0yNi4zMDcgOS43MjVsNC4wNjctOS43MjVoLjc4MWw0LjA2OCA5LjcyNWgtMS4wMjZsLTEuMjYtMy4wNGgtNC4zN2wtMS4yNDcgMy4wNHptNC40NjUtOC42MDJsLTIgNC44MDhoMy45NDZ6bS0xMC44MzcgOC42MDJ2LTkuNzI1aDQuMDI3YzEuNjcgMCAyLjg3NyAxLjU0OCAyLjg3NyAzLjA1NCAwIDEuNTktMS4xMjMgMy4wODItMi43OTQgMy4wODJoLTMuMTY1djMuNTg4aC0uOTQ1em0uOTQ1LTQuNDM4aDMuMTFjMS4xNSAwIDEuODktMS4wNTUgMS44OS0yLjIzMyAwLTEuMjItLjkwNC0yLjIwNS0xLjk3My0yLjIwNWgtMy4wMjd6bS0xMC45MTMgNC40MzhsNC4wNjktOS43MjVoLjc4bDQuMDY5IDkuNzI1aC0xLjAyOGwtMS4yNi0zLjA0aC00LjM3bC0xLjI0NyAzLjA0em00LjQ2Ny04LjYwMmwtMiA0LjgwOGgzLjk0NnoiLz48L2c+PHBhdGggZmlsbD0icmVkIiBkPSJNLjA3MDYgMzYuMzQxNmMtLjYwNS0uNjA0IDIuODA3LTcuMzI1IDUuMjYyLTEwLjM2Ni45OTctMS4yMzQgMi42NTEtMi45NDYgMy42NzctMy44MDUgMS4wMjYtLjg1OCAxLjg2Ny0xLjc0NSAxLjg3LTEuOTcxLjAwMy0uMjI2LS40NTgtLjY3MS0xLjAyMy0uOTg5LTEuOTQzLTEuMDkyLTQuOC00LjA1Ny02LjMyLTYuNTYtMS4wMTUtMS42NzItMS41Ny0yLjcxLTIuMDc4LTIuODk2LS40NDUuMjM4LS44MDUtLjItLjgwMi0uNDgtLjQ2Ni0xLjE1MyAxLjY3OC0yLjM0NSAyLjU4MS0yLjA1OC44NjEuMTQ4IDEuMTEzIDEuNzI4IDEuMzUgMi4zNTIuNTQ2IDEuNDQyLjQ3MyAxLjMgMS4wNTMgMi40NTggMS4yNzMgMi41NDUgMy4zMjcgNC43MjggNS40NTcgNS44MDQgMS42My44MjMgMi4yMTMgMS42MjcgMS44NjMgMi41NzMtLjExOC4zMTktMS45NjggMi4zMjYtNC4xMSA0LjQ2LTMuOTcyIDMuOTU3LTUuODA4IDYuMzQ5LTcuMDM2IDkuMTY3YTQ5LjQgNDkuNCAwIDAxLS44NyAxLjkxNGMtLjIyOS40MzMtLjY0OC42MjMtLjg3NC4zOTd6bTI2LjQ3Ny0uNTVjLS4yMS0uMjA5LjAyOC0uNjE5LjM1OC0uNjE5LjEzNiAwIC4yNDcuMTc4LjI0Ny4zOTQgMCAuNDA2LS4zMS41MjEtLjYwNS4yMjZ6bS0xLjYwOC0zLjU1N2MtLjY1NC0xLjM3MS0xLjgxLTMuMjUxLTIuNTctNC4xNzgtMS40My0xLjc0MS01Ljk4OC01LjcyNi02LjI1NS02LjAwMy0uNTYtLjU4MS0xLjU2OC0uODQ3LTEuMzkyLTIuMjguMDgtLjMwOCAxLjE1My0xLjI0OCAyLjQzOC0yLjEzOCAzLjMyLTIuMjk2IDQuOTA1LTQuNDUxIDUuOTI5LTguMDU3LjUyOS0xLjg2NCAxLjE0MS0yLjYxMSAxLjkzMy0yLjM2LjcwMi4yMjMuNjk2LjUxNy0uMDU1IDIuNTg4LTEuMjkzIDMuNTYtMy4zMjYgNi4wMTYtNy4wNjQgOC41MzItMS4xMTkuNzUzLTIuMDU2IDEuMzE3LTIuMDM0IDEuNzgxLjAzNy43ODcuNzEyIDEuMTgyIDIuNzU1IDIuOTk4IDEuNzk2IDEuNTk3IDQuMDI4IDQuMTQ5IDUuMzgxIDYuMTU1IDEuNjA3IDIuMzggMi43MzQgNC44NDQgMi4zNzkgNS4yLS4xNjcuMTY2LS42NzMtLjYyLTEuNDQ1LTIuMjM4eiIvPjxnIGZpbGw9IiMwMDdmMDAiIHRyYW5zZm9ybT0idHJhbnNsYXRlKC0uMTU0NCAtMjYwLjM0ODQpIj48Y2lyY2xlIHI9Ii44ODUiIGN5PSIyNzMuOSIgY3g9IjEzLjk0OCIvPjxjaXJjbGUgcj0iLjcwMiIgY3k9IjI3NS41OSIgY3g9IjE1LjYzOSIvPjxjaXJjbGUgcj0iLjQ4NCIgY3k9IjI3NS42MiIgY3g9IjE0LjA2NSIvPjxjaXJjbGUgcj0iLjMzNiIgY3k9IjI3Ny41OCIgY3g9IjE1LjAxNSIvPjxjaXJjbGUgcj0iLjI0NCIgY3k9IjI3Ni44OSIgY3g9IjE0LjEyNiIvPjxjaXJjbGUgcj0iLjMzNiIgY3k9IjI3OS42NCIgY3g9IjEzLjk1Ii8+PGNpcmNsZSByPSIuMjE1IiBjeT0iMjgxLjYyIiBjeD0iMTQuMzQ4Ii8+PGNpcmNsZSByPSIuMjE1IiBjeT0iMjgzLjQ1IiBjeD0iMTIuNzk4Ii8+PC9nPjxnIGZpbGw9InJlZCIgdHJhbnNmb3JtPSJ0cmFuc2xhdGUoLS4xNTQ0IC0yNjAuMzQ4NCkiPjxjaXJjbGUgcj0iLjg4NSIgY3k9IjI3NS4yIiBjeD0iMTEuNTkzIi8+PGNpcmNsZSByPSIuNzAyIiBjeT0iMjc0Ljk2IiBjeD0iOS41MyIvPjxjaXJjbGUgcj0iLjcwMiIgY3k9IjI3Ny4zNCIgY3g9IjEyLjAzNSIgdHJhbnNmb3JtPSJ0cmFuc2xhdGUoMS40MDYgMzAuOTMyKSBzY2FsZSguODg4MzkpIi8+PGNpcmNsZSByPSIuNzAyIiBjeT0iMjc3LjM0IiBjeD0iMTIuMDM1IiB0cmFuc2Zvcm09InRyYW5zbGF0ZSg0LjMyOCAxNDQuODcpIHNjYWxlKC40NzM4NSkiLz48Y2lyY2xlIHI9Ii43MDIiIGN5PSIyNzcuMzQiIGN4PSIxMi4wMzUiIHRyYW5zZm9ybT0idHJhbnNsYXRlKDEwLjk4NyAxNTUuMjMpIHNjYWxlKC40NDAyNSkiLz48Y2lyY2xlIHI9Ii43MDIiIGN5PSIyNzcuMzQiIGN4PSIxMi4wMzUiIHRyYW5zZm9ybT0idHJhbnNsYXRlKDYuNzk4IDE4MC45KSBzY2FsZSguMzQ2MykiLz48Y2lyY2xlIHI9Ii43MDIiIGN5PSIyNzcuMzQiIGN4PSIxMi4wMzUiIHRyYW5zZm9ybT0idHJhbnNsYXRlKDEwLjcwNyAxODIuNDMpIHNjYWxlKC4zNDYzKSIvPjxjaXJjbGUgcj0iLjcwMiIgY3k9IjI3Ny4zNCIgY3g9IjEyLjAzNSIgdHJhbnNmb3JtPSJ0cmFuc2xhdGUoOS45MyAxODQuNTUpIHNjYWxlKC4zNDYzKSIvPjxjaXJjbGUgcj0iLjcwMiIgY3k9IjI3Ny4zNCIgY3g9IjEyLjAzNSIgdHJhbnNmb3JtPSJ0cmFuc2xhdGUoOC44ODYgMTg2LjM2KSBzY2FsZSguMzQ2MykiLz48Y2lyY2xlIHI9Ii43MDIiIGN5PSIyNzcuMzQiIGN4PSIxMi4wMzUiIHRyYW5zZm9ybT0idHJhbnNsYXRlKDEwLjE4MSAxODYuNCkgc2NhbGUoLjM0NjMpIi8+PGNpcmNsZSByPSIuNzAyIiBjeT0iMjc3LjM0IiBjeD0iMTIuMDM1IiB0cmFuc2Zvcm09InRyYW5zbGF0ZSgxMS4yMzQgMTg3LjY2KSBzY2FsZSguMzQ2MykiLz48L2c+PGcgZmlsbD0iIzAwZiIgdHJhbnNmb3JtPSJ0cmFuc2xhdGUoLS4xNTQ0IC0yNjAuMzQ4NCkiPjxjaXJjbGUgcj0iLjcwMiIgY3k9IjI3NC41NyIgY3g9IjE3LjQ3NSIgdHJhbnNmb3JtPSJ0cmFuc2xhdGUoLTIuMzY0IC0zNy4xNDYpIHNjYWxlKDEuMTM1MykiLz48Y2lyY2xlIHI9Ii43MDIiIGN5PSIyNzQuNTciIGN4PSIxNy40NzUiIHRyYW5zZm9ybT0idHJhbnNsYXRlKDUuOTE0IDkwLjU0NSkgc2NhbGUoLjY3NjgxKSIvPjxjaXJjbGUgcj0iLjcwMiIgY3k9IjI3NC41NyIgY3g9IjE3LjQ3NSIgdHJhbnNmb3JtPSJ0cmFuc2xhdGUoMTAuNTEgMTQyLjUzKSBzY2FsZSguNDgxMzkpIi8+PGNpcmNsZSByPSIuNzAyIiBjeT0iMjc0LjU3IiBjeD0iMTcuNDc1IiB0cmFuc2Zvcm09InRyYW5zbGF0ZSgxMy42NTIgMTk1LjA0KSBzY2FsZSguMjkzNDkpIi8+PGNpcmNsZSByPSIuNzAyIiBjeT0iMjc0LjU3IiBjeD0iMTcuNDc1IiB0cmFuc2Zvcm09InRyYW5zbGF0ZSg3LjM5OCAxODYuOCkgc2NhbGUoLjMyMzU2KSIvPjxjaXJjbGUgcj0iLjcwMiIgY3k9IjI3NC41NyIgY3g9IjE3LjQ3NSIgdHJhbnNmb3JtPSJ0cmFuc2xhdGUoLjMwNCA2Ny43MTQpIHNjYWxlKC43NjYwOSkiLz48Y2lyY2xlIHI9Ii4yMTUiIGN5PSIyODEuNiIgY3g9IjEzLjU2Ii8+PGNpcmNsZSByPSIuMjE1IiBjeT0iMjgzLjE5IiBjeD0iMTMuODIiLz48Y2lyY2xlIHI9Ii4yMTUiIGN5PSIyODIuOTQiIGN4PSIxNS4xMzEiLz48L2c+PGcgZmlsbD0iI2MwYWY4MyIgdHJhbnNmb3JtPSJ0cmFuc2xhdGUoLS4xNTQ0IC0yNjAuMzQ4NCkiPjxjaXJjbGUgcj0iLjIxNSIgY3k9IjI4My43IiBjeD0iMTIuMDEzIi8+PGNpcmNsZSByPSIuMjE1IiBjeT0iMjg0Ljc4IiBjeD0iMTEuNTExIi8+PGNpcmNsZSByPSIuMjE1IiBjeT0iMjg0LjUxIiBjeD0iMTIuNTYyIi8+PGNpcmNsZSByPSIuMjE1IiBjeT0iMjg0Ljc2IiBjeD0iMTMuMzEyIi8+PGNpcmNsZSByPSIuMjE1IiBjeT0iMjgzLjk4IiBjeD0iMTMuNjEzIi8+PGNpcmNsZSByPSIuMjE1IiBjeT0iMjgzLjk3IiBjeD0iMTQuNjA1Ii8+PGNpcmNsZSByPSIuMjE1IiBjeT0iMjg0Ljc4IiBjeD0iMTQuODcxIi8+PGNpcmNsZSByPSIuMjE1IiBjeT0iMjg0Ljc3IiBjeD0iMTUuNjc0Ii8+PGNpcmNsZSByPSIuMjE1IiBjeT0iMjg0LjUiIGN4PSIxNi40MTgiLz48L2c+PGcgdHJhbnNmb3JtPSJ0cmFuc2xhdGUoLTM5Mi45OTQ0IC01NDUuMjk4NCkgc2NhbGUoMy4xNTk3KSI+PGcgc3Ryb2tlLXdpZHRoPSIwIiB0cmFuc2Zvcm09InRyYW5zbGF0ZSguMDEyIC0xLjcyOCkiPjxlbGxpcHNlIHJ5PSIuNjQ1IiByeD0iMi4yMTgiIGN5PSIxODQuMjgiIGN4PSIxMjguNzgiLz48ZWxsaXBzZSByeT0iLjY0NSIgcng9IjIuMjE4IiBjeT0iMTg0LjM2IiBjeD0iMTI4Ljc4Ii8+PGVsbGlwc2UgZmlsbD0iI2NjYWY1ZiIgcnk9Ii42MTgiIHJ4PSIyLjE4MyIgY3k9IjE4NC4yNyIgY3g9IjEyOC43OCIvPjwvZz48cGF0aCBkPSJNMTI4LjgyNCAxODAuNDQ5Yy0uNjYyLS4wMTMtMS4zNTcuMTc4LTEuMzU2LjM5NGwuMDAxLjM4MmMuMS45NjguNjAzIDEuNjAyIDEuMjkgMS41OTguNjg1LS4wMDQuOTU0LS4zNCAxLjIwNy0uODQ2IDEuMDkyLS41NDMuNzM3LS45MDQuMTE0LS44MmwuMDA5LS4zNGMuMDA2LS4yMS0uNjAzLS4zNTUtMS4yNjUtLjM2OHptMS41MDEuODMxYy4xODUuMDE2LjE3Ny4xNzkuMDU1LjNsLS4zODIuMjg3Yy4wMjctLjE3NS4wNjktLjE4NS4wOTYtLjU3NnoiLz48cGF0aCBmaWxsPSIjY2NhZjVmIiBkPSJNMTI3LjUyIDE4MS4zYy4wMjcuMjg5LjI5OCAxLjQ1NyAxLjI2IDEuNDgyLjk2Mi4wMjYgMS4yOC0uOTggMS4yNjctMS40OC0uODMuNDU2LTEuODg3LjI5My0yLjUyNy0uMDAyeiIvPjxwYXRoIGZpbGw9IiNmZmYiIGQ9Ik0xMjcuNTEgMTgxLjI0Yy4xODIuMTA0Ljg2Mi4zMDQgMS4yNTcuMjk5LjM5NS0uMDA2LjY0IDAgMS4yNzItLjI5MWwuMDA2LS4yNTZjLS4wNC4xMDMtLjY3NC4zMjYtMS4yNzcuMzI4LS42MDMuMDAyLTEuMDQ2LS4xMjYtMS4yNi0uMzM0eiIvPjxlbGxpcHNlIGZpbGw9IiNmZmYiIHJ5PSIuMzciIHJ4PSIxLjI3NSIgY3k9IjE4MC44NiIgY3g9IjEyOS41MyIgdHJhbnNmb3JtPSJza2V3WCgtLjIzNikgc2NhbGUoMSAuOTk5OTkpIi8+PHBhdGggZmlsbD0iI2MwYWY4MyIgZD0iTTEyNy43MSAxODEuMDZjLjIxNi4wODguNDc4LjIxNCAxLjA1LjE5OS41NzQtLjAxNi45NC0uMTEyIDEuMTEyLS4yMS0uMjM1LS4xNTItLjQ1OC0uMjQxLTEuMDUxLS4yNDgtLjU5My0uMDA3LS45MzUuMTEtMS4xMTEuMjU5eiIvPjwvZz48cGF0aCBmaWxsPSIjY2RjZGNkIiBkPSJNOS43OTk2Ljg3MTZsLjAwNS4wMTJ2LS4wMDhjMC0uMDAyLS4wMDQtLjAwMi0uMDA1LS4wMDN6bTQuNDIuMDFsLS4wMDQuMDAyYy0uMTU4LjAxMi0uMzM1LjA3Ny0uNjI3LjEzNS0uMDg4LjAxOC0uMTc4LjAzLS4yOTEuMDQ1LS4wNTUuMDA2LS4xMDMuMDE0LS4xNjQuMDE5YTYuMTUyIDYuMTUyIDAgMDEtLjY1NC4wMiA2NS43ODcgNjUuNzg3IDAgMDEtMS43MDQtLjA0NyA2Ljk2MyA2Ljk2MyAwIDAxLS42MzItLjA1bC0uMDQ2LS4wMDRjLjAyMy4xNS4wNjUuMzkuMDQ4Ljc2NWwuMDI1LjAwMmMuMzMyIDAgLjU2My4wMDkuNzQ1LjAyNWguMDA2Yy42NzUtLjAwNSAxLjA2LjI1NCAyLjg4My0uMDg5LjY0LS4xMDkuOTEuMjg2IDEuMTQ3LjQ5NC44NzguNzcyIDEuNzA1IDEuMjU5IDIuNDM4LjkwOC43MjItLjE3NyAxLjEwMy0uNzg1IDEuODYzLTEuMTE0LjAwNC0uMjk0LS4wMS0uNTU4LS4wNTYtLjczNC0uMDQ2LjAzNi0uMDY2LjA1NC0uMTIyLjA5NGE5Ljc0IDkuNzQgMCAwMS0uODM1LjU0NGMtLjY0Ny4zMS0xLjA2LjYxMy0xLjUyMy41OTZhMS4wNzYgMS4wNzYgMCAwMS0uMzYxLS4wOGMtLjI1Mi0uMS0uNTM2LS4zLS44ODMtLjYzOGE5Ljk1IDkuOTUgMCAwMS0uNTc2LS42MTloLS4wMDJhMi4wNjIgMi4wNjIgMCAwMC0uMzU1LS4yMDUuODkzLjg5MyAwIDAwLS4wNzEtLjAyNy42MjIuNjIyIDAgMDAtLjI0Ny0uMDQyem0tMi45MzcuOTk1Yy4wMTMuMDA2LjAzLjAxMy4wNC4wMi0uMDEtLjAwNy0uMDI3LS4wMTQtLjA0LS4wMnptLjA5Ni4wNzV6bS0uMzMzLjc2MmMtLjI4OC4wNjItLjY4OC0uMDIzLTEuMDY2LjA1LTEuMDE3IDAtMS45MDguMzM4LTIuODExIDEuMDU5LS40OC4zODItLjgyLjQ4Mi0xLjU2Ny43MzUtLjM4Ni4xMy0uNjg2LjI0MS0uOTQuMzMyYTUuNjg0IDUuNjg0IDAgMDEtLjQ0Mi4xNDJjLS4wMDEgMC0uMDA0IDAtLjAwNi4wMDItLjA4LjAyLS4xNTYuMDM2LS4yMTMuMDM4bC4yNzcuOTQyYy4wMTggMCAuMDM4LjAwNi4wNi4wMzYuOTg3LS41MSAxLjA3NC0uNTg2IDEuNzc0LS43NTYgMCAwIC41NDUgMS44Mi41ODIgMi4xMDcuMDM4LjI4Ni43NSAxLjk5NyAxLjAyMiAyLjczMi4yNzIuNzM1Ljk4IDIuMTAxIDEuMDc5IDIuNDkzLjE2LjE2LjE1My4wMDUgMS43NjQtLjU3OS43MDMtLjM1NiAxLjQyLS42NTggMS41OTEtLjc3Mi0uMDMtLjEwOC0uMDU4LS4yNjctLjA4OS0uMzk0LS4wMjMuMTgtLjEyMi4yMTQtLjMyMy4yOTFsLTIuNzgxIDEuMzA4LS4yMy0uNmEzMC4wMjIgMzAuMDIyIDAgMDAtLjgzNi0xLjg0N2MtLjQzNS0uODk4LS44MzItMS45ODYtLjk3LTIuNzUxbC0uNTE4LTIuMTA1Yy45NzQtLjc3IDIuNTk1LTEuNSAzLjY2Ni0xLjY5M2wuOTc3LS4xNzd6bS0uOTMgMS4yMjNjLS41NS4wNjUtMS41NDEuNTE3LTIuMzY3IDEuMTI0LS43NC41NDMtLjguNjg4LS42MDkgMS40NTdhMjYuODk1IDI2Ljg5NSAwIDAwMS4zNDUgNC4wMzVoLjAwMWMuMzE1Ljc2LjYwNiAxLjM4MS42NDggMS4zODEuMDQxIDAgMS42OTUtLjcyIDIuMzU1LTEuMDQuMTA2LS4xMzguMS0uMjEtLjAxOC0uNzgyLS4wNzQtLjM2LS4yNzgtMS44MjgtLjQ1My0zLjI2MS0uMTc2LTEuNDM0LS40MTEtMi43MTItLjUyMi0yLjg0LS4wNjMtLjA3My0uMTk3LS4wOTUtLjM4LS4wNzR6bS4wMTguMzE5Yy40MS0uMTMxLjczOCAzLjI4IDEuMDM1IDUuMzM2LjAwNS4xOTUuMjUuOTYtLjA2IDEuMDczbC0xLjgwNy43NzQtLjU0OS0xLjQ0M2ExOS40OTQgMTkuNDk0IDAgMDEtLjk0Mi0yLjgzNWMtLjMzMy0xLjQzNC0uMzI1LTEuNDguMjc3LTEuODc0LjM0LS4yMjMgMS42NjUtLjg4NiAyLjA0Ni0xLjAzMXptNy40OC40MjNjLS4wODcuMTA2LS4xNjEuMzUzLS4yMy42ODJsLjkzMy4xNTVjLjUxOC4wODUgMS40ODIuMTU3IDIuMTQ0LjE1OS44Mi4wMDEgMS40MjMuMTc0IDEuODkzLjU0NGwuODgyLjU4MmMuMTYtLjI3Mi4yNjctLjUwNy4yNzYtLjg2NGwtLjAxOS0uMDFjLS4yMDctLjA4Mi0uNDc3LS4yMzEtLjkxMS0uNDY3YTMuOTAyIDMuOTAyIDAgMDAtLjUyMi0uMjA0IDExLjMxIDExLjMxIDAgMDAtMS40NzgtLjM1NiAxMC42OTQgMTAuNjk0IDAgMDAtLjc3NS0uMTA4Yy0uOTQ4LS4wOTQtMS45MjUtLjA3Mi0yLjE5My0uMTEzem0uMDYgMS4yMWwtLjUgMy4wODljLS4zNDcgMi45OTctLjcxOCAzLjI4Ny0uMTQ0IDMuMDMxIDAtLjg5Mi4zMTYtMy4xNS40MzgtMy40NDJsLjQxLTIuMzE2LjgyMS4yNmMuNDUxLjE0MiAxLjE3LjI0OSAxLjYwMS4yMzYgMS44MzkuMjA3IDEuNTU1LjI0MS44ODMgMi4wNzNoLjAwMmMtLjM0MSAxLjAyNS0uNTE3IDIuMzU5LS45NCAzLjQwOS0uMjI0LjcyOC0uMTk3LjY1OC0uMDY3LjY1OC4zNyAwIC43NTgtMS42NzIgMS4wMjYtMi44NjcuMTM0LS41OTkuNDM0LTEuNTQ0LjY2Ni0yLjEwMi40MjYtMS4wMTguMjkyLTEuMzk3LS41MjYtMS40NzZhMjkuMjE2IDI5LjIxNiAwIDAxLTIuNjE4LS4zNjF6Ii8+PHBhdGggZmlsbD0iIzYxNjE2MiIgZD0iTTE5LjIzMzYuMzE4NmMtLjEyLjAwNi0uMjc0LjA2LS40Ny4xNjktLjAyNy4wMi0uMDI3LjAyNC0uMDUzLjA0NC4zMTYtLjE0Ny41NTctLjIxOS42NTctLjE5YS4zMy4zMyAwIDAwLS4xMzQtLjAyM3ptLjE4MS4wNDhjLjA5OC4yMi4xNi4zMzcuMTg1LjQxMi0uMDE2LS4yMDMtLjA3NC0uMzQ0LS4xODUtLjQxMnptLTkuNjEyLjUwN2wtLjAwNC4xMjNjLjA0Ny4zMjUuMDkuNTg4LjAwOC43OTIuMTA0LS4wMS4yMTEtLjAyMy4zMS0uMDI0aC4wMjdjLjAxOC0uMzc1LS4wMjUtLjYxNS0uMDQ4LS43NjUtLjE2LS4wMjktLjI0NS0uMDctLjI5My0uMTI2em05Ljc5Mi4wNWEuNDQ0LjQ0NCAwIDAxLS4wNzEuMDc0IDguNTA3IDguNTA3IDAgMDEtLjMyOC4yNmMuMTMuNS4wOTggMS41OC0uMzMgMi42NjYuMTM1LjAwMS4yMTYtLjAwNy4zNDQtLjAwOC4wNTYtMS4wMTMuMzktMi4yNjMuMzg1LTIuOTkzem0tOC4yMjcgMS40ODRjLS4wMi4wOC0uMDQuMTQ1LS4wNC4xNDVsLjU1OC4wODNjLS4wMS4xNTctLjAxNC4zMTctLjAyNC40NzUtLjE2LS4wMjMtLjMyMS0uMDU0LS40ODItLjA3NWwtLjAwMy4wMDdjLS4wMTgtLjI0Mi0uMDMyLS4zMzQtLjA1LS40OS0uMDYuMDgtLjE2Mi4xMjItLjI4My4xNDN2MS4zMDNjLjE4OSAyLjY5My4zMTQgMy4yMDQuNjUgNC44NjkuMjQ3IDEuOTI4LjY5NyAzLjUzNC44NTkgMy4wNjMtLjAyNS0uMTktLjE0LS43NTYtLjIxMi0xLjE0NC4yMzYuMDM0LjQ3MS4wNjkuNzA4LjExOC0uMDEuMS0uMDE3LjItLjAyOC4yOTl2LjAwM2wuMTYtLjA0My4wMjktLjIyMWMuMjU1LjA1Ni41MS4xMTEuNzY3LjE4Ni4xMDYuMDQzLjE5NS4wNy4yOTQuMTA3LS4wMTkuMTIzLS4wMzQuMjQ3LS4wNTUuMzdsLjAwMS4wMDUuMTYyLS4wMzVjLjAxOC0uMDkyLjAzNC0uMTg4LjA1MS0uMjgxLjM2LjEyNS42NzIuMjEuOTYxLjI3My0uMDMuMTM4LS4wNTUuMjc1LS4wODcuNDEydi4wMDVsLjE2NC0uMDI1Yy4wMy0uMTE4LjA1OC0uMjQyLjA4Ny0uMzYxLjE3NC4wMzQuMzQ1LjA2Ny41MTMuMDk3LS4wNDIuMTM2LS4wNzguMjgyLS4xMjQuNDEyLjI1MS4yMDQuMTUuMDkuMzAyLS4xNTYuMjEyLS4zNDQuNjItMi42OTUuOTk3LTUuNzU1LjEwOC0uODc1LjIyMS0xLjMzOS4zNy0xLjUyLS4wMTItLjAwMS0uMDYyIDAtLjA3LS4wMDMtLjEyMy0uMDI5LS4yMzItLjA0NC0uMjU4LS4xOTQtLjAwMy4wMDMtLjAwMS4wMDItLjAwMy4wMDVhMTYuNDUgMTYuNDUgMCAwMS0uNDUyLS4wOTdjLjAzOS0uMTUuMDczLS4zMDIuMTE0LS40NS4wOTEuMDE1LjI2NC4wNi40MS4wOTUuMDE3LS4wNTcuMDE5LS4wODcuMDQyLS4xNTZ2LS4wMDFjLjAxMy4wMDEuMDc5LjAwMS4wOTYuMDAzLS4yNDQtLjA0NS0uNTM1LS4wOTctLjY3Mi0uMTI2LS40ODEtLjExLS45NTMtLjI0Ni0xLjQyMi0uMzkxbC0uMTYtLjA1MWMtLjMwMy0uMDk2LS42MDctLjE5LS45MDgtLjI5My0uMTg1LS4wNjItLjM3LS4xMTMtLjU1NS0uMTY0bC0uMTYtLjA0NmExNC45NSAxNC45NSAwIDAwLTEuNTU3LS4yOTZsLS4xNjUtLjAyNy0uNTI2LS4wNzZ6bS00LjAxNi4xMzRhMi4zNiAyLjM2IDAgMDAtLjU0Ni4yNTZjLjE4LS4wOTUuMzYzLS4xNzIuNTQ2LS4yNTZ6bTQuNjk4LjExOWMuNTEuMDggMS4wMjUuMTg3IDEuNTQuMzItLjAyMy4xNDUtLjA0LjI5My0uMDYyLjQzOS0uNS0uMTI4LTEtLjIwOC0xLjUwMi0uMjgzLjAxLS4xNTguMDEzLS4zMTkuMDI0LS40NzZ6bTEuNzAxLjM2MWMuMTcuMDQ2LjM0LjA3Ny41MS4xMy4zNC4xNDMuNjUuMjU5Ljk0LjM1NS0uMDI5LjEzNC0uMDUyLjI3LS4wNzkuNDAzLS4yOC0uMDktLjU2My0uMTczLS44NDEtLjI2OWExMC40NTEgMTAuNDUxIDAgMDAtLjU5LS4xNzVjLjAyLS4xNDcuMDM3LS4yOTcuMDYtLjQ0NHptLTIuMzY0LjE2NmMuMTU1LjAyMS4zMTIuMDUxLjQ2Ny4wNzQtLjAxLjE3OS0uMDIuMzU3LS4wMjcuNTM3LS4xMzUtLjAyLS4yNy0uMDQ0LS40MDUtLjA2NHptLjYzMy4xYy40OTIuMDc3Ljk4Ni4xOCAxLjQ4My4zMDctLjAyNC4xNjctLjA0OS4zMzMtLjA3LjUwMWExNC45MyAxNC45MyAwIDAwLTEuNDM5LS4yN2MuMDA3LS4xOC4wMTctLjM2LjAyNi0uNTM4em0zLjM0LjI2N2MuNTI2LjE2Ny45ODkuMjc2IDEuNDIuMzU1LS4wNC4xNDYtLjA3NS4yOTUtLjExNC40NDJhMTkuODEgMTkuODEgMCAwMS0xLjM4NC0uMzkxYy4wMjYtLjEzNS4wNS0uMjcyLjA3OC0uNDA2em0tMS42OTMuMDgxYy4xODIuMDQ5LjM2My4wODMuNTQ1LjEzOS4zMTQuMTMzLjYuMjQuODcuMzMyLS4wMy4xNTYtLjA2MS4zMTEtLjA5LjQ2OC0uMjYyLS4wODQtLjUyNy0uMTYzLS43ODctLjI1M2ExMC40NjYgMTAuNDY2IDAgMDAtLjYwOC0uMThjLjAyLS4xNy4wNDYtLjMzNy4wNy0uNTA2em0tMi4yMzYuMjUybC4zODguMDYyYy0uMDA2LjE3Ny0uMDIyLjM1LS4wMjUuNTI4bC0uMzMtLjA1Yy0uMDEtLjE5OC0uMDIxLS4zNS0uMDMzLS41NHptLjU1Ni4wODljLjQ3Mi4wNzUuOTQ2LjE3NCAxLjQyMi4yOTQtLjAyLjE2NS0uMDQ2LjMyNy0uMDY0LjQ5M2ExNS4xMDkgMTUuMTA5IDAgMDAtMS4zNzYtLjI1N2MuMDAyLS4xNzkuMDEyLS4zNTMuMDE4LS41M3ptMy4yNTYuMTgxYy41MTIuMTY3Ljk2My4yNzcgMS4zODIuMzU3LS4wNDQuMTY2LS4wODkuMzMyLS4xMy41LS40NTMtLjExLS44OTgtLjI0OS0xLjM0Mi0uMzg5LjAyNy0uMTU4LjA2LS4zMTIuMDktLjQ2OHptLTEuNjcuMTU1Yy4xODguMDUuMzc0LjA4NS41NjMuMTQzLjI5Ni4xMjUuNTYzLjIyNC44Mi4zMTMtLjAyNy4xNTMtLjA1OC4zMDMtLjA4Mi40NTctLjIzMS0uMDc2LS40NjUtLjE0My0uNjk1LS4yMjNhMTAuNTUgMTAuNTUgMCAwMC0uNjYyLS4xOTZjLjAxNi0uMTY2LjAzNy0uMzMuMDU2LS40OTR6bTMuMjE3LjIzYy4xNTYuMDI4LjMxMy4wNTYuNDY2LjA4MS0uMDMyLjE2Ni0uMDY1LjMzNC0uMDk1LjUzMmExNi41NCAxNi41NCAwIDAxLS41LS4xMDdjLjA0LS4xNy4wODYtLjMzNy4xMjktLjUwNnptLTUuMzE2LjAzN2MuMTA3LjAxOC4yMTMuMDM1LjMyLjA1LS4wMDIuMjAzLS4wMDIuNDA2LjAwMi42MS0uMTMtLjAyLS4yMDQtLjA0MS0uMjg1LS4wNjMtLjAxNC0uMjU0LS4wMjItLjMyOS0uMDM3LS41OTd6bS40OTguMDhjLjQ1LjA3Mi45MDMuMTY1IDEuMzU3LjI3OC0uMDE4LjE5Mi0uMDM1LjM4Ni0uMDQ5LjU4LS40MzItLjEwOC0uODYyLS4xOC0xLjI5NC0uMjQ4LS4wMDMtLjIwMy0uMDE1LS40MS0uMDE0LS42MXptMy4xNDQuMTY1Yy40OTcuMTY0LjkzMi4yNyAxLjMzOC4zNS0uMDM4LjE2Mi0uMDguMzIyLS4xMTYuNDg1YTIxLjQ2IDIxLjQ2IDAgMDEtMS4yOTgtLjM4MmMuMDIzLS4xNTIuMDUxLS4zMDIuMDc2LS40NTN6bS01LjA1Ni4wNmMtLjEwNi0uMDIxLS40NTQuMTE0LS43MS4yNDUtLjI4Ny4xNDctLjY5Ni4zODctLjkxMi41MzMtLjIxNi4xNDYtLjcwMi40Ny0uNTM5LjQ3MS4xNjQuMDAzLjk5Ni0uMzcyIDIuMDItMS4wMS4xOC0uMTU5LjIwNC0uMjI2LjE0LS4yMzl6bTEzLjQ4NiAwYy4xNDUuMDc0LjMyLjE1Mi40Mi4yMTcuMTIyLjA4LjA2Ny40NzYuMDY1LjU1Ny0uMDkxLjMzNS0uMjI4LjM4Ny0uNTIuMjc1LS4wMTUuNjEzLS4yNC43NTYtLjcyNCAxLjYxMy0uNDU3IDEuMDY2LS41MDEgMS4yNDQtMS4wNTUgMi44MjEtLjAwNi41NTktMS41MiAzLjEyOC0xLjg0NCAzLjEyOC0uMTQ5IDAtLjE5NS4xMjItLjEwMi4yNzIuMjg4LjQ2NS45NDQtLjI5NSAxLjctMS45NjguMzg4LS44Ni43OTEtMS45NzQuODk1LTIuNDc1LjEwNS0uNTAyLjUzLTEuNDMyLjk0Ni0yLjA2Ni44NjQtMS4zMi45OTUtMi4zNjYuMjk3LTIuMzY5LS4wMTggMC0uMDU3LS4wMDUtLjA3OC0uMDA1em0tMTAuMDQuMDk5Yy4yMDIuMDUzLjQwMi4wOS42MDUuMTUzLjI2Ni4xMTIuNTAzLjE5OS43MzcuMjgxYTE5Ljk1IDE5Ljk1IDAgMDAtLjA3OC41NTRjLS4yMzMtLjA3Ni0uNDY4LS4xNDQtLjY5OS0uMjI0YTEwLjQ4IDEwLjQ4IDAgMDAtLjYwMy0uMTc4Yy4wMTQtLjE5NS4wMjEtLjM5My4wMzgtLjU4NnptLTkuNjU3LjA1OWMuMTA4LjQ3OC4zMTcgMS4yNzcuNzgxIDIuNzQ1IDEuMTI1IDMuMzI0IDIuMzQ0IDUuNzAzIDMuMDM3IDUuOTIuMjU0LjA4LjIwMi0uMDg4LS4xNTctLjUwMy0uOTk1LTEuMTUtMS41OTMtMi4xNC0yLjYwNS01LjA1bC0uOTA1LTMuMDg1YS4yOC4yOCAwIDAxLS4xNTEtLjAyN3ptMTIuNzcyLjE2NWMuMTc2LjAzMy4zNS4wNjMuNTIyLjA5MS0uMDI1LjE2OC0uMDUuMzM4LS4wNzMuNTI0YTE1LjM2IDE1LjM2IDAgMDEtLjU1OS0uMTJjLjAzNS0uMTY2LjA3Mi0uMzMuMTEtLjQ5NXptLTUuMTAxLjEyNGEyLjIgMi4yIDAgMDAuMjc3LjA2Yy4wMDQuMTkuMDE2LjM4Mi4wMjUuNTc0LS4xMjItLjAxOC0uMTktLjAyLS4yNjYtLjAyNHptLjQ3LjA5MWMuNDI0LjA3MS44NDkuMTYzIDEuMjc1LjI3LS4wMS4xNzUtLjAxMi4zNTMtLjAxOS41MjktLjQxNC0uMS0uODI4LS4xNjctMS4yNDItLjIzLS4wMDktLjE5LS4wMDktLjM4LS4wMTMtLjU2OXptMy4wMjIuMDU3Yy40ODMuMTY0LjkuMjY3IDEuMjk0LjM0OS0uMDQuMTg3LS4wOC4zNzUtLjExNS41NjVhMjEuNDY3IDIxLjQ2NyAwIDAxLTEuMjU0LS4zN2MuMDIzLS4xODIuMDUtLjM2My4wNzUtLjU0NHptLTQuNzQxLjIxOGEuNTI5LjUyOSAwIDAwLS4xNjQuMDJsLTEuNzEyLjgzYy0uMTkuMTA4LS4yMi4wODktLjM0NS4xOTktLjEyNi4xMTEtLjAxLjEzMi4xMDMuMTMyLjIxNSAwIDEuODctLjgzIDIuMTQ4LTEuMDQuMDgxLS4xMDMuMDM3LS4xMzQtLjAzLS4xNHptMy4xODUuMDQ1Yy4xODMuMDQ5LjM2NC4wODIuNTQ4LjEzOC4yNjcuMTE0LjUwNi4yLjc0MS4yODQtLjAxOC4xNTktLjAyNy4zMjItLjA0MS40ODMtLjIyMi0uMDctLjQ0Ni0uMTMzLS42NjctLjIwNy0uMjA1LS4wNjYtLjQxLS4xMjItLjYxNC0uMTc1LjAwOC0uMTc2LjAyMi0uMzQ4LjAzMy0uNTIzem0zLjAyNS4xMjRjLjE5NC4wMzguMzgyLjA3LjU3LjEtLjAyNC4xODUtLjA1MS40MDgtLjA3NS42MDUtLjItLjA0LS40LS4wODEtLjU5OS0uMTI5LjAzNi0uMTkxLjA2NS0uMzg3LjEwNC0uNTc2em0tNC45MTUuMjE1Yy4wNzQuMDA2LjE0OC4wMS4yNjQuMDI3LjAwOS4xNDcuMDEyLjI5Mi4wMjQuNDQuMDA2LjA2Ni4wMDkuMTMxLjAxNC4xOTYtLjExOS0uMDE3LS4xODctLjAzLS4yNjUtLjA0NC0uMDA5LS4xMDEtLjAyNi0uNDMxLS4wMzctLjYyem0uNDQyLjA1NGMuNDEuMDY2LjgyLjE1MSAxLjIzMi4yNTEtLjAwMy4xMjItLjAxMi4yNC0uMDEzLjM2MyAwIC4wOC0uMDAzLjE2LS4wMDMuMjM5YTE0LjEwNyAxNC4xMDcgMCAwMC0xLjE3Ni0uMjE3Yy0uMDEzLS4wOTQtLjAxOC0uMTctLjAzMi0uMjY4LS4wMDctLjEyNS0uMDAxLS4yNDQtLjAwOC0uMzY4em0yLjkwNi4wOTRoLjAwMWMuNDYzLjE1Ni44NjQuMjU0IDEuMjQ0LjMzNC0uMDI3LjE1NC0uMDQ2LjMxMy0uMDcuNDctLjQxLS4xMDEtLjgxOS0uMjEtMS4yMjEtLjMzNS4wMTUtLjE1Ni4wMjgtLjMxNC4wNDYtLjQ3em0tOS4wNzUuMDNsLTEuMTYzLjQ2LjA5OS4xMTEgMS4xNTgtLjQ0N3ptNC40NDcuMDhsLTEuMDY0LjUzYy0uNTAyLjI0LS45NjIuNDc5LTEuMDIxLjUzM3YuMDAxYy0uMDU1LjEzOC0uMDEzLjE3LjA1LjE4LjIzLS4xODQgMS43NjYtLjgxMiAyLjEwNC0xLjA5NXptMy4xMjUuMDljLjE4Ny4wNDguMzcyLjA4Mi41NTkuMTM3LjI1OC4xMDYuNDkuMTg3LjcxNy4yNjQtLjAwOC4xMTQtLjAyMy4yMjUtLjAzLjM0LS4wMDQuMDc2LS4wMTIuMTUzLS4wMTYuMjMtLjIxNS0uMDY4LS40MzMtLjEzLS42NDctLjJhOS45MTggOS45MTggMCAwMC0uNTk0LS4xN2MtLjAwNS0uMTA5LS4wMDEtLjE5OC0uMDA4LS4zMTMuMDAzLS4wOTguMDE1LS4xOS4wMi0uMjg4em0tNy40MDguMTZsLTEuMTYzLjQ2LjA5OS4xMSAxLjE1Ni0uNDQ4em0xMC4zNDUuMDE3Yy4yMDguMDQuNDEuMDc1LjYxLjEwOGwtLjA2Mi41MDVjLS4yMTMtLjA0My0uNDIyLS4wOTgtLjYzMi0uMTQ3LjAyNS0uMTU3LjA1Ny0uMzEuMDg0LS40NjZ6bTEuNzU1LjIzbC0uMDY1LjE0IDMuMjU3LjQ3OC4wNzItLjEyOXptLTYuNDkzIDBjLjA3Ny4wMTYuMTUyLjAyOC4yNjcuMDQ1LjAxNy4xOTIuMDI3LjM4My4wNDIuNTc1YTcuNSA3LjUgMCAwMC0uMjEzLS4wM2MtLjAxOC0uMS0uMDgzLS40MTQtLjA4OC0uNDU3LS4wMDItLjAxMi0uMDA3LS4xMTYtLjAwOC0uMTMyem0zLjIzOC4wNTNjLjQ2LjE1Ljg0LjIzMiAxLjIxNy4zMDctLjAxOC4xMjgtLjA0Mi4yNTMtLjA1Ny4zODItLjAwNy4wNjQtLjAxOC4xMjctLjAyNi4xOTFhMjAuMDUzIDIwLjA1MyAwIDAxLTEuMTY3LS4zMTVjMC0uMTA3LjAwOC0uMTk1LjAwOC0uMzA3LjAwNi0uMDg3LjAxNy0uMTcxLjAyNS0uMjU4em0tMi43ODUuMDIxYy4zODUuMDYyLjc3LjE0MiAxLjE1Ny4yMzUgMCAuMTg5LS4wMDcuMzc3LS4wMS41NjVhMTIuOTIgMTIuOTIgMCAwMC0xLjA4LS4yMmMtLjAxOS0uMTg1LS4wNDEtLjM4LS4wNjctLjU4em0tNS45MjYuMDI1bC0xLjE2My40Ni4wOTguMTEgMS4xNTctLjQ0N3ptNy4yNjguMjU1Yy4xNzUuMDQ1LjM0OS4wNzcuNTI1LjEyOS4yNTEuMTAzLjQ3OC4xODMuNjk4LjI1OC0uMDEuMTkzLS4wMjcuMzg2LS4wNC41NzktLjIwMy0uMDcyLS40MS0uMTM3LS42MTItLjIxMmE4LjgyMiA4LjgyMiAwIDAwLS41NTUtLjE3NmMtLjAwMi0uMTg0LS4wMS0uMzgtLjAxNi0uNTc4em0yLjgzMi4wNGMuMjEyLjA0LjQzNS4wODguNjQxLjEyMi0uMDI2LjIxLS4wNDYuMzkzLS4wNzUuNjA4LS4yMi0uMDQtLjQzMS0uMTAxLS42NDctLjE1MS4wMDYtLjA4Ny4wMTUtLjE1NS4wMi0uMjQ1LjAxNy0uMTE0LjA0NC0uMjIxLjA2LS4zMzR6bTEuNzcyLjAyMmwtLjA2NS4xNCAzLjI1Ny40NzcuMDcyLS4xMjd6bS02LjI3NS4zMjFjLjA2LjAxLjExNy4wMTYuMTk2LjAyOS4wMTguMjM5LjAzOC40NzcuMDUyLjcxNi0uMS0uMDE1LS4xMDMtLjAxOC0uMTEtLjAyMnptLjQxNC4wNjdjLjM1Mi4wNjQuNzA1LjE0NiAxLjA2LjI0LS4wMDQuMjIyLS4wMDUuNDQ0LS4wMTIuNjY2YTEyLjMwMiAxMi4zMDIgMCAwMC0uOTk2LS4xOTMgMTkuNzE2IDE5LjcxNiAwIDAwLS4wNTItLjcxM3ptMi42NjIgMGMuNDM0LjE0Ljc5Ni4yMTQgMS4xNTMuMjg1LS4wMjQuMjEtLjA1NS40MTgtLjA4LjYyN2ExNy42OTggMTcuNjk4IDAgMDEtMS4wODgtLjMyNmMuMDA5LS4xODUuMDExLS4zODYuMDE1LS41ODZ6bTMuMDc3LjA1NGwtLjA2NC4xNCAzLjI1Ny40NzguMDcyLS4xMjh6bS0xMS40OTQuMTVsLTEuMTYzLjQ2LjA5OS4xMSAxLjE1OC0uNDQ4em03LjAzMy4wOTVjLjE2My4wNDcuMzI2LjA4LjQ5LjEzNC4yMzguMTA4LjQ1MS4xOS42Ni4yNy0uMDE0LjIwMy0uMDI2LjQwNi0uMDQzLjYwOC0uMjA0LS4wNjYtLjQxLS4xMjgtLjYxMi0uMTk5YTguODU1IDguODU1IDAgMDAtLjUxLS4xNWMuMDA3LS4yMTUuMDE1LS40MjguMDE1LS42NjN6bTIuNzE2LjAyMWMuMi4wMzcuNDEzLjA4Ni42MDYuMTE2bC4wNS0uMDg0Yy0uMDMyLjI1My0uMDgyLjUwOS0uMTIuNzY1LS4yLS4wNDEtLjM5NC0uMTAyLS41OTItLjE1My4wMjEtLjIwNC4wNC0uNDIxLjA1Ni0uNjQ0em0xLjY0LjE5OWwtLjA2My4xNCAzLjI1Ny40NzcuMDcxLS4xMjl6bS0xMS4yMjUuMDE0bC0xLjE2My40Ni4wOTcuMTEgMS4xNTktLjQ0N3ptOC4yMzIuMjUyYy40MDEuMTQyLjc0LjIyMiAxLjA3MS4yOTYtLjAyNS4xOTUtLjA0OS4zOTEtLjA3Ni41ODctLjM1LS4wODgtLjY5OC0uMTgtMS4wNC0uMjg4LjAxNi0uMTk1LjAzNC0uMzgzLjA0NS0uNTk1em0tMi44OS4wMDZjLjAxNi4wMDUuMDIyLjAwOC4wOTMuMDIuMDEyLjIwNi4wMjIuNDE0LjAzLjYyMS0uMDU4LS4yOTMtLjA3MS0uMzgzLS4xMjItLjY0em0uMzEuMDU0Yy4zMjcuMDU3LjY1Ni4xMy45ODYuMjEyLS4wMDcuMjI5LS4wMTUuNDU4LS4wMjcuNjg3YTEyLjM2IDEyLjM2IDAgMDAtLjkzLS4xODEgMTkuOTU0IDE5Ljk1NCAwIDAwLS4wMy0uNzE4em0tNS41MTguMDM0bC0xLjE2My40Ni4wOTcuMTEgMS4xNTgtLjQ0NnptMTAuOTc5LjE0M2wtLjA2Ni4xNCAzLjI1Ny40NzkuMDczLS4xM3ptLTQuMjY1LjA4OWMuMTUzLjA0LjMwNS4wNjkuNDU5LjExNi4yMzcuMTAxLjQ1LjE4MS42NTguMjU1LS4wMTguMjEtLjAzNy40MTktLjA1OS42MjgtLjE3NS0uMDU2LS4zNTItLjEwOC0uNTI2LS4xNjctLjE5LS4wNjItLjM3OC0uMDk4LS41NjctLjE0Ni4wMTctLjIxNi4wMjYtLjQ1MS4wMzUtLjY4NnptMi42NjkuMDE0Yy4xODMuMDM3LjM4LjA4OC41NTcuMTJsLjA0LS4wNzdjLS4xMTIuNzUyLS4yMzMgMS41MDQtLjQwNiAyLjIwNC0uMTgyLS4wMzktLjM2LS4wODYtLjU0LS4xMy4wNDQtLjIuMDg1LS40MDcuMTI2LS42MjEuMTQ1LjAyNi4zMDMuMDY1LjQ0NC4wODdsLjA1Ni0uMDk4Yy0uMTYxLS4wMy0uMzE2LS4wNzctLjQ3NS0uMTEzLjAzNy0uMTk4LjA2Ny0uNDIyLjA5OS0uNjM4LjE2MS4wMy4zMzYuMDczLjQ5My4wOThsLjA2LS4xMWMtLjE4LS4wMzMtLjM1NC0uMDg3LS41MzItLjEyOC4wMjctLjE5My4wNTUtLjM4NC4wNzgtLjU5NHptMS41MzguMzYzbC0uMDY1LjE0IDMuMjIuNDI4LjA3NC0uMTI5em0tMi44ODMuMDY1Yy4zODcuMTI3LjcxNS4xOTkgMS4wMzYuMjY0LS4wMy4yMTEtLjA2LjQyMi0uMDk0LjYzMmExNi42MDYgMTYuNjA2IDAgMDEtMS4wMDQtLjI3NWMuMDI1LS4xOTUuMDQyLS40MS4wNjItLjYyem0tMi40OTIuMTNjLjMwNy4wNDkuNjE1LjExLjkyMy4xODQtLjAxMS4yMTMtLjAyMi40MjctLjAzOC42NC0uMjk2LS4wNjItLjU5LS4xMTEtLjg4Ni0uMTUzLjAwNS0uMjE0LjAwMy0uNDQ0LjAwMS0uNjcyem0tNS4yNC4wMTVsLTEuMTY0LjQ2LjA5OS4xMTEgMS4xNTgtLjQ0OHptNi4zNjQuMjE4Yy4xNzQuMDQ0LjM0Ny4wNzguNTIyLjEzLjIwMy4wODYuMzg1LjE1LjU2NS4yMTQtLjAyMi4xOTgtLjA0LjM5Ny0uMDY2LjU5NS0uMTEtLjAzNy0uMjItLjA2OC0uMzMtLjEwNS0uMjQ5LS4wODQtLjQ5OS0uMTM2LS43NDgtLjE5NC4wMjItLjIwNC4wMzktLjQyMy4wNTctLjY0em00LjE3NS4wMDdsLS4wNjUuMTQgMy4xNzUuNDM1LjA3NC0uMTI4em0tMTAuMzc1LjEwNWwtMS4xNjQuNDYuMDk4LjExIDEuMTU4LS40NDh6bTcuNDg0LjMwMmE4LjM4IDguMzggMCAwMDEuMDA0LjI1M2MtLjAzNC4yMDUtLjA2Ny40MS0uMTA1LjYxNi0uMzMtLjA4Ny0uNjU5LS4xODEtLjk4My0uMjg3LjAzLS4xODYuMDU3LS4zODQuMDg0LS41ODJ6bS03LjM1LjA0M2wtMS4xNjQuNDYuMDk3LjExMSAxLjE1OS0uNDQ3em0xMC4xOC4wMTdsLS4wNjUuMTQgMy4xMzIuNDMzLjA3My0uMTI5em0tNS4yNDIuMDc3Yy4yOTIuMDQzLjU4NC4wOTIuODc4LjE1N2EyNy4xMiAyNy4xMiAwIDAxLS4wNDkuNTMgMTIuOTUgMTIuOTUgMCAwMC0uNzQtLjExOGMtLjAzMi0uMTczLS4wNTUtLjMwMS0uMDkyLS40OTMgMC0uMDI2LjAwMi0uMDUuMDAzLS4wNzZ6bTEuMDU4LjIwM2MuMjM3LjA1Ni40NzMuMTA3LjcxMS4xOC4xMy4wNTUuMjQyLjA5Mi4zNjEuMTM3LS4wMjEuMTYzLS4wNDguMzI2LS4wNzMuNDg4LS4wODctLjAyOC0uMTc1LS4wNTItLjI2Mi0uMDgtLjI2OC0uMDg2LS41MzYtLjE0Mi0uODA0LS4yLjAyMy0uMTczLjA0Ni0uMzQ2LjA2Ny0uNTI1em00LjEyNi4yMDdsLS4wNjYuMTQgMy4wNTUuNDQ3LjA3NC0uMTI5em0tMi44NzcuMTc3Yy4zNjYuMTMuNjguMjE0Ljk3Ny4yOC0uMDMuMTU1LS4wNjUuMzEtLjA5Ny40NjRhMTguMTU4IDE4LjE1OCAwIDAxLS45NjYtLjI3M2MuMDI5LS4xNTcuMDU5LS4zMS4wODYtLjQ3MXptMi43NzIuMjczbC0uMDY2LjE0IDMuMDI4LjQ4NC4wNzMtLjEyOXptLTEuNjE0LjA0M2MuMTgyLjAzNy4zNjcuMDc3LjU0My4xMDktLjA0Mi4xNjYtLjA5NS4zMTctLjE0Mi40NzVhMjEuODM1IDIxLjgzNSAwIDAxLS41MTItLjExN2MuMDM3LS4xNTUuMDc1LS4zMDcuMTExLS40Njd6Ii8+PHBhdGggZmlsbD0iI2Y4ZjFkNyIgZD0iTTEzLjQ1NzYgNS43NDM2Yy0uMDEuMTc1LS4wMjUuMzQ3LS4wMzMuNTI0LjIwNC4wNTMuNDEuMTA5LjYxNC4xNzUuMjIuMDc0LjQ0NS4xMzcuNjY3LjIwNy4wMTQtLjE2MS4wMjMtLjMyNS4wNDEtLjQ4NC0uMjM1LS4wODMtLjQ3NC0uMTctLjc0MS0uMjgzLS4xODQtLjA1Ni0uMzY1LS4wOS0uNTQ4LS4xMzl6Ii8+PHBhdGggZmlsbD0iI2ViZWNhMSIgZD0iTTExLjk4MzYgMy45NzY2Yy0uMDA1LjE3Ny0uMDE2LjM1MS0uMDE4LjUzLjQ2LjA3LjkxNy4xNDQgMS4zNzYuMjU3LjAxOC0uMTY2LjA0NC0uMzI4LjA2NC0uNDkzLS40NzYtLjEyLS45NS0uMjE5LTEuNDIyLS4yOTR6Ii8+PHBhdGggZmlsbD0iIzIxOGYyMSIgZD0iTTkuOTc1NiA5LjEyMzZjLS40My0xLjU2LS40MzctMS42OTUtLjA4LTEuODEzLjE2My0uMDU1LjM2Mi4zMzYuNDU1Ljg5LjA5MS41NDQuMjM4IDEuMjUyLjMyNiAxLjU3NS4wOS4zMjQuMDQ1LjU4Ni0uMDk5LjU4Ni0uMTQzIDAtLjQxNC0uNTU3LS42MDItMS4yMzh6Ii8+PHBhdGggZmlsbD0iI2ZlMmUyZSIgZD0iTTkuNDgzNiAxMC43MjQ2Yy0uMzI3LS43MTItLjI4MS0xLjMzLS42Mi0xLjk4Ny4xNDItLjA4OC4yNzEtLjIxNi4zNzgtLjE1LjM4LjIzMi43MDggMS42MS44MSAyLjAzMy0uMzU4LjE4LS41MDYuMjY4LS41NjguMTA0eiIvPjxwYXRoIGZpbGw9IiMyZTUxYTgiIGQ9Ik0xNy4zOTU2IDMuODg5NmMuMzU3LjA1MSAyLjE2LjA3OCAyLjczOS4wMjQuOTE4LS4wNjYgMy4wOS43MTYgMy44MyAxLjIwMi4xMjIuMDguMDY3LjQ3Ny4wNjYuNTU4LS4xNTcuNTc0LS4zOC4zOC0xLjQ1LS4yLS41MzgtLjI3LTEuNzg3LS41NzEtMi43NzYtLjY2OS0uOTg4LS4wOTctMi4wNTMtLjA2Ny0yLjI2Mi0uMTE3LS4yMS0uMDUtLjQxOC4wMDItLjE0Ni0uNzk3em0tMTMuNjUuOWMwLS4zMDgtLjE5LS41MS0uMDE4LS42ODIuNjI3LS4xOTYuOTA4LjA4IDEuNzc3LS41MjYgMS40NzctMSAzLjMxLTEuNzk3IDQuNjEyLTEuODAyIDEuMDc3IDAgMS4zLjA0MyAxLjMuMzE3IDAgLjIwMy0uMDg5LjQ3LS4wODkuNDctLjI0NC4zMjItLjgyMi4xMDgtMS4zNDkuMjExLTEuMDE3IDAtMS45MDguMzM3LTIuODExIDEuMDU4LS40OC4zODItLjgyLjQ4Mi0xLjU2OC43MzUtMS4zNTguNDYtMS44NTQuNzc4LTEuODU0LjIxOXptMTEuMTQ5LTMuNjIxYy0uODc3LS42MDctLjYxNC0uMDE4LTIuNDE2LS4wNTItMi43Mi0uMDUzLTIuNzM3LS4wNTYtMi43MzctLjQ5NSAwLS4yOTMuMzcxLS4zNjQgMS42OTUtLjMyNi45MzIuMDI4IDEuMjQyLS4wMDYgMS42OTQtLjE0Ni4xOTEtLjA4Mi41MjktLjEzNS44My0uMTQ4LjY4NC0uMDI3IDEuMzE3LjI5NCAxLjg1MS44Ni4yOTYuMjkzLjczNi45Ni44ODguODguMTA3LjA4Mi44MTgtLjUwMyAxLjIyNy0uNzU0LjY5NC0uNDY2IDEuMzQzLS43MTUgMS40NzctLjYzMi4yMzUuNTM4LjI5LjUwOS4xMi42NTctLjE3MS4xNDktLjU2Ny40NzUtMS4yODYuODk3LTEuMjkzLjYxNy0xLjY1MiAxLjIxNy0zLjM0My0uNzQxeiIvPjwvc3ZnPg=="
                                       alt="${this.extensionName}" 
                                       title="${this.extensionName}"
                                       width="32" 
                                       height="32">
                </div>
                <div class="description">${this.description}</div>
            </div>
            ${this._renderCardLinks()}
        </div>
        `;
    }

    _renderCardLinks(){
        return html`${pages.map(page => html`
                            <qwc-extension-link slot="link"
                                namespace="${this.namespace}"
                                extensionName="${this.extensionName}"
                                iconName="${page.icon}"
                                displayName="${page.title}"
                                staticLabel="${page.staticLabel}"
                                dynamicLabel="${page.dynamicLabel}"
                                streamingLabel="${page.streamingLabel}"
                                path="${page.id}"
                                ?embed=${page.embed}
                                externalUrl="${page.metadata.externalUrl}"
                                webcomponent="${page.componentLink}" >
                            </qwc-extension-link>
                        `)}`;
    }

}
customElements.define('qwc-poi-card', QwcPoiCard);