package io.quarkiverse.quinoa.deployment;

import static java.util.stream.Collectors.toList;

import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;

import org.eclipse.microprofile.config.Config;
import org.eclipse.microprofile.config.ConfigProvider;

import io.quarkiverse.quinoa.QuinoaHandlerConfig;
import io.quarkiverse.quinoa.deployment.packagemanager.PackageManagerCommandConfig;
import io.quarkiverse.quinoa.deployment.packagemanager.PackageManagerInstallConfig;
import io.quarkus.runtime.annotations.ConfigItem;
import io.quarkus.runtime.annotations.ConfigPhase;
import io.quarkus.runtime.annotations.ConfigRoot;
import io.quarkus.vertx.http.runtime.HttpBuildTimeConfig;

@ConfigRoot(phase = ConfigPhase.BUILD_TIME)
public class QuinoaConfig {

    private static final String DEFAULT_WEB_UI_DIR = "src/main/webui";
    private static final String DEFAULT_INDEX_PAGE = "index.html";

    /**
     * Indicate if the extension should be enabled.
     * Default is true if the Web UI directory exists and dev and prod mode.
     * Default is false in test mode (to avoid building the Web UI during backend tests).
     */
    @ConfigItem(name = ConfigItem.PARENT, defaultValueDocumentation = "disabled in test mode")
    Optional<Boolean> enable;

    /**
     * Indicate if Quinoa should just do the build part.
     * If true, Quinoa will NOT serve the Web UI built resources.
     * This is handy when the output of the build is used
     * to be served via something else (nginx, cdn, ...)
     * Quinoa put the built files in 'target/quinoa-build' (or 'build/quinoa-build with Gradle).
     *
     * Default is false.
     */
    @ConfigItem
    public boolean justBuild;

    /**
     * Path to the Web UI (NodeJS) root directory.
     * If not set ${project.root}/src/main/webui/ will be used.
     * otherwise the path will be considered relative to the project root.
     */
    @ConfigItem(defaultValue = DEFAULT_WEB_UI_DIR)
    public String uiDir;

    /**
     * This the Web UI internal build system (webpack, ...) output directory.
     * After the build, Quinoa will take the files from this directory,
     * move them to 'target/quinoa-build' (or build/quinoa-build with Gradle) and serve them at runtime.
     * The path is relative to the Web UI path.
     * If not set "build/" will be used
     */
    @ConfigItem(defaultValue = "build/")
    public String buildDir;

    /**
     * Name of the package manager binary.
     * If not set, it will be auto-detected depending on the lockfile falling back to "npm".
     * Only npm, pnpm and yarn are supported for the moment.
     */
    @ConfigItem(defaultValueDocumentation = "auto-detected with lockfile")
    public Optional<String> packageManager;

    /**
     * Configuration for installing the package manager
     */
    @ConfigItem
    public PackageManagerInstallConfig packageManagerInstall;

    /**
     * Configuration for overriding build commands
     */
    @ConfigItem
    public PackageManagerCommandConfig packageManagerCommand;

    /**
     * Name of the index page.
     * If not set, "index.html" will be used.
     */
    @ConfigItem(defaultValue = DEFAULT_INDEX_PAGE)
    public String indexPage;

    /**
     * Indicate if the Web UI should also be tested during the build phase (i.e: npm test).
     * To be used in a {@link io.quarkus.test.junit.QuarkusTestProfile} to have Web UI test running during a
     * {@link io.quarkus.test.junit.QuarkusTest}
     * Default is false.
     */
    @ConfigItem(name = "run-tests")
    boolean runTests;

    /**
     * Install the packages using a frozen lockfile. Don’t generate a lockfile and fail if an update is needed (useful in CI).
     * If not set it is true if environment CI=true, else it is false.
     */
    @ConfigItem(defaultValueDocumentation = "true if environment CI=true")
    public Optional<Boolean> frozenLockfile;

    /**
     * Force install packages before building.
     * If not set, it will install packages only if the node_modules directory is absent or when the package.json is modified in
     * dev-mode.
     */
    @ConfigItem
    public boolean forceInstall;

    /**
     * Enable SPA (Single Page Application) routing, all relevant requests will be re-routed to the "index.html".
     * Currently, for technical reasons, the Quinoa SPA routing configuration won't work with RESTEasy Classic.
     * If not set, it is disabled.
     */
    @ConfigItem
    public boolean enableSPARouting;

    /**
     * List of path prefixes to be ignored by Quinoa.
     * If not set, "quarkus.resteasy-reactive.path", "quarkus.resteasy.path" and "quarkus.http.non-application-root-path" will
     * be ignored.
     */
    @ConfigItem
    public Optional<List<String>> ignoredPathPrefixes;

    /**
     * Configuration for the external dev server (live coding server)
     */
    @ConfigItem
    public DevServerConfig devServer;

    public boolean isDevServerMode() {
        return devServer.enabled && devServer.port.isPresent();
    }

    public List<String> getNormalizedIgnoredPathPrefixes() {
        return ignoredPathPrefixes.orElseGet(() -> {
            Config config = ConfigProvider.getConfig();
            List<String> defaultIgnore = new ArrayList<>();
            readExternalConfigPath(config, "quarkus.resteasy.path").ifPresent(defaultIgnore::add);
            readExternalConfigPath(config, "quarkus.resteasy-reactive.path").ifPresent(defaultIgnore::add);
            readExternalConfigPath(config, "quarkus.http.non-application-root-path").ifPresent(defaultIgnore::add);
            return defaultIgnore;
        }).stream().map(s -> s.startsWith("/") ? s : "/" + s).collect(toList());
    }

    public QuinoaHandlerConfig toHandlerConfig(boolean prodMode, final HttpBuildTimeConfig httpBuildTimeConfig) {
        final Set<String> compressMediaTypes = httpBuildTimeConfig.compressMediaTypes.map(Set::copyOf).orElse(Set.of());
        return new QuinoaHandlerConfig(getNormalizedIgnoredPathPrefixes(), indexPage, prodMode,
                httpBuildTimeConfig.enableCompression, compressMediaTypes);
    }

    private Optional<String> readExternalConfigPath(Config config, String key) {
        return config.getOptionalValue(key, String.class)
                .filter(s -> !Objects.equals(s, "/"))
                .map(s -> s.endsWith("/") ? s : s + "/");
    }

    public boolean isEnabled() {
        return enable.orElse(true);
    }

    @Override
    public boolean equals(Object o) {
        if (this == o)
            return true;
        if (o == null || getClass() != o.getClass())
            return false;
        QuinoaConfig that = (QuinoaConfig) o;
        return runTests == that.runTests && forceInstall == that.forceInstall && enableSPARouting == that.enableSPARouting
                && Objects.equals(enable, that.enable) && Objects.equals(uiDir, that.uiDir)
                && Objects.equals(buildDir, that.buildDir) && Objects.equals(packageManager, that.packageManager)
                && Objects.equals(packageManagerInstall, that.packageManagerInstall)
                && Objects.equals(packageManagerCommand, that.packageManagerCommand)
                && Objects.equals(indexPage, that.indexPage) && Objects.equals(frozenLockfile, that.frozenLockfile)
                && Objects.equals(ignoredPathPrefixes, that.ignoredPathPrefixes) && Objects.equals(devServer, that.devServer);
    }

    @Override
    public int hashCode() {
        return Objects.hash(enable, uiDir, buildDir, packageManager, packageManagerInstall, packageManagerCommand, indexPage,
                runTests, frozenLockfile, forceInstall, enableSPARouting, ignoredPathPrefixes, devServer);
    }
}
