/*
 * Decompiled with CFR 0.152.
 */
package io.quarkus.vault.client;

import io.quarkus.vault.client.VaultClientException;
import io.quarkus.vault.client.api.VaultAuthAccessor;
import io.quarkus.vault.client.api.VaultSecretsAccessor;
import io.quarkus.vault.client.api.VaultSysAccessor;
import io.quarkus.vault.client.auth.VaultAppRoleAuthOptions;
import io.quarkus.vault.client.auth.VaultAppRoleTokenProvider;
import io.quarkus.vault.client.auth.VaultAuthRequest;
import io.quarkus.vault.client.auth.VaultKubernetesAuthOptions;
import io.quarkus.vault.client.auth.VaultKubernetesTokenProvider;
import io.quarkus.vault.client.auth.VaultStaticClientTokenAuthOptions;
import io.quarkus.vault.client.auth.VaultStaticClientTokenProvider;
import io.quarkus.vault.client.auth.VaultToken;
import io.quarkus.vault.client.auth.VaultTokenException;
import io.quarkus.vault.client.auth.VaultTokenProvider;
import io.quarkus.vault.client.auth.VaultUserPassAuthOptions;
import io.quarkus.vault.client.auth.VaultUserPassTokenProvider;
import io.quarkus.vault.client.common.VaultRequest;
import io.quarkus.vault.client.common.VaultRequestExecutor;
import io.quarkus.vault.client.common.VaultResponse;
import io.quarkus.vault.client.common.VaultTracingExecutor;
import io.quarkus.vault.client.logging.LogConfidentialityLevel;
import java.net.URL;
import java.nio.file.Path;
import java.time.Duration;
import java.time.InstantSource;
import java.util.Objects;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Supplier;
import java.util.logging.Level;
import java.util.logging.Logger;

public class VaultClient
implements VaultRequestExecutor {
    private static final Logger log = Logger.getLogger(VaultClient.class.getName());
    private final AtomicReference<VaultSecretsAccessor> secrets = new AtomicReference();
    private final AtomicReference<VaultAuthAccessor> auth = new AtomicReference();
    private final URL baseUrl;
    private final String apiVersion;
    private final VaultRequestExecutor executor;
    private final Duration requestTimeout;
    private final LogConfidentialityLevel logConfidentialityLevel;
    private final VaultTokenProvider tokenProvider;
    private final String namespace;
    private final InstantSource instantSource;
    private final int maxAttempts;

    public static Builder builder() {
        return new Builder();
    }

    private VaultClient(Builder builder) {
        this.baseUrl = Objects.requireNonNull(builder.baseUrl, "baseUrl is required");
        this.executor = Objects.requireNonNull(builder.executor, "executor is required");
        this.apiVersion = builder.apiVersion;
        this.logConfidentialityLevel = builder.logConfidentialityLevel;
        this.requestTimeout = builder.requestTimeout;
        this.tokenProvider = builder.tokenProvider;
        this.namespace = builder.namespace;
        this.instantSource = builder.instantSource;
        this.maxAttempts = builder.maxRetries + 1;
    }

    public VaultSecretsAccessor secrets() {
        if (this.secrets.get() == null) {
            this.secrets.set(new VaultSecretsAccessor(this));
        }
        return this.secrets.get();
    }

    public VaultAuthAccessor auth() {
        if (this.auth.get() == null) {
            this.auth.set(new VaultAuthAccessor(this));
        }
        return this.auth.get();
    }

    public VaultSysAccessor sys() {
        return new VaultSysAccessor(this);
    }

    public URL getBaseUrl() {
        return this.baseUrl;
    }

    public String getApiVersion() {
        return this.apiVersion;
    }

    public VaultRequestExecutor getExecutor() {
        return this.executor;
    }

    public Duration getRequestTimeout() {
        return this.requestTimeout;
    }

    public LogConfidentialityLevel getLogConfidentialityLevel() {
        return this.logConfidentialityLevel;
    }

    public VaultTokenProvider getTokenProvider() {
        return this.tokenProvider;
    }

    public String getNamespace() {
        return this.namespace;
    }

    @Override
    public <T> CompletionStage<VaultResponse<T>> execute(VaultRequest<T> request) {
        VaultRequest.Builder requestBuilder = request.builder();
        requestBuilder.baseUrl(this.baseUrl);
        if (this.requestTimeout != null) {
            requestBuilder.timeout(this.requestTimeout);
        }
        if (this.apiVersion != null) {
            requestBuilder.apiVersion(this.apiVersion);
        }
        if (this.logConfidentialityLevel != null) {
            requestBuilder.logConfidentialityLevel(this.logConfidentialityLevel);
        }
        if (!request.hasNamespace() && this.namespace != null) {
            requestBuilder.namespace(this.namespace);
        }
        if (request.hasToken() || this.tokenProvider == null) {
            log.finer(() -> "Executing authorized request " + request.getOperation());
            return this.executor.execute(requestBuilder.rebuild());
        }
        log.finer(() -> "Executing unauthorized request " + request.getOperation());
        Supplier<CompletionStage> providedToken = () -> {
            log.finer(() -> "Requesting token for request " + request.getOperation());
            return (CompletionStage)this.tokenProvider.apply(VaultAuthRequest.of(this, request, this.instantSource));
        };
        AtomicReference<Object> appliedToken = new AtomicReference<Object>(null);
        Supplier<CompletionStage<VaultResponse<T>>> responseSupplier = () -> ((CompletionStage)providedToken.get()).thenCompose(token -> {
            appliedToken.set(token);
            if (token == null) {
                requestBuilder.noToken();
            } else {
                requestBuilder.token(token.getClientTokenForUsage());
            }
            return this.executor.execute(requestBuilder.rebuild());
        });
        CompletionStage<VaultResponse<VaultResponse>> execution = this.attempt(0, responseSupplier, request, appliedToken);
        if (log.isLoggable(Level.FINER)) {
            return execution.whenComplete((response, error) -> {
                if (error != null) {
                    log.finer(() -> "Failed request " + request.getOperation() + " with error " + String.valueOf(error));
                } else {
                    log.finer(() -> "Successful request " + request.getOperation());
                }
            });
        }
        return execution;
    }

    public <T> CompletionStage<VaultResponse<T>> attempt(Integer attempt, Supplier<CompletionStage<VaultResponse<T>>> responseSupplier, VaultRequest<T> request, AtomicReference<VaultToken> appliedToken) {
        log.finer(() -> "Attempt %d of %d for request %s".formatted(attempt + 1, this.maxAttempts, request.getOperation()));
        return responseSupplier.get().exceptionallyCompose(failure -> {
            if (attempt < this.maxAttempts && this.shouldRetry((Throwable)failure, (VaultToken)appliedToken.get())) {
                log.finer(() -> "Retrying request %s due to %s".formatted(request.getOperation(), failure));
                return this.attempt(attempt + 1, responseSupplier, request, appliedToken);
            }
            return CompletableFuture.failedFuture(failure);
        });
    }

    private boolean shouldRetry(Throwable failure, VaultToken appliedToken) {
        if ((failure = VaultClient.unwrapException(failure)) instanceof VaultTokenException) {
            return true;
        }
        if (failure instanceof VaultClientException) {
            boolean cachedPermissionFailure;
            VaultClientException ve = (VaultClientException)failure;
            boolean bl = cachedPermissionFailure = ve.isPermissionDenied() && appliedToken != null && appliedToken.isFromCache();
            if (cachedPermissionFailure) {
                this.tokenProvider.invalidateCache();
            }
            return cachedPermissionFailure;
        }
        return false;
    }

    public Builder configure() {
        Builder builder = new Builder();
        builder.baseUrl = this.baseUrl;
        builder.apiVersion = this.apiVersion;
        builder.executor = this.executor;
        builder.tokenProvider = this.tokenProvider;
        builder.namespace = this.namespace;
        builder.requestTimeout = this.requestTimeout;
        builder.logConfidentialityLevel = this.logConfidentialityLevel;
        return builder;
    }

    private static Throwable unwrapException(Throwable e) {
        if (e instanceof CompletionException || e instanceof ExecutionException) {
            return e.getCause();
        }
        return e;
    }

    public static class Builder {
        private URL baseUrl;
        private String apiVersion;
        private VaultRequestExecutor executor;
        private VaultTokenProvider tokenProvider;
        private String namespace;
        private Duration requestTimeout;
        private LogConfidentialityLevel logConfidentialityLevel;
        private InstantSource instantSource = InstantSource.system();
        private int maxRetries = 2;

        public Builder baseUrl(URL baseUrl) {
            this.baseUrl = Objects.requireNonNull(baseUrl, "baseUrl is required");
            return this;
        }

        public Builder baseUrl(String baseUrl) {
            try {
                return this.baseUrl(new URL(baseUrl));
            }
            catch (Exception e) {
                throw new IllegalArgumentException("Invalid URL: " + baseUrl, e);
            }
        }

        public Builder apiVersion(String apiVersion) {
            this.apiVersion = Objects.requireNonNull(apiVersion, "apiVersion is required");
            assert (apiVersion.startsWith("v"));
            return this;
        }

        public Builder executor(VaultRequestExecutor executor) {
            this.executor = executor;
            return this;
        }

        public Builder clientToken(String token) {
            return this.clientToken(VaultStaticClientTokenAuthOptions.builder().token(token).build());
        }

        public Builder clientToken(VaultStaticClientTokenAuthOptions options) {
            return this.tokenProvider(new VaultStaticClientTokenProvider(options));
        }

        public Builder userPass(String username, String password) {
            return this.userPass(VaultUserPassAuthOptions.builder().username(username).password(password).build());
        }

        public Builder userPass(VaultUserPassAuthOptions options) {
            return this.tokenProvider(new VaultUserPassTokenProvider(options).caching(options.cachingRenewGracePeriod));
        }

        public Builder appRole(String roleId, String secretId) {
            return this.appRole(VaultAppRoleAuthOptions.builder().roleId(roleId).secretId(secretId).build());
        }

        public Builder appRole(VaultAppRoleAuthOptions options) {
            return this.tokenProvider(new VaultAppRoleTokenProvider(options).caching(options.cachingRenewGracePeriod));
        }

        public Builder kubernetes(String role, Path jwtTokenPath) {
            return this.kubernetes(VaultKubernetesAuthOptions.builder().role(role).jwtTokenPath(jwtTokenPath).build());
        }

        public Builder kubernetes(VaultKubernetesAuthOptions options) {
            return this.tokenProvider(new VaultKubernetesTokenProvider(options).caching(options.cachingRenewGracePeriod));
        }

        public Builder tokenProvider(VaultTokenProvider tokenProvider) {
            this.tokenProvider = tokenProvider;
            return this;
        }

        public Builder namespace(String namespace) {
            this.namespace = namespace;
            return this;
        }

        public Builder requestTimeout(Duration requestTimeout) {
            this.requestTimeout = requestTimeout;
            return this;
        }

        public Builder maxRetries(int maxRetries) {
            if (maxRetries < 0) {
                throw new IllegalArgumentException("maxRetries must be greater than or equal to 0");
            }
            this.maxRetries = maxRetries;
            return this;
        }

        public Builder logConfidentialityLevel(LogConfidentialityLevel logConfidentialityLevel) {
            this.logConfidentialityLevel = logConfidentialityLevel != null ? logConfidentialityLevel : LogConfidentialityLevel.HIGH;
            return this;
        }

        public Builder traceRequests() {
            Objects.requireNonNull(this.executor, "executor must be configured before tracing");
            if (!(this.executor instanceof VaultTracingExecutor)) {
                this.executor = new VaultTracingExecutor(this.executor);
            }
            return this;
        }

        public Builder instantSource(InstantSource instantSource) {
            this.instantSource = instantSource;
            return this;
        }

        public VaultClient build() {
            return new VaultClient(this);
        }
    }
}

