package io.quarkus.gizmo2;

import static io.smallrye.common.constraint.Assert.checkNotNullParam;
import static java.lang.constant.ConstantDescs.CD_Object;

import java.io.ByteArrayInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.UncheckedIOException;
import java.lang.constant.ClassDesc;
import java.util.Map;
import java.util.function.Supplier;

import io.github.dmlloyd.classfile.ClassHierarchyResolver;
import io.quarkus.gizmo2.impl.ClassHierarchyLocatorCachedImpl;
import io.quarkus.gizmo2.impl.ClassHierarchyLocatorResultImpl;
import io.quarkus.gizmo2.impl.Util;
import io.smallrye.common.resource.Resource;
import io.smallrye.common.resource.ResourceLoader;

/**
 * Obtains the class hierarchy information for classes that are known to be accessible
 * to classes generated by {@link Gizmo}. This is used when the bytecode generator needs
 * to understand the subclass/superclass and interface hierarchy.
 */
@FunctionalInterface
public interface ClassHierarchyLocator {
    /**
     * A result of locating class hierarchy information for a given {@code ClassDesc}.
     * The result either says that the given {@code ClassDesc} names an interface
     * ({@link #ofInterface()}) or that it names a class with a given superclass
     * ({@link #ofClass(ClassDesc)}).
     * <p>
     * The {@code of*} factory methods are the core building blocks. The {@code for*}
     * factory methods are helpers for common situations.
     */
    sealed interface Result permits ClassHierarchyLocatorResultImpl {
        /**
         * {@return a result meaning that the {@code ClassDesc} given to {@link #locate(ClassDesc)} names an interface}
         */
        static Result ofInterface() {
            return ClassHierarchyLocatorResultImpl.OF_INTERFACE;
        }

        /**
         * {@return a result meaning that the {@code ClassDesc} given to {@link #locate(ClassDesc)} names a class
         * with given {@code superClass}}
         *
         * @param superClass the superclass of the class named by the {@code ClassDesc} given to {@link #locate(ClassDesc)}
         *        (may be {@code null} in case the {@code ClassDesc} given to {@link #locate(ClassDesc)}
         *        names {@code java.lang.Object})
         */
        static Result ofClass(ClassDesc superClass) {
            if (superClass == null) {
                return ClassHierarchyLocatorResultImpl.OF_JAVA_LANG_OBJECT;
            }
            return new ClassHierarchyLocatorResultImpl(ClassHierarchyResolver.ClassHierarchyInfo.ofClass(superClass));
        }

        /**
         * {@return a class hierarchy information for the given {@code clazz}}
         * Returns {@code null} if the given {@code clazz} is {@code null}.
         *
         * @param clazz the class whose hierarchy information we return (may be {@code null})
         * @throws IllegalArgumentException if the given {@code clazz} is primitive, array or a hidden class
         */
        static Result forClass(Class<?> clazz) {
            if (clazz == null) {
                return null;
            }
            if (clazz.isPrimitive()) {
                throw new IllegalArgumentException("Primitive class " + clazz);
            }
            if (clazz.isArray()) {
                throw new IllegalArgumentException("Array class " + clazz);
            }
            if (clazz.isHidden()) {
                throw new IllegalArgumentException("Hidden class " + clazz);
            }

            if (clazz.isInterface()) {
                return ofInterface();
            }
            Class<?> superclass = clazz.getSuperclass();
            return superclass == null
                    ? ofClass(null)
                    : ofClass(superclass.describeConstable().orElseThrow(IllegalArgumentException::new));
        }

        /**
         * {@return a class hierarchy information for a {@code .class} file in the given {@code inputStream}}
         * Returns {@code null} if the given {@code inputStream} is {@code null}.
         *
         * @param inputStream the input stream for a {@code .class} file whose hierarchy information we return
         *        (may be {@code null})
         */
        static Result forInputStream(InputStream inputStream) {
            if (inputStream == null) {
                return null;
            }
            return new ClassHierarchyLocatorResultImpl(
                    // the class name here doesn't matter
                    ClassHierarchyResolver.ofResourceParsing(ignored -> inputStream).getClassInfo(CD_Object));
        }

        /**
         * {@return a class hierarchy information for a {@code .class} file in the given {@code bytes}}
         * Returns {@code null} if the given {@code bytes} is {@code null}.
         *
         * @param bytes the byte array for a {@code .class} file whose hierarchy information we return
         *        (may be {@code null})
         */
        static Result forBytes(byte[] bytes) {
            return forInputStream(new ByteArrayInputStream(bytes));
        }
    }

    /**
     * {@return a {@link Result} for the given {@code clazz}, or {@code null} when the given class
     * is not known to be accessible to the generated classes}
     *
     * @param clazz the class name (never {@code null})
     */
    Result locate(ClassDesc clazz);

    /**
     * Composes this {@code ClassHierarchyLocator} with the {@code other} one such that
     * when this locator does not know the given class (and {@link #locate(ClassDesc)}
     * returns {@code null}), the other one is consulted instead.
     *
     * @param other the {@code ClassHierarchyLocator} to compose this one with (must not be {@code null})
     * @return the composed {@code ClassHierarchyLocator} (never {@code null})
     */
    default ClassHierarchyLocator orElse(ClassHierarchyLocator other) {
        checkNotNullParam("other", other);
        return new ClassHierarchyLocator() {
            @Override
            public Result locate(ClassDesc clazz) {
                Result result = ClassHierarchyLocator.this.locate(clazz);
                return result != null ? result : other.locate(clazz);
            }
        };
    }

    /**
     * {@return this {@code ClassHierarchyLocator} wrapped in a caching facade}
     * The cached values are stored in a map provided by the given {@code cacheFactory}.
     *
     * @param cacheFactory supplies a {@code Map} that should be used to cache the located information
     *        (must not be {@code null} and must not return {@code null})
     * @return this {@code ClassHierarchyLocator}, but cached (never {@code null})
     */
    default ClassHierarchyLocator cached(Supplier<Map<ClassDesc, Result>> cacheFactory) {
        checkNotNullParam("cacheFactory", cacheFactory);
        return new ClassHierarchyLocatorCachedImpl(this, cacheFactory);
    }

    /**
     * {@return a {@link ClassHierarchyLocator} for the given classloader}
     * The classloader must make the {@code .class} files available as resources,
     * because internally, classes are not loaded to the JVM -- instead, their
     * contents are loaded from the classloader and minimally parsed.
     *
     * @param cl the classloader (must not be {@code null})
     */
    static ClassHierarchyLocator forClassParsing(ClassLoader cl) {
        checkNotNullParam("cl", cl);
        ClassHierarchyResolver resolver = ClassHierarchyResolver.ofResourceParsing(cl);
        return new ClassHierarchyLocator() {
            @Override
            public Result locate(ClassDesc clazz) {
                ClassHierarchyResolver.ClassHierarchyInfo info = resolver.getClassInfo(clazz);
                return info != null ? new ClassHierarchyLocatorResultImpl(info) : null;
            }
        };
    }

    /**
     * {@return a {@link ClassHierarchyLocator} for the given classloader}
     * Note that the requested classes are loaded to the JVM from the classloader.
     *
     * @param cl the classloader (must not be {@code null})
     */
    static ClassHierarchyLocator forClassLoading(ClassLoader cl) {
        checkNotNullParam("cl", cl);
        ClassHierarchyResolver resolver = ClassHierarchyResolver.ofClassLoading(cl);
        return new ClassHierarchyLocator() {
            @Override
            public Result locate(ClassDesc clazz) {
                ClassHierarchyResolver.ClassHierarchyInfo info = resolver.getClassInfo(clazz);
                return info != null ? new ClassHierarchyLocatorResultImpl(info) : null;
            }
        };
    }

    /**
     * {@return a {@link ClassHierarchyLocator} for the given resource loader}
     * The resource loader must provide access to all the {@code .class} files.
     * The classes are not loaded to the JVM -- instead, their contents are loaded
     * from the resource loader and minimally parsed.
     *
     * @param rl the resource loader (must not be {@code null})
     */
    static ClassHierarchyLocator forResourceParsing(ResourceLoader rl) {
        checkNotNullParam("rl", rl);
        ClassHierarchyResolver resolver = ClassHierarchyResolver.ofResourceParsing(clazz -> {
            try {
                Resource resource = rl.findResource(Util.internalName(clazz) + ".class");
                return resource != null ? resource.openStream() : null;
            } catch (IOException e) {
                throw new UncheckedIOException(e);
            }
        });
        return new ClassHierarchyLocator() {
            @Override
            public Result locate(ClassDesc clazz) {
                ClassHierarchyResolver.ClassHierarchyInfo info = resolver.getClassInfo(clazz);
                return info != null ? new ClassHierarchyLocatorResultImpl(info) : null;
            }
        };
    }
}
