/*
 * Decompiled with CFR 0.152.
 */
package io.quarkus.kubernetes.deployment;

import io.dekorate.utils.Serialization;
import io.fabric8.kubernetes.api.model.APIResourceList;
import io.fabric8.kubernetes.api.model.GenericKubernetesResource;
import io.fabric8.kubernetes.api.model.HasMetadata;
import io.fabric8.kubernetes.api.model.KubernetesList;
import io.fabric8.kubernetes.api.model.Service;
import io.fabric8.kubernetes.api.model.batch.v1.Job;
import io.fabric8.kubernetes.client.KubernetesClient;
import io.fabric8.kubernetes.client.KubernetesClientException;
import io.fabric8.kubernetes.client.dsl.Resource;
import io.fabric8.kubernetes.client.dsl.base.PatchContext;
import io.fabric8.kubernetes.client.dsl.base.PatchType;
import io.fabric8.kubernetes.client.dsl.base.ResourceDefinitionContext;
import io.fabric8.kubernetes.client.utils.ApiVersionUtil;
import io.fabric8.openshift.api.model.Route;
import io.fabric8.openshift.client.OpenShiftClient;
import io.quarkus.builder.item.BuildItem;
import io.quarkus.container.image.deployment.ContainerImageCapabilitiesUtil;
import io.quarkus.container.image.deployment.ContainerImageConfig;
import io.quarkus.container.spi.ContainerImageInfoBuildItem;
import io.quarkus.deployment.Capabilities;
import io.quarkus.deployment.IsNormalNotRemoteDev;
import io.quarkus.deployment.annotations.BuildProducer;
import io.quarkus.deployment.annotations.BuildStep;
import io.quarkus.deployment.builditem.ApplicationInfoBuildItem;
import io.quarkus.deployment.pkg.builditem.ArtifactResultBuildItem;
import io.quarkus.deployment.pkg.builditem.DeploymentResultBuildItem;
import io.quarkus.deployment.pkg.builditem.OutputTargetBuildItem;
import io.quarkus.kubernetes.client.deployment.KubernetesClientErrorHandler;
import io.quarkus.kubernetes.client.spi.KubernetesClientBuildItem;
import io.quarkus.kubernetes.deployment.DeploymentResourceKind;
import io.quarkus.kubernetes.deployment.DeploymentTargetEntry;
import io.quarkus.kubernetes.deployment.EnabledKubernetesDeploymentTargetsBuildItem;
import io.quarkus.kubernetes.deployment.KubernetesConfigUtil;
import io.quarkus.kubernetes.deployment.KubernetesDeploy;
import io.quarkus.kubernetes.deployment.OpenshiftConfig;
import io.quarkus.kubernetes.deployment.PreventImplicitContainerImagePushBuildItem;
import io.quarkus.kubernetes.deployment.ResourceNameUtil;
import io.quarkus.kubernetes.deployment.SelectedKubernetesDeploymentTargetBuildItem;
import io.quarkus.kubernetes.spi.DeployStrategy;
import io.quarkus.kubernetes.spi.GeneratedKubernetesResourceBuildItem;
import io.quarkus.kubernetes.spi.KubernetesDeploymentClusterBuildItem;
import io.quarkus.kubernetes.spi.KubernetesOptionalResourceDefinitionBuildItem;
import io.quarkus.kubernetes.spi.KubernetesOutputDirectoryBuildItem;
import io.quarkus.logging.Log;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Path;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.stream.Collectors;
import org.jboss.logging.Logger;

public class KubernetesDeployer {
    private static final Logger log = Logger.getLogger(KubernetesDeployer.class);
    private static final String CONTAINER_IMAGE_EXTENSIONS_STR = ContainerImageCapabilitiesUtil.CAPABILITY_TO_EXTENSION_NAME.values().stream().map(s -> "\"" + s + "\"").collect(Collectors.joining(", "));

    @BuildStep(onlyIf={IsNormalNotRemoteDev.class})
    public void selectDeploymentTarget(ContainerImageInfoBuildItem containerImageInfo, EnabledKubernetesDeploymentTargetsBuildItem targets, Capabilities capabilities, ContainerImageConfig containerImageConfig, BuildProducer<SelectedKubernetesDeploymentTargetBuildItem> selectedDeploymentTarget, BuildProducer<PreventImplicitContainerImagePushBuildItem> preventImplicitContainerImagePush) {
        Optional activeContainerImageCapability = ContainerImageCapabilitiesUtil.getActiveContainerImageCapability((Capabilities)capabilities);
        if (!containerImageConfig.isBuildExplicitlyDisabled() && !containerImageConfig.isPushExplicitlyDisabled() && activeContainerImageCapability.isEmpty()) {
            return;
        }
        DeploymentTargetEntry selectedTarget = this.determineDeploymentTarget(containerImageInfo, targets, containerImageConfig);
        selectedDeploymentTarget.produce((BuildItem)new SelectedKubernetesDeploymentTargetBuildItem(selectedTarget));
        if ("minikube".equals(selectedTarget.getName()) || "kind".equals(selectedTarget.getName())) {
            preventImplicitContainerImagePush.produce((BuildItem)new PreventImplicitContainerImagePushBuildItem());
        }
    }

    @BuildStep
    public void checkEnvironment(Optional<SelectedKubernetesDeploymentTargetBuildItem> selectedDeploymentTarget, KubernetesClientBuildItem kubernetesClientBuilder, List<GeneratedKubernetesResourceBuildItem> resources, BuildProducer<KubernetesDeploymentClusterBuildItem> deploymentCluster) {
        if (!KubernetesDeploy.INSTANCE.checkSilently(kubernetesClientBuilder)) {
            return;
        }
        String target = selectedDeploymentTarget.map(s -> s.getEntry().getName()).orElse("kubernetes");
        if (target.equals("kubernetes")) {
            deploymentCluster.produce((BuildItem)new KubernetesDeploymentClusterBuildItem("kubernetes"));
        }
    }

    @BuildStep(onlyIf={IsNormalNotRemoteDev.class})
    public void deploy(KubernetesClientBuildItem kubernetesClientBuilder, Capabilities capabilities, List<KubernetesDeploymentClusterBuildItem> deploymentClusters, Optional<SelectedKubernetesDeploymentTargetBuildItem> selectedDeploymentTarget, OutputTargetBuildItem outputTarget, KubernetesOutputDirectoryBuildItem outputDirectoryBuildItem, OpenshiftConfig openshiftConfig, ContainerImageConfig containerImageConfig, ApplicationInfoBuildItem applicationInfo, List<KubernetesOptionalResourceDefinitionBuildItem> optionalResourceDefinitions, BuildProducer<DeploymentResultBuildItem> deploymentResult, List<ArtifactResultBuildItem> artifactResults) {
        if (!KubernetesDeploy.INSTANCE.check(kubernetesClientBuilder)) {
            return;
        }
        if (selectedDeploymentTarget.isEmpty()) {
            if (!containerImageConfig.isBuildExplicitlyDisabled() && !containerImageConfig.isPushExplicitlyDisabled() && ContainerImageCapabilitiesUtil.getActiveContainerImageCapability((Capabilities)capabilities).isEmpty()) {
                throw new RuntimeException("A Kubernetes deployment was requested but no extension was found to build a container image. Consider adding one of following extensions: " + CONTAINER_IMAGE_EXTENSIONS_STR + ".");
            }
            return;
        }
        try (KubernetesClient client = kubernetesClientBuilder.buildClient();){
            deploymentResult.produce((BuildItem)this.deploy(selectedDeploymentTarget.get().getEntry(), client, outputDirectoryBuildItem.getOutputDirectory(), openshiftConfig, applicationInfo, optionalResourceDefinitions));
        }
    }

    private DeploymentTargetEntry determineDeploymentTarget(ContainerImageInfoBuildItem containerImageInfo, EnabledKubernetesDeploymentTargetsBuildItem targets, ContainerImageConfig containerImageConfig) {
        DeploymentTargetEntry selectedTarget;
        List<String> userSpecifiedDeploymentTargets = KubernetesConfigUtil.getExplicitlyConfiguredDeploymentTargets();
        if (userSpecifiedDeploymentTargets.isEmpty()) {
            selectedTarget = targets.getEntriesSortedByPriority().get(0);
            if (targets.getEntriesSortedByPriority().size() > 1) {
                log.info((Object)("Selecting target '" + selectedTarget.getName() + "' since it has the highest priority among the implicitly enabled deployment targets"));
            }
        } else {
            String firstUserSpecifiedDeploymentTarget = userSpecifiedDeploymentTargets.get(0);
            selectedTarget = targets.getEntriesSortedByPriority().stream().filter(d -> d.getName().equals(firstUserSpecifiedDeploymentTarget)).findFirst().orElseThrow(() -> new IllegalArgumentException("The specified value '" + firstUserSpecifiedDeploymentTarget + "' is not one of the allowed values of \"quarkus.kubernetes.deployment-target\""));
            if (userSpecifiedDeploymentTargets.size() > 1) {
                log.info((Object)("Only the first deployment target (which is '" + firstUserSpecifiedDeploymentTarget + "') selected via \"quarkus.kubernetes.deployment-target\" will be deployed"));
            }
        }
        return selectedTarget;
    }

    private DeploymentResultBuildItem deploy(DeploymentTargetEntry deploymentTarget, KubernetesClient client, Path outputDir, OpenshiftConfig openshiftConfig, ApplicationInfoBuildItem applicationInfo, List<KubernetesOptionalResourceDefinitionBuildItem> optionalResourceDefinitions) {
        DeploymentResultBuildItem deploymentResultBuildItem;
        String namespace = Optional.ofNullable(client.getNamespace()).orElse("default");
        log.info((Object)("Deploying to " + deploymentTarget.getName().toLowerCase() + " server: " + client.getMasterUrl() + " in namespace: " + namespace + "."));
        File manifest = outputDir.resolve(deploymentTarget.getName().toLowerCase() + ".yml").toFile();
        FileInputStream fis = new FileInputStream(manifest);
        try {
            KubernetesList list = Serialization.unmarshalAsList((InputStream)fis);
            Optional<GenericKubernetesResource> conflictingResource = this.findConflictingResource(client, deploymentTarget, list.getItems());
            if (conflictingResource.isPresent()) {
                String messsage = "Skipping deployment of " + deploymentTarget.getDeploymentResourceKind() + " " + conflictingResource.get().getMetadata().getName() + " because a " + conflictingResource.get().getKind() + " with the same name exists.";
                log.warn((Object)messsage);
                Log.warn((Object)"This may occur when switching deployment targets, or when the default deployment target is changed.");
                Log.warn((Object)"Please remove conflicting resource and try again.");
                throw new IllegalStateException(messsage);
            }
            list.getItems().stream().filter(KubernetesDeployer.distinctByResourceKey()).forEach(i -> {
                this.deployResource(deploymentTarget, client, (HasMetadata)i, optionalResourceDefinitions);
                log.info((Object)("Applied: " + i.getKind() + " " + i.getMetadata().getName() + "."));
            });
            this.printExposeInformation(client, list, openshiftConfig, applicationInfo);
            HasMetadata m = list.getItems().stream().filter(r -> deploymentTarget.getDeploymentResourceKind().matches((HasMetadata)r)).findFirst().orElseThrow(() -> new IllegalStateException("No " + deploymentTarget.getDeploymentResourceKind() + " found under: " + manifest.getAbsolutePath()));
            deploymentResultBuildItem = new DeploymentResultBuildItem(m.getMetadata().getName(), m.getMetadata().getLabels());
        }
        catch (Throwable throwable) {
            try {
                try {
                    fis.close();
                }
                catch (Throwable throwable2) {
                    throwable.addSuppressed(throwable2);
                }
                throw throwable;
            }
            catch (FileNotFoundException e) {
                throw new IllegalStateException("Can't find generated kubernetes manifest: " + manifest.getAbsolutePath());
            }
            catch (KubernetesClientException e) {
                KubernetesClientErrorHandler.handle((Exception)((Object)e));
                throw e;
            }
            catch (IOException e) {
                throw new RuntimeException("Error closing file: " + manifest.getAbsolutePath());
            }
        }
        fis.close();
        return deploymentResultBuildItem;
    }

    private void deployResource(DeploymentTargetEntry deploymentTarget, KubernetesClient client, HasMetadata metadata, List<KubernetesOptionalResourceDefinitionBuildItem> optionalResourceDefinitions) {
        Resource<?> r = this.findResource(client, metadata);
        if (KubernetesDeployer.shouldDeleteExisting(deploymentTarget, metadata)) {
            this.deleteResource(metadata, r);
        }
        try {
            switch (deploymentTarget.getDeployStrategy()) {
                case Create: {
                    r.create();
                    break;
                }
                case Replace: {
                    r.replace();
                    break;
                }
                case ServerSideApply: {
                    r.patch(PatchContext.of((PatchType)PatchType.SERVER_SIDE_APPLY));
                    break;
                }
                default: {
                    r.createOrReplace();
                    break;
                }
            }
        }
        catch (Exception e) {
            if (e instanceof InterruptedException) {
                throw e;
            }
            if (KubernetesDeployer.isOptional(optionalResourceDefinitions, metadata)) {
                log.warn((Object)("Failed to apply: " + metadata.getKind() + " " + metadata.getMetadata().getName() + ", possibly due to missing a CRD apiVersion: " + metadata.getApiVersion() + " and kind: " + metadata.getKind() + "."));
            }
            throw e;
        }
    }

    private Optional<GenericKubernetesResource> findConflictingResource(KubernetesClient clinet, DeploymentTargetEntry deploymentTarget, List<HasMetadata> generated) {
        HasMetadata deploymentResource = generated.stream().filter(r -> deploymentTarget.getDeploymentResourceKind().matches((HasMetadata)r)).findFirst().orElseThrow(() -> new IllegalStateException("No " + deploymentTarget.getDeploymentResourceKind() + " found under: " + deploymentTarget.getName()));
        String name = deploymentResource.getMetadata().getName();
        for (DeploymentResourceKind deploymentKind : DeploymentResourceKind.values()) {
            if (deploymentKind.matches(deploymentResource)) continue;
            try {
                GenericKubernetesResource resource = (GenericKubernetesResource)((Resource)clinet.genericKubernetesResources(deploymentKind.getApiVersion(), deploymentKind.getKind()).withName(name)).get();
                if (resource == null) continue;
                Log.warn((Object)("Found conflicting resource:" + resource.getApiVersion() + "/" + resource.getKind() + ":" + resource.getMetadata().getName()));
                return Optional.of(resource);
            }
            catch (KubernetesClientException kubernetesClientException) {
                // empty catch block
            }
        }
        return Optional.empty();
    }

    private void deleteResource(HasMetadata metadata, Resource<?> r) {
        r.delete();
        try {
            r.waitUntilCondition(Objects::isNull, 10L, TimeUnit.SECONDS);
        }
        catch (Exception e) {
            if (e instanceof InterruptedException) {
                throw e;
            }
            log.warn((Object)("Failed to wait for the deletion of: " + metadata.getApiVersion() + " " + metadata.getKind() + " " + metadata.getMetadata().getName() + ". Is the resource waitable?"));
        }
    }

    private Resource<?> findResource(KubernetesClient client, HasMetadata metadata) {
        if (metadata instanceof GenericKubernetesResource) {
            GenericKubernetesResource genericResource = (GenericKubernetesResource)metadata;
            ResourceDefinitionContext context = KubernetesDeployer.getGenericResourceContext(client, genericResource).orElseThrow(() -> new IllegalStateException("Could not retrieve API resource information for:" + metadata.getApiVersion() + " " + metadata.getKind() + ". Is the CRD for the resource available?"));
            return (Resource)client.genericKubernetesResources(context).resource((Object)genericResource);
        }
        return client.resource(metadata);
    }

    private void printExposeInformation(KubernetesClient client, KubernetesList list, OpenshiftConfig openshiftConfig, ApplicationInfoBuildItem applicationInfo) {
        String generatedRouteName = ResourceNameUtil.getResourceName(openshiftConfig, applicationInfo);
        List items = list.getItems();
        for (HasMetadata item : items) {
            if (!"route.openshift.io/v1".equals(item.getApiVersion()) || !"Route".equals(item.getKind()) || !generatedRouteName.equals(item.getMetadata().getName())) continue;
            try {
                OpenShiftClient openShiftClient = (OpenShiftClient)client.adapt(OpenShiftClient.class);
                Route route = (Route)((Resource)openShiftClient.routes().withName(generatedRouteName)).get();
                boolean isTLS = route.getSpec().getTls() != null;
                String host = route.getSpec().getHost();
                log.infov("The deployed application can be accessed at: http{0}://{1}", (Object)(isTLS ? "s" : ""), (Object)host);
            }
            catch (KubernetesClientException kubernetesClientException) {}
            break;
        }
    }

    private static Optional<ResourceDefinitionContext> getGenericResourceContext(KubernetesClient client, GenericKubernetesResource resource) {
        APIResourceList apiResourceList = client.getApiResources(resource.getApiVersion());
        if (apiResourceList == null || apiResourceList.getResources() == null || apiResourceList.getResources().isEmpty()) {
            return Optional.empty();
        }
        return client.getApiResources(resource.getApiVersion()).getResources().stream().filter(r -> r.getKind().equals(resource.getKind())).map(r -> new ResourceDefinitionContext.Builder().withGroup(ApiVersionUtil.trimGroup((String)resource.getApiVersion())).withVersion(ApiVersionUtil.trimVersion((String)resource.getApiVersion())).withKind(r.getKind()).withNamespaced(r.getNamespaced().booleanValue()).withPlural(r.getName()).build()).findFirst();
    }

    private static boolean isOptional(List<KubernetesOptionalResourceDefinitionBuildItem> optionalResourceDefinitions, HasMetadata resource) {
        return optionalResourceDefinitions.stream().anyMatch(t -> t.getApiVersion().equals(resource.getApiVersion()) && t.getKind().equals(resource.getKind()));
    }

    private static boolean shouldDeleteExisting(DeploymentTargetEntry deploymentTarget, HasMetadata resource) {
        if (deploymentTarget.getDeployStrategy() != DeployStrategy.CreateOrUpdate) {
            return false;
        }
        return "knative".equalsIgnoreCase(deploymentTarget.getName()) || resource instanceof Service || Objects.equals("v1", resource.getApiVersion()) && Objects.equals("Service", resource.getKind()) || resource instanceof Job || Objects.equals("batch/v1", resource.getApiVersion()) && Objects.equals("Job", resource.getKind());
    }

    private static Predicate<HasMetadata> distinctByResourceKey() {
        ConcurrentHashMap seen = new ConcurrentHashMap();
        return t -> seen.putIfAbsent(t.getApiVersion() + "/" + t.getKind() + ":" + t.getMetadata().getName(), Boolean.TRUE) == null;
    }
}

