/*
 * Decompiled with CFR 0.152.
 */
package io.quarkus.mongodb.runtime;

import com.mongodb.AuthenticationMechanism;
import com.mongodb.Block;
import com.mongodb.ConnectionString;
import com.mongodb.MongoClientSettings;
import com.mongodb.MongoCredential;
import com.mongodb.MongoException;
import com.mongodb.ReadPreference;
import com.mongodb.ServerAddress;
import com.mongodb.WriteConcern;
import com.mongodb.client.MongoClient;
import com.mongodb.connection.ClusterConnectionMode;
import com.mongodb.connection.ClusterSettings;
import com.mongodb.connection.ConnectionPoolSettings;
import com.mongodb.connection.ServerSettings;
import com.mongodb.connection.SocketSettings;
import com.mongodb.connection.SslSettings;
import com.mongodb.event.ConnectionPoolListener;
import io.quarkus.mongodb.impl.ReactiveMongoClientImpl;
import io.quarkus.mongodb.reactive.ReactiveMongoClient;
import io.quarkus.mongodb.runtime.MongoClientBeanUtil;
import io.quarkus.mongodb.runtime.MongoClientConfig;
import io.quarkus.mongodb.runtime.MongoClientSupport;
import io.quarkus.mongodb.runtime.MongodbConfig;
import io.quarkus.mongodb.runtime.WriteConcernConfig;
import java.lang.reflect.Constructor;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.TimeUnit;
import java.util.function.Function;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import javax.annotation.PreDestroy;
import javax.inject.Singleton;
import org.bson.codecs.configuration.CodecProvider;
import org.bson.codecs.configuration.CodecRegistries;
import org.bson.codecs.configuration.CodecRegistry;
import org.bson.codecs.pojo.ClassModel;
import org.bson.codecs.pojo.Conventions;
import org.bson.codecs.pojo.PojoCodecProvider;
import org.jboss.logging.Logger;

@Singleton
public class MongoClients {
    private static final Logger LOGGER = Logger.getLogger((String)MongoClients.class.getName());
    private static final Pattern COLON_PATTERN = Pattern.compile(":");
    private final MongodbConfig mongodbConfig;
    private final MongoClientSupport mongoClientSupport;
    private Map<String, MongoClient> mongoclients = new HashMap<String, MongoClient>();
    private Map<String, ReactiveMongoClient> reactiveMongoClients = new HashMap<String, ReactiveMongoClient>();

    public MongoClients(MongodbConfig mongodbConfig, MongoClientSupport mongoClientSupport) {
        this.mongodbConfig = mongodbConfig;
        this.mongoClientSupport = mongoClientSupport;
    }

    public MongoClient createMongoClient(String clientName) throws MongoException {
        MongoClientSettings mongoConfiguration = this.createMongoConfiguration(this.getMatchingMongoClientConfig(clientName));
        MongoClient client = com.mongodb.client.MongoClients.create((MongoClientSettings)mongoConfiguration);
        this.mongoclients.put(clientName, client);
        return client;
    }

    public ReactiveMongoClient createReactiveMongoClient(String clientName) throws MongoException {
        MongoClientSettings mongoConfiguration = this.createMongoConfiguration(this.getMatchingMongoClientConfig(clientName));
        com.mongodb.reactivestreams.client.MongoClient client = com.mongodb.reactivestreams.client.MongoClients.create((MongoClientSettings)mongoConfiguration);
        ReactiveMongoClientImpl reactive = new ReactiveMongoClientImpl(client);
        this.reactiveMongoClients.put(clientName, reactive);
        return reactive;
    }

    private MongoClientConfig getMatchingMongoClientConfig(String clientName) {
        return MongoClientBeanUtil.isDefault(clientName) ? this.mongodbConfig.defaultMongoClientConfig : this.mongodbConfig.mongoClientConfigs.get(clientName);
    }

    private MongoClientSettings createMongoConfiguration(MongoClientConfig config) {
        MongoCredential credential;
        if (config == null) {
            throw new RuntimeException("mongo config is missing for creating mongo client.");
        }
        CodecRegistry defaultCodecRegistry = MongoClientSettings.getDefaultCodecRegistry();
        MongoClientSettings.Builder settings = MongoClientSettings.builder();
        Optional<String> maybeConnectionString = config.connectionString;
        if (maybeConnectionString.isPresent()) {
            ConnectionString connectionString = new ConnectionString(maybeConnectionString.get());
            settings.applyConnectionString(connectionString);
        }
        ArrayList<Object> providers = new ArrayList<Object>();
        if (!this.mongoClientSupport.getCodecProviders().isEmpty()) {
            providers.addAll(this.getCodecProviders(this.mongoClientSupport.getCodecProviders()));
        }
        PojoCodecProvider.Builder pojoCodecProviderBuilder = PojoCodecProvider.builder().automatic(true).conventions(Conventions.DEFAULT_CONVENTIONS);
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        for (String bsonDiscriminator : this.mongoClientSupport.getBsonDiscriminators()) {
            try {
                pojoCodecProviderBuilder.register(new ClassModel[]{ClassModel.builder(Class.forName(bsonDiscriminator, true, classLoader)).enableDiscriminator(true).build()});
            }
            catch (ClassNotFoundException classNotFoundException) {}
        }
        providers.add(pojoCodecProviderBuilder.build());
        CodecRegistry registry = CodecRegistries.fromRegistries((CodecRegistry[])new CodecRegistry[]{defaultCodecRegistry, CodecRegistries.fromProviders(providers)});
        settings.codecRegistry(registry);
        config.applicationName.ifPresent(arg_0 -> ((MongoClientSettings.Builder)settings).applicationName(arg_0));
        if (config.credentials != null && (credential = this.createMongoCredential(config)) != null) {
            settings.credential(credential);
        }
        if (config.writeConcern != null) {
            Optional<String> maybeW;
            WriteConcernConfig wc = config.writeConcern;
            WriteConcern concern = (wc.safe ? WriteConcern.ACKNOWLEDGED : WriteConcern.UNACKNOWLEDGED).withJournal(Boolean.valueOf(wc.journal));
            if (wc.wTimeout.isPresent()) {
                concern = concern.withWTimeout(wc.wTimeout.get().toMillis(), TimeUnit.MILLISECONDS);
            }
            if ((maybeW = wc.w).isPresent()) {
                concern = concern.withW(maybeW.get());
            }
            settings.writeConcern(concern);
            settings.retryWrites(wc.retryWrites);
        }
        if (config.tls) {
            settings.applyToSslSettings((Block)new SslSettingsBuilder(config, this.mongoClientSupport.isDisableSslSupport()));
        }
        settings.applyToClusterSettings((Block)new ClusterSettingBuilder(config));
        settings.applyToConnectionPoolSettings((Block)new ConnectionPoolSettingsBuilder(config, this.mongoClientSupport.getConnectionPoolListeners()));
        settings.applyToServerSettings((Block)new ServerSettingsBuilder(config));
        settings.applyToSocketSettings((Block)new SocketSettingsBuilder(config));
        if (config.readPreference.isPresent()) {
            settings.readPreference(ReadPreference.valueOf((String)config.readPreference.get()));
        }
        return settings.build();
    }

    private static List<ServerAddress> parseHosts(List<String> addresses) {
        if (addresses.isEmpty()) {
            return Collections.singletonList(new ServerAddress(ServerAddress.defaultHost(), ServerAddress.defaultPort()));
        }
        return addresses.stream().map(String::trim).map(new addressParser()).collect(Collectors.toList());
    }

    private MongoCredential createMongoCredential(MongoClientConfig config) {
        MongoCredential credential;
        String username = config.credentials.username.orElse(null);
        if (username == null) {
            return null;
        }
        char[] password = config.credentials.password.map(String::toCharArray).orElse(null);
        String authSource = config.credentials.authSource.orElse(config.database.orElse("admin"));
        AuthenticationMechanism mechanism = null;
        Optional<String> maybeMechanism = config.credentials.authMechanism;
        if (maybeMechanism.isPresent()) {
            mechanism = this.getAuthenticationMechanism(maybeMechanism.get());
        }
        if (mechanism == AuthenticationMechanism.GSSAPI) {
            credential = MongoCredential.createGSSAPICredential((String)username);
        } else if (mechanism == AuthenticationMechanism.PLAIN) {
            credential = MongoCredential.createPlainCredential((String)username, (String)authSource, (char[])password);
        } else if (mechanism == AuthenticationMechanism.MONGODB_X509) {
            credential = MongoCredential.createMongoX509Credential((String)username);
        } else if (mechanism == AuthenticationMechanism.SCRAM_SHA_1) {
            credential = MongoCredential.createScramSha1Credential((String)username, (String)authSource, (char[])password);
        } else if (mechanism == null) {
            credential = MongoCredential.createCredential((String)username, (String)authSource, (char[])password);
        } else {
            throw new IllegalArgumentException("Unsupported authentication mechanism " + mechanism);
        }
        if (!config.credentials.authMechanismProperties.isEmpty()) {
            for (Map.Entry<String, String> entry : config.credentials.authMechanismProperties.entrySet()) {
                credential = credential.withMechanismProperty(entry.getKey(), (Object)entry.getValue());
            }
        }
        return credential;
    }

    private AuthenticationMechanism getAuthenticationMechanism(String authMechanism) {
        AuthenticationMechanism mechanism;
        try {
            mechanism = AuthenticationMechanism.fromMechanismName((String)authMechanism.toUpperCase());
        }
        catch (IllegalArgumentException e) {
            throw new IllegalArgumentException("Invalid authMechanism '" + authMechanism + "'");
        }
        return mechanism;
    }

    private List<CodecProvider> getCodecProviders(List<String> classNames) {
        ArrayList<CodecProvider> providers = new ArrayList<CodecProvider>();
        for (String name : classNames) {
            try {
                Class<?> clazz = Thread.currentThread().getContextClassLoader().loadClass(name);
                Constructor<?> clazzConstructor = clazz.getConstructor(new Class[0]);
                providers.add((CodecProvider)clazzConstructor.newInstance(new Object[0]));
            }
            catch (Exception e) {
                LOGGER.warnf((Throwable)e, "Unable to load the codec provider class %s", (Object)name);
            }
        }
        return providers;
    }

    @PreDestroy
    public void stop() {
        for (MongoClient client : this.mongoclients.values()) {
            if (client == null) continue;
            client.close();
        }
        for (ReactiveMongoClient reactive : this.reactiveMongoClients.values()) {
            if (reactive == null) continue;
            reactive.close();
        }
    }

    private static class addressParser
    implements Function<String, ServerAddress> {
        private addressParser() {
        }

        @Override
        public ServerAddress apply(String address) {
            String[] segments = COLON_PATTERN.split(address);
            if (segments.length == 1) {
                return new ServerAddress(address);
            }
            if (segments.length == 2) {
                return new ServerAddress(segments[0], Integer.parseInt(segments[1]));
            }
            throw new IllegalArgumentException("Invalid server address " + address);
        }
    }

    private static class ServerSettingsBuilder
    implements Block<ServerSettings.Builder> {
        private MongoClientConfig config;

        public ServerSettingsBuilder(MongoClientConfig config) {
            this.config = config;
        }

        public void apply(ServerSettings.Builder builder) {
            if (this.config.heartbeatFrequency.isPresent()) {
                builder.heartbeatFrequency((long)((int)this.config.heartbeatFrequency.get().toMillis()), TimeUnit.MILLISECONDS);
            }
        }
    }

    private static class SocketSettingsBuilder
    implements Block<SocketSettings.Builder> {
        private MongoClientConfig config;

        public SocketSettingsBuilder(MongoClientConfig config) {
            this.config = config;
        }

        public void apply(SocketSettings.Builder builder) {
            if (this.config.connectTimeout.isPresent()) {
                builder.connectTimeout((int)this.config.connectTimeout.get().toMillis(), TimeUnit.MILLISECONDS);
            }
            if (this.config.readTimeout.isPresent()) {
                builder.readTimeout((int)this.config.readTimeout.get().toMillis(), TimeUnit.MILLISECONDS);
            }
        }
    }

    private static class SslSettingsBuilder
    implements Block<SslSettings.Builder> {
        private MongoClientConfig config;
        private boolean disableSslSupport;

        public SslSettingsBuilder(MongoClientConfig config, boolean disableSslSupport) {
            this.config = config;
            this.disableSslSupport = disableSslSupport;
        }

        public void apply(SslSettings.Builder builder) {
            builder.enabled(!this.disableSslSupport).invalidHostNameAllowed(this.config.tlsInsecure);
        }
    }

    private static class ConnectionPoolSettingsBuilder
    implements Block<ConnectionPoolSettings.Builder> {
        private MongoClientConfig config;
        private List<ConnectionPoolListener> connectionPoolListeners;

        public ConnectionPoolSettingsBuilder(MongoClientConfig config, List<ConnectionPoolListener> connectionPoolListeners) {
            this.config = config;
            this.connectionPoolListeners = connectionPoolListeners;
        }

        public void apply(ConnectionPoolSettings.Builder builder) {
            this.config.maxPoolSize.ifPresent(arg_0 -> ((ConnectionPoolSettings.Builder)builder).maxSize(arg_0));
            this.config.minPoolSize.ifPresent(arg_0 -> ((ConnectionPoolSettings.Builder)builder).minSize(arg_0));
            if (this.config.maxConnectionIdleTime.isPresent()) {
                builder.maxConnectionIdleTime(this.config.maxConnectionIdleTime.get().toMillis(), TimeUnit.MILLISECONDS);
            }
            if (this.config.maxConnectionLifeTime.isPresent()) {
                builder.maxConnectionLifeTime(this.config.maxConnectionLifeTime.get().toMillis(), TimeUnit.MILLISECONDS);
            }
            if (this.config.maintenanceFrequency.isPresent()) {
                builder.maintenanceFrequency(this.config.maintenanceFrequency.get().toMillis(), TimeUnit.MILLISECONDS);
            }
            if (this.config.maintenanceInitialDelay.isPresent()) {
                builder.maintenanceInitialDelay(this.config.maintenanceInitialDelay.get().toMillis(), TimeUnit.MILLISECONDS);
            }
            for (ConnectionPoolListener connectionPoolListener : this.connectionPoolListeners) {
                builder.addConnectionPoolListener(connectionPoolListener);
            }
        }
    }

    private static class ClusterSettingBuilder
    implements Block<ClusterSettings.Builder> {
        private MongoClientConfig config;

        public ClusterSettingBuilder(MongoClientConfig config) {
            this.config = config;
        }

        public void apply(ClusterSettings.Builder builder) {
            Optional<String> maybeConnectionString = this.config.connectionString;
            if (!maybeConnectionString.isPresent()) {
                List hosts = MongoClients.parseHosts(this.config.hosts);
                builder.hosts(hosts);
                if (hosts.size() == 1 && !this.config.replicaSetName.isPresent()) {
                    builder.mode(ClusterConnectionMode.SINGLE);
                } else {
                    builder.mode(ClusterConnectionMode.MULTIPLE);
                }
            }
            if (this.config.localThreshold.isPresent()) {
                builder.localThreshold(this.config.localThreshold.get().toMillis(), TimeUnit.MILLISECONDS);
            }
            this.config.replicaSetName.ifPresent(arg_0 -> ((ClusterSettings.Builder)builder).requiredReplicaSetName(arg_0));
            if (this.config.serverSelectionTimeout.isPresent()) {
                builder.serverSelectionTimeout(this.config.serverSelectionTimeout.get().toMillis(), TimeUnit.MILLISECONDS);
            }
        }
    }
}

