/*
 * Decompiled with CFR 0.152.
 */
package io.quarkus.oidc.runtime;

import io.quarkus.oidc.AuthorizationCodeTokens;
import io.quarkus.oidc.OIDCException;
import io.quarkus.oidc.OidcConfigurationMetadata;
import io.quarkus.oidc.OidcTenantConfig;
import io.quarkus.oidc.TokenIntrospection;
import io.quarkus.oidc.UserInfo;
import io.quarkus.oidc.runtime.JsonWebKeySet;
import io.quarkus.oidc.runtime.OidcProviderClient;
import io.quarkus.oidc.runtime.OidcUtils;
import io.quarkus.oidc.runtime.TokenVerificationResult;
import io.quarkus.security.AuthenticationFailedException;
import io.smallrye.jwt.algorithm.SignatureAlgorithm;
import io.smallrye.jwt.util.KeyUtils;
import io.smallrye.mutiny.Uni;
import java.io.Closeable;
import java.security.Key;
import java.time.Duration;
import java.util.List;
import java.util.function.BiFunction;
import java.util.function.Function;
import org.eclipse.microprofile.jwt.Claims;
import org.jboss.logging.Logger;
import org.jose4j.jwa.AlgorithmConstraints;
import org.jose4j.jws.JsonWebSignature;
import org.jose4j.jwt.consumer.ErrorCodeValidator;
import org.jose4j.jwt.consumer.InvalidJwtException;
import org.jose4j.jwt.consumer.JwtConsumer;
import org.jose4j.jwt.consumer.JwtConsumerBuilder;
import org.jose4j.jwx.JsonWebStructure;
import org.jose4j.keys.resolvers.VerificationKeyResolver;
import org.jose4j.lang.UnresolvableKeyException;

public class OidcProvider
implements Closeable {
    private static final Logger LOG = Logger.getLogger(OidcProvider.class);
    private static final String ANY_ISSUER = "any";
    private static final String[] ASYMMETRIC_SUPPORTED_ALGORITHMS = new String[]{SignatureAlgorithm.RS256.getAlgorithm(), SignatureAlgorithm.RS384.getAlgorithm(), SignatureAlgorithm.RS512.getAlgorithm(), SignatureAlgorithm.ES256.getAlgorithm(), SignatureAlgorithm.ES384.getAlgorithm(), SignatureAlgorithm.ES512.getAlgorithm()};
    private static final AlgorithmConstraints ASYMMETRIC_ALGORITHM_CONSTRAINTS = new AlgorithmConstraints(AlgorithmConstraints.ConstraintType.PERMIT, ASYMMETRIC_SUPPORTED_ALGORITHMS);
    private static final AlgorithmConstraints SYMMETRIC_ALGORITHM_CONSTRAINTS = new AlgorithmConstraints(AlgorithmConstraints.ConstraintType.PERMIT, new String[]{SignatureAlgorithm.HS256.getAlgorithm()});
    final OidcProviderClient client;
    final RefreshableVerificationKeyResolver asymmetricKeyResolver;
    final OidcTenantConfig oidcConfig;
    final String issuer;
    final String[] audience;
    final Key tokenDecryptionKey;

    public OidcProvider(OidcProviderClient client, OidcTenantConfig oidcConfig, JsonWebKeySet jwks, Key tokenDecryptionKey) {
        this.client = client;
        this.oidcConfig = oidcConfig;
        this.asymmetricKeyResolver = jwks == null ? null : new JsonWebKeyResolver(jwks, oidcConfig.token.forcedJwkRefreshInterval);
        this.issuer = this.checkIssuerProp();
        this.audience = this.checkAudienceProp();
        this.tokenDecryptionKey = tokenDecryptionKey;
    }

    public OidcProvider(String publicKeyEnc, OidcTenantConfig oidcConfig, Key tokenDecryptionKey) {
        this.client = null;
        this.oidcConfig = oidcConfig;
        this.asymmetricKeyResolver = new LocalPublicKeyResolver(publicKeyEnc);
        this.issuer = this.checkIssuerProp();
        this.audience = this.checkAudienceProp();
        this.tokenDecryptionKey = tokenDecryptionKey;
    }

    private String checkIssuerProp() {
        String issuerProp = null;
        if (this.oidcConfig != null && (issuerProp = (String)this.oidcConfig.token.issuer.orElse(null)) == null && this.client != null) {
            issuerProp = this.client.getMetadata().getIssuer();
        }
        return ANY_ISSUER.equals(issuerProp) ? null : issuerProp;
    }

    private String[] checkAudienceProp() {
        List audienceProp = this.oidcConfig != null ? (List)this.oidcConfig.token.audience.orElse(null) : null;
        return audienceProp != null ? audienceProp.toArray(new String[0]) : null;
    }

    public TokenVerificationResult verifySelfSignedJwtToken(String token) throws InvalidJwtException {
        return this.verifyJwtTokenInternal(token, SYMMETRIC_ALGORITHM_CONSTRAINTS, new SymmetricKeyResolver(), true);
    }

    public TokenVerificationResult verifyJwtToken(String token) throws InvalidJwtException {
        return this.verifyJwtTokenInternal(token, ASYMMETRIC_ALGORITHM_CONSTRAINTS, this.asymmetricKeyResolver, true);
    }

    public TokenVerificationResult verifyLogoutJwtToken(String token) throws InvalidJwtException {
        Long exp;
        boolean enforceExpReq = !this.oidcConfig.token.age.isPresent();
        TokenVerificationResult result = this.verifyJwtTokenInternal(token, ASYMMETRIC_ALGORITHM_CONSTRAINTS, this.asymmetricKeyResolver, enforceExpReq);
        if (!enforceExpReq && (exp = result.localVerificationResult.getLong(Claims.exp.name())) != null) {
            this.verifyExpiry(exp);
        }
        return result;
    }

    private TokenVerificationResult verifyJwtTokenInternal(String token, AlgorithmConstraints algConstraints, VerificationKeyResolver verificationKeyResolver, boolean enforceExpReq) throws InvalidJwtException {
        long iat;
        long now;
        JwtConsumerBuilder builder = new JwtConsumerBuilder();
        builder.setVerificationKeyResolver(verificationKeyResolver);
        builder.setJwsAlgorithmConstraints(algConstraints);
        if (enforceExpReq) {
            builder.setRequireExpirationTime();
        }
        builder.setRequireIssuedAt();
        if (this.issuer != null) {
            builder.setExpectedIssuer(this.issuer);
        }
        if (this.audience != null) {
            builder.setExpectedAudience(this.audience);
        } else {
            builder.setSkipDefaultAudienceValidation();
        }
        if (this.oidcConfig.token.lifespanGrace.isPresent()) {
            int lifespanGrace = this.oidcConfig.token.lifespanGrace.getAsInt();
            builder.setAllowedClockSkewInSeconds(lifespanGrace);
        }
        builder.setRelaxVerificationKeyValidation();
        try {
            JwtConsumer jwtConsumer = builder.build();
            jwtConsumer.processToClaims(token);
        }
        catch (InvalidJwtException ex) {
            String detail = "";
            List details = ex.getErrorDetails();
            if (!details.isEmpty()) {
                detail = ((ErrorCodeValidator.Error)details.get(0)).getErrorMessage();
            }
            if (this.oidcConfig.clientId.isPresent()) {
                LOG.debugf("Verification of the token issued to client %s has failed: %s", this.oidcConfig.clientId.get(), (Object)detail);
            } else {
                LOG.debugf("Token verification has failed: %s", (Object)detail);
            }
            throw ex;
        }
        TokenVerificationResult result = new TokenVerificationResult(OidcUtils.decodeJwtContent(token), null);
        if (this.oidcConfig.token.age.isPresent() && (now = OidcProvider.now() / 1000L) - (iat = result.localVerificationResult.getLong(Claims.iat.name()).longValue()) > this.oidcConfig.token.age.get().toSeconds() + (long)this.getLifespanGrace()) {
            String errorMessage = "Logout token age exceeds the configured token age property";
            LOG.debugf("Logout token age exceeds the configured token age property", new Object[0]);
            throw new InvalidJwtException("Logout token age exceeds the configured token age property", List.of(new ErrorCodeValidator.Error(24, "Logout token age exceeds the configured token age property")), null);
        }
        return result;
    }

    public Uni<TokenVerificationResult> refreshJwksAndVerifyJwtToken(final String token) {
        return this.asymmetricKeyResolver.refresh().onItem().transformToUni((Function)new Function<Void, Uni<? extends TokenVerificationResult>>(){

            @Override
            public Uni<? extends TokenVerificationResult> apply(Void v) {
                try {
                    return Uni.createFrom().item((Object)OidcProvider.this.verifyJwtToken(token));
                }
                catch (Throwable t) {
                    return Uni.createFrom().failure(t);
                }
            }
        });
    }

    public Uni<TokenIntrospection> introspectToken(String token) {
        if (this.client.getMetadata().getIntrospectionUri() == null) {
            LOG.debugf("Token issued to client %s can not be introspected because the introspection endpoint address is unknown - please check if your OpenId Connect Provider supports the token introspection", this.oidcConfig.clientId.get());
            throw new AuthenticationFailedException();
        }
        return this.client.introspectToken(token).onItemOrFailure().transform((BiFunction)new BiFunction<TokenIntrospection, Throwable, TokenIntrospection>(){

            @Override
            public TokenIntrospection apply(TokenIntrospection introspectionResult, Throwable t) {
                if (t != null) {
                    throw new AuthenticationFailedException(t);
                }
                if (!Boolean.TRUE.equals(introspectionResult.getBoolean("active"))) {
                    LOG.debugf("Token issued to client %s is not active", OidcProvider.this.oidcConfig.clientId.get());
                    throw new AuthenticationFailedException();
                }
                Long exp = introspectionResult.getLong("exp");
                if (exp != null) {
                    OidcProvider.this.verifyExpiry(exp);
                }
                return introspectionResult;
            }
        });
    }

    private void verifyExpiry(Long exp) {
        if (OidcProvider.now() / 1000L > exp + (long)this.getLifespanGrace()) {
            LOG.debugf("Token issued to client %s has expired", this.oidcConfig.clientId.get());
            throw new AuthenticationFailedException();
        }
    }

    private int getLifespanGrace() {
        return this.client.getOidcConfig().token.lifespanGrace.isPresent() ? this.client.getOidcConfig().token.lifespanGrace.getAsInt() : 0;
    }

    private static final long now() {
        return System.currentTimeMillis();
    }

    public Uni<UserInfo> getUserInfo(String accessToken) {
        return this.client.getUserInfo(accessToken);
    }

    public Uni<AuthorizationCodeTokens> getCodeFlowTokens(String code, String redirectUri, String codeVerifier) {
        return this.client.getAuthorizationCodeTokens(code, redirectUri, codeVerifier);
    }

    public Uni<AuthorizationCodeTokens> refreshTokens(String refreshToken) {
        return this.client.refreshAuthorizationCodeTokens(refreshToken);
    }

    @Override
    public void close() {
        if (this.client != null) {
            this.client.close();
        }
    }

    public OidcConfigurationMetadata getMetadata() {
        return this.client.getMetadata();
    }

    private static interface RefreshableVerificationKeyResolver
    extends VerificationKeyResolver {
        default public Uni<Void> refresh() {
            return Uni.createFrom().voidItem();
        }
    }

    private class SymmetricKeyResolver
    implements VerificationKeyResolver {
        private SymmetricKeyResolver() {
        }

        public Key resolveKey(JsonWebSignature jws, List<JsonWebStructure> nestingContext) throws UnresolvableKeyException {
            return KeyUtils.createSecretKeyFromSecret((String)((String)OidcProvider.this.oidcConfig.credentials.secret.get()));
        }
    }

    private static class LocalPublicKeyResolver
    implements RefreshableVerificationKeyResolver {
        Key key;

        LocalPublicKeyResolver(String publicKeyEnc) {
            try {
                this.key = KeyUtils.decodePublicKey((String)publicKeyEnc);
            }
            catch (Exception ex) {
                throw new OIDCException(ex);
            }
        }

        public Key resolveKey(JsonWebSignature jws, List<JsonWebStructure> nestingContext) throws UnresolvableKeyException {
            return this.key;
        }
    }

    private class JsonWebKeyResolver
    implements RefreshableVerificationKeyResolver {
        volatile JsonWebKeySet jwks;
        volatile long lastForcedRefreshTime;
        volatile long forcedJwksRefreshIntervalMilliSecs;

        JsonWebKeyResolver(JsonWebKeySet jwks, Duration forcedJwksRefreshInterval) {
            this.jwks = jwks;
            this.forcedJwksRefreshIntervalMilliSecs = forcedJwksRefreshInterval.toMillis();
        }

        public Key resolveKey(JsonWebSignature jws, List<JsonWebStructure> nestingContext) throws UnresolvableKeyException {
            Key key = null;
            String kid = jws.getKeyIdHeaderValue();
            if (kid != null && (key = this.getKeyWithId(jws, kid)) == null) {
                throw new UnresolvableKeyException(String.format("JWK with kid '%s' is not available", kid));
            }
            String thumbprint = null;
            if (key == null && (thumbprint = jws.getHeader("x5t")) != null && (key = this.getKeyWithThumbprint(jws, thumbprint)) == null) {
                throw new UnresolvableKeyException(String.format("JWK with thumbprint '%s' is not available", thumbprint));
            }
            if (key == null && kid == null && thumbprint == null) {
                key = this.jwks.getKeyWithoutKeyIdAndThumbprint();
            }
            if (key == null) {
                throw new UnresolvableKeyException(String.format("JWK is not available, neither 'kid' nor 'x5t' token headers are set", kid));
            }
            return key;
        }

        private Key getKeyWithId(JsonWebSignature jws, String kid) {
            if (kid != null) {
                return this.jwks.getKeyWithId(kid);
            }
            LOG.debug((Object)"Token 'kid' header is not set");
            return null;
        }

        private Key getKeyWithThumbprint(JsonWebSignature jws, String thumbprint) {
            if (thumbprint != null) {
                return this.jwks.getKeyWithThumbprint(thumbprint);
            }
            LOG.debug((Object)"Token 'x5t' header is not set");
            return null;
        }

        @Override
        public Uni<Void> refresh() {
            long now = OidcProvider.now();
            if (now > this.lastForcedRefreshTime + this.forcedJwksRefreshIntervalMilliSecs) {
                this.lastForcedRefreshTime = now;
                return OidcProvider.this.client.getJsonWebKeySet().onItem().transformToUni((Function)new Function<JsonWebKeySet, Uni<? extends Void>>(){

                    @Override
                    public Uni<? extends Void> apply(JsonWebKeySet t) {
                        JsonWebKeyResolver.this.jwks = t;
                        return Uni.createFrom().voidItem();
                    }
                });
            }
            return Uni.createFrom().voidItem();
        }
    }
}

