/*
 * Decompiled with CFR 0.152.
 */
package io.quarkus.scheduler.runtime;

import com.cronutils.model.Cron;
import com.cronutils.model.CronType;
import com.cronutils.model.definition.CronDefinition;
import com.cronutils.model.definition.CronDefinitionBuilder;
import com.cronutils.model.time.ExecutionTime;
import com.cronutils.parser.CronParser;
import io.quarkus.runtime.StartupEvent;
import io.quarkus.scheduler.Scheduled;
import io.quarkus.scheduler.ScheduledExecution;
import io.quarkus.scheduler.Scheduler;
import io.quarkus.scheduler.Trigger;
import io.quarkus.scheduler.runtime.ScheduledInvoker;
import io.quarkus.scheduler.runtime.ScheduledMethodMetadata;
import io.quarkus.scheduler.runtime.SchedulerContext;
import java.time.Duration;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.List;
import java.util.Optional;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import javax.annotation.PreDestroy;
import javax.annotation.Priority;
import javax.enterprise.event.Observes;
import javax.enterprise.inject.Typed;
import javax.inject.Singleton;
import org.eclipse.microprofile.config.Config;
import org.eclipse.microprofile.config.spi.ConfigProviderResolver;
import org.jboss.logging.Logger;
import org.jboss.threads.JBossScheduledThreadPoolExecutor;

@Typed(value={Scheduler.class})
@Singleton
public class SimpleScheduler
implements Scheduler {
    private static final Logger LOGGER = Logger.getLogger(SimpleScheduler.class);
    private static final long CHECK_PERIOD = 1000L;
    private final ScheduledExecutorService scheduledExecutor;
    private final ExecutorService executor;
    private volatile boolean running = true;
    private final List<ScheduledTask> scheduledTasks = new ArrayList<ScheduledTask>();
    private final AtomicInteger triggerNameSequence = new AtomicInteger();
    private final Config config;

    public SimpleScheduler(SchedulerContext context, Config config) {
        this.executor = context.getExecutor();
        this.config = config;
        if (context.getScheduledMethods().isEmpty()) {
            this.scheduledExecutor = null;
        } else {
            this.scheduledExecutor = new JBossScheduledThreadPoolExecutor(1, new Runnable(){

                @Override
                public void run() {
                }
            });
            CronDefinition definition = CronDefinitionBuilder.instanceDefinitionFor((CronType)context.getCronType());
            CronParser parser = new CronParser(definition);
            for (ScheduledMethodMetadata method : context.getScheduledMethods()) {
                ScheduledInvoker invoker = context.createInvoker(method.getInvokerClassName());
                for (Scheduled scheduled : method.getSchedules()) {
                    SimpleTrigger trigger = this.createTrigger(method.getInvokerClassName(), parser, scheduled);
                    this.scheduledTasks.add(new ScheduledTask(trigger, invoker));
                }
            }
        }
    }

    void start(@Observes @Priority(value=0) StartupEvent event) {
        if (this.scheduledExecutor == null) {
            return;
        }
        LocalDateTime now = LocalDateTime.now();
        LocalDateTime trunc = now.plusSeconds(1L).truncatedTo(ChronoUnit.SECONDS);
        this.scheduledExecutor.scheduleAtFixedRate(this::checkTriggers, ChronoUnit.MILLIS.between(now, trunc), 1000L, TimeUnit.MILLISECONDS);
    }

    @PreDestroy
    void stop() {
        try {
            if (this.scheduledExecutor != null) {
                this.scheduledExecutor.shutdownNow();
            }
        }
        catch (Exception e) {
            LOGGER.warn((Object)"Unable to shutdown the scheduler executor", (Throwable)e);
        }
    }

    void checkTriggers() {
        if (!this.running) {
            LOGGER.trace((Object)"Skip all triggers - scheduler paused");
        }
        final ZonedDateTime now = ZonedDateTime.now();
        for (final ScheduledTask task : this.scheduledTasks) {
            LOGGER.tracef("Evaluate trigger %s", (Object)task.trigger);
            final ZonedDateTime scheduledFireTime = task.trigger.evaluate(now);
            if (scheduledFireTime == null) continue;
            try {
                this.executor.execute(new Runnable(){

                    @Override
                    public void run() {
                        try {
                            task.invoker.invoke(new SimpleScheduledExecution(now, scheduledFireTime, task.trigger));
                        }
                        catch (Throwable t) {
                            LOGGER.errorf(t, "Error occured while executing task for trigger %s", (Object)task.trigger);
                        }
                    }
                });
                LOGGER.debugf("Executing scheduled task for trigger %s", (Object)task.trigger);
            }
            catch (RejectedExecutionException e) {
                LOGGER.warnf("Rejected execution of a scheduled task for trigger %s", (Object)task.trigger);
            }
        }
    }

    @Override
    public void pause() {
        this.running = false;
    }

    @Override
    public void resume() {
        this.running = true;
    }

    SimpleTrigger createTrigger(String invokerClass, CronParser parser, Scheduled scheduled) {
        String cron;
        String id = this.triggerNameSequence.getAndIncrement() + "_" + invokerClass;
        ZonedDateTime start = ZonedDateTime.now().truncatedTo(ChronoUnit.SECONDS);
        Long millisToAdd = null;
        if (scheduled.delay() > 0L) {
            millisToAdd = scheduled.delayUnit().toMillis(scheduled.delay());
        } else if (!scheduled.delayed().isEmpty()) {
            millisToAdd = Math.abs(SimpleScheduler.parseDuration(scheduled, scheduled.delayed(), "delayed").toMillis());
        }
        if (millisToAdd != null) {
            start = start.toInstant().plusMillis(millisToAdd).atZone(start.getZone());
        }
        if (!(cron = scheduled.cron().trim()).isEmpty()) {
            Cron cronExpr;
            if (SchedulerContext.isConfigValue(cron)) {
                cron = (String)this.config.getValue(SchedulerContext.getConfigProperty(cron), String.class);
            }
            try {
                cronExpr = parser.parse(cron);
            }
            catch (IllegalArgumentException e) {
                throw new IllegalArgumentException("Cannot parse cron expression: " + cron, e);
            }
            return new CronTrigger(id, start, cronExpr);
        }
        if (!scheduled.every().isEmpty()) {
            return new IntervalTrigger(id, start, Math.abs(SimpleScheduler.parseDuration(scheduled, scheduled.every(), "every").toMillis()));
        }
        throw new IllegalArgumentException("Invalid schedule configuration: " + scheduled);
    }

    public static Duration parseDuration(Scheduled scheduled, String value, String memberName) {
        if (SchedulerContext.isConfigValue(value = value.trim())) {
            value = (String)ConfigProviderResolver.instance().getConfig().getValue(SchedulerContext.getConfigProperty(value), String.class);
        }
        if (Character.isDigit(value.charAt(0))) {
            value = "PT" + value;
        }
        try {
            return Duration.parse(value);
        }
        catch (Exception e) {
            throw new IllegalStateException("Invalid " + memberName + "() expression on: " + scheduled, e);
        }
    }

    static class SimpleScheduledExecution
    implements ScheduledExecution {
        private final ZonedDateTime fireTime;
        private final ZonedDateTime scheduledFireTime;
        private final Trigger trigger;

        public SimpleScheduledExecution(ZonedDateTime fireTime, ZonedDateTime scheduledFireTime, SimpleTrigger trigger) {
            this.fireTime = fireTime;
            this.scheduledFireTime = scheduledFireTime;
            this.trigger = trigger;
        }

        @Override
        public Trigger getTrigger() {
            return this.trigger;
        }

        @Override
        public Instant getFireTime() {
            return this.fireTime.toInstant();
        }

        @Override
        public Instant getScheduledFireTime() {
            return this.scheduledFireTime.toInstant();
        }
    }

    static class CronTrigger
    extends SimpleTrigger {
        private static final long DIFF_THRESHOLD = 1000000L;
        private final Cron cron;
        private final ExecutionTime executionTime;

        public CronTrigger(String id, ZonedDateTime start, Cron cron) {
            super(id, start);
            this.cron = cron;
            this.executionTime = ExecutionTime.forCron((Cron)cron);
        }

        @Override
        public Instant getNextFireTime() {
            Optional nextFireTime = this.executionTime.nextExecution(ZonedDateTime.now());
            return nextFireTime.isPresent() ? ((ZonedDateTime)nextFireTime.get()).toInstant() : null;
        }

        @Override
        public Instant getPreviousFireTime() {
            Optional prevFireTime = this.executionTime.lastExecution(ZonedDateTime.now());
            return prevFireTime.isPresent() ? ((ZonedDateTime)prevFireTime.get()).toInstant() : null;
        }

        @Override
        ZonedDateTime evaluate(ZonedDateTime now) {
            if (now.isBefore(this.start)) {
                return null;
            }
            Optional lastFireTime = this.executionTime.lastExecution(now);
            if (lastFireTime.isPresent()) {
                ZonedDateTime trunc = ((ZonedDateTime)lastFireTime.get()).truncatedTo(ChronoUnit.SECONDS);
                if (now.isBefore(trunc)) {
                    return null;
                }
                long diff = ChronoUnit.MICROS.between(trunc, now);
                if (diff <= 1000000L) {
                    LOGGER.debugf("%s fired, diff=%s \u03bcs", (Object)this, (Object)diff);
                    return trunc;
                }
            }
            return null;
        }

        public String toString() {
            StringBuilder builder = new StringBuilder();
            builder.append("CronTrigger [id=").append(this.getId()).append(", cron=").append(this.cron.asString()).append("]");
            return builder.toString();
        }
    }

    static class IntervalTrigger
    extends SimpleTrigger {
        private final long interval;
        private volatile ZonedDateTime lastFireTime;

        public IntervalTrigger(String id, ZonedDateTime start, long interval) {
            super(id, start);
            this.interval = interval;
        }

        @Override
        ZonedDateTime evaluate(ZonedDateTime now) {
            if (now.isBefore(this.start)) {
                return null;
            }
            if (this.lastFireTime == null) {
                this.lastFireTime = now.truncatedTo(ChronoUnit.SECONDS);
                return now;
            }
            if (ChronoUnit.MILLIS.between(this.lastFireTime, now) >= this.interval) {
                ZonedDateTime scheduledFireTime = this.lastFireTime.plus(Duration.ofMillis(this.interval));
                this.lastFireTime = now.truncatedTo(ChronoUnit.SECONDS);
                return scheduledFireTime;
            }
            return null;
        }

        @Override
        public Instant getNextFireTime() {
            return this.lastFireTime.plus(Duration.ofMillis(this.interval)).toInstant();
        }

        @Override
        public Instant getPreviousFireTime() {
            return this.lastFireTime.toInstant();
        }

        public String toString() {
            StringBuilder builder = new StringBuilder();
            builder.append("IntervalTrigger [id=").append(this.getId()).append(", interval=").append(this.interval).append("]");
            return builder.toString();
        }
    }

    static abstract class SimpleTrigger
    implements Trigger {
        private final String id;
        protected final ZonedDateTime start;

        public SimpleTrigger(String id, ZonedDateTime start) {
            this.id = id;
            this.start = start;
        }

        abstract ZonedDateTime evaluate(ZonedDateTime var1);

        @Override
        public String getId() {
            return this.id;
        }
    }

    static class ScheduledTask {
        final SimpleTrigger trigger;
        final ScheduledInvoker invoker;

        public ScheduledTask(SimpleTrigger trigger, ScheduledInvoker invoker) {
            this.trigger = trigger;
            this.invoker = invoker;
        }
    }
}

