/*
 * Decompiled with CFR 0.152.
 */
package io.quarkus.vault.runtime;

import io.quarkus.vault.VaultException;
import io.quarkus.vault.runtime.LogConfidentialityLevel;
import io.quarkus.vault.runtime.VaultToken;
import io.quarkus.vault.runtime.client.VaultClient;
import io.quarkus.vault.runtime.client.VaultClientException;
import io.quarkus.vault.runtime.client.dto.auth.AbstractVaultAuthAuth;
import io.quarkus.vault.runtime.client.dto.auth.VaultAppRoleGenerateNewSecretID;
import io.quarkus.vault.runtime.client.dto.auth.VaultAppRoleGenerateNewSecretIDData;
import io.quarkus.vault.runtime.client.dto.auth.VaultKubernetesAuthAuth;
import io.quarkus.vault.runtime.client.dto.auth.VaultRenewSelfAuth;
import io.quarkus.vault.runtime.client.dto.auth.VaultTokenCreate;
import io.quarkus.vault.runtime.client.dto.auth.VaultTokenCreateAuth;
import io.quarkus.vault.runtime.client.dto.kv.VaultKvSecretV1;
import io.quarkus.vault.runtime.client.dto.kv.VaultKvSecretV2;
import io.quarkus.vault.runtime.client.dto.kv.VaultKvSecretV2Data;
import io.quarkus.vault.runtime.config.VaultAuthenticationType;
import io.quarkus.vault.runtime.config.VaultRuntimeConfig;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Paths;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.Semaphore;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Function;
import org.jboss.logging.Logger;

public class VaultAuthManager {
    private static final Logger log = Logger.getLogger((String)VaultAuthManager.class.getName());
    public static final String USERPASS_WRAPPING_TOKEN_PASSWORD_KEY = "password";
    private VaultRuntimeConfig serverConfig;
    private VaultClient vaultClient;
    private AtomicReference<VaultToken> loginCache = new AtomicReference<Object>(null);
    private Map<String, String> wrappedCache = new ConcurrentHashMap<String, String>();
    private Semaphore unwrapSem = new Semaphore(1);

    public VaultAuthManager(VaultClient vaultClient, VaultRuntimeConfig serverConfig) {
        this.vaultClient = vaultClient;
        this.serverConfig = serverConfig;
    }

    public String getClientToken() {
        return this.serverConfig.authentication.isDirectClientToken() ? this.getDirectClientToken() : this.login().clientToken;
    }

    private String getDirectClientToken() {
        Optional<String> clientTokenOption = this.serverConfig.authentication.clientToken;
        if (clientTokenOption.isPresent()) {
            return clientTokenOption.get();
        }
        return this.unwrapWrappingTokenOnce("client token", this.serverConfig.authentication.clientTokenWrappingToken.get(), unwrap -> ((VaultTokenCreateAuth)unwrap.auth).clientToken, VaultTokenCreate.class);
    }

    private VaultToken login() {
        VaultToken vaultToken = this.login(this.loginCache.get());
        this.loginCache.set(vaultToken);
        return vaultToken;
    }

    public VaultToken login(VaultToken currentVaultToken) {
        VaultToken vaultToken = currentVaultToken;
        if (vaultToken != null) {
            vaultToken = this.validate(vaultToken);
        }
        if (vaultToken != null && vaultToken.shouldExtend(this.serverConfig.renewGracePeriod)) {
            vaultToken = this.extend(vaultToken.clientToken);
        }
        if (vaultToken == null || vaultToken.isExpired() || vaultToken.expiresSoon(this.serverConfig.renewGracePeriod)) {
            vaultToken = this.vaultLogin();
        }
        return vaultToken;
    }

    private VaultToken validate(VaultToken vaultToken) {
        try {
            this.vaultClient.lookupSelf(vaultToken.clientToken);
            return vaultToken;
        }
        catch (VaultClientException e) {
            if (e.getStatus() == 403) {
                log.debug((Object)("login token " + vaultToken.clientToken + " has become invalid"));
                return null;
            }
            throw e;
        }
    }

    private VaultToken extend(String clientToken) {
        VaultRenewSelfAuth auth = (VaultRenewSelfAuth)this.vaultClient.renewSelf((String)clientToken, null).auth;
        VaultToken vaultToken = new VaultToken(auth.clientToken, auth.renewable, auth.leaseDurationSecs);
        this.sanityCheck(vaultToken);
        log.debug((Object)("extended login token: " + vaultToken.getConfidentialInfo(this.serverConfig.logConfidentialityLevel)));
        return vaultToken;
    }

    private VaultToken vaultLogin() {
        VaultToken vaultToken = this.login(this.serverConfig.getAuthenticationType());
        this.sanityCheck(vaultToken);
        log.debug((Object)("created new login token: " + vaultToken.getConfidentialInfo(this.serverConfig.logConfidentialityLevel)));
        return vaultToken;
    }

    private VaultToken login(VaultAuthenticationType type) {
        VaultKubernetesAuthAuth auth;
        if (type == VaultAuthenticationType.KUBERNETES) {
            auth = this.loginKubernetes();
        } else if (type == VaultAuthenticationType.USERPASS) {
            String username = this.serverConfig.authentication.userpass.username.get();
            String password = this.getPassword();
            auth = (AbstractVaultAuthAuth)this.vaultClient.loginUserPass((String)username, (String)password).auth;
        } else if (type == VaultAuthenticationType.APPROLE) {
            String roleId = this.serverConfig.authentication.appRole.roleId.get();
            String secretId = this.getSecretId();
            auth = (AbstractVaultAuthAuth)this.vaultClient.loginAppRole((String)roleId, (String)secretId).auth;
        } else {
            throw new UnsupportedOperationException("unknown authType " + (Object)((Object)this.serverConfig.getAuthenticationType()));
        }
        return new VaultToken(auth.clientToken, auth.renewable, auth.leaseDurationSecs);
    }

    private String getSecretId() {
        Optional<String> secretIdOption = this.serverConfig.authentication.appRole.secretId;
        if (secretIdOption.isPresent()) {
            return secretIdOption.get();
        }
        return this.unwrapWrappingTokenOnce("secret id", this.serverConfig.authentication.appRole.secretIdWrappingToken.get(), unwrap -> ((VaultAppRoleGenerateNewSecretIDData)unwrap.data).secretId, VaultAppRoleGenerateNewSecretID.class);
    }

    private String getPassword() {
        Optional<String> passwordOption = this.serverConfig.authentication.userpass.password;
        if (passwordOption.isPresent()) {
            return passwordOption.get();
        }
        String wrappingToken = this.serverConfig.authentication.userpass.passwordWrappingToken.get();
        if (this.serverConfig.kvSecretEngineVersion == 1) {
            Function<VaultKvSecretV1, String> f = unwrap -> (String)((Map)unwrap.data).get(USERPASS_WRAPPING_TOKEN_PASSWORD_KEY);
            return this.unwrapWrappingTokenOnce(USERPASS_WRAPPING_TOKEN_PASSWORD_KEY, wrappingToken, f, VaultKvSecretV1.class);
        }
        Function<VaultKvSecretV2, String> f = unwrap -> (String)((VaultKvSecretV2Data)unwrap.data).data.get(USERPASS_WRAPPING_TOKEN_PASSWORD_KEY);
        return this.unwrapWrappingTokenOnce(USERPASS_WRAPPING_TOKEN_PASSWORD_KEY, wrappingToken, f, VaultKvSecretV2.class);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    private <T> String unwrapWrappingTokenOnce(String type, String wrappingToken, Function<T, String> f, Class<T> clazz) {
        String wrappedValue = this.wrappedCache.get(wrappingToken);
        if (wrappedValue != null) {
            return wrappedValue;
        }
        try {
            boolean success = this.unwrapSem.tryAcquire(1, 10L, TimeUnit.SECONDS);
            if (!success) {
                throw new RuntimeException("unable to enter critical section when unwrapping " + type);
            }
            try {
                T unwrap;
                wrappedValue = this.wrappedCache.get(wrappingToken);
                if (wrappedValue != null) {
                    String string = wrappedValue;
                    return string;
                }
                try {
                    unwrap = this.vaultClient.unwrap(wrappingToken, clazz);
                }
                catch (VaultClientException e) {
                    if (e.getStatus() != 400) throw e;
                    String message = "wrapping token is not valid or does not exist; this means that the token has already expired (if so you can increase the ttl on the wrapping token) or has been consumed by somebody else (potentially indicating that the wrapping token has been stolen)";
                    throw new VaultException(message, e);
                }
                wrappedValue = f.apply(unwrap);
                this.wrappedCache.put(wrappingToken, wrappedValue);
                String displayValue = this.serverConfig.logConfidentialityLevel.maskWithTolerance(wrappedValue, LogConfidentialityLevel.LOW);
                log.debug((Object)("unwrapped " + type + ": " + displayValue));
                String string = wrappedValue;
                return string;
            }
            finally {
                this.unwrapSem.release();
            }
        }
        catch (InterruptedException e) {
            throw new RuntimeException("unable to enter critical section when unwrapping " + type, e);
        }
    }

    private VaultKubernetesAuthAuth loginKubernetes() {
        String jwt = new String(this.read(this.serverConfig.authentication.kubernetes.jwtTokenPath), StandardCharsets.UTF_8);
        log.debug((Object)("authenticate with jwt at: " + this.serverConfig.authentication.kubernetes.jwtTokenPath + " => " + this.serverConfig.logConfidentialityLevel.maskWithTolerance(jwt, LogConfidentialityLevel.LOW)));
        return (VaultKubernetesAuthAuth)this.vaultClient.loginKubernetes((String)this.serverConfig.authentication.kubernetes.role.get(), (String)jwt).auth;
    }

    private byte[] read(String path) {
        try {
            return Files.readAllBytes(Paths.get(path, new String[0]));
        }
        catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    private void sanityCheck(VaultToken vaultToken) {
        vaultToken.leaseDurationSanityCheck("auth", this.serverConfig.renewGracePeriod);
    }
}

