/*
 * Decompiled with CFR 0.152.
 */
package io.r2dbc.h2;

import io.r2dbc.h2.H2Batch;
import io.r2dbc.h2.H2ConnectionMetadata;
import io.r2dbc.h2.H2DatabaseExceptionFactory;
import io.r2dbc.h2.H2Statement;
import io.r2dbc.h2.client.Client;
import io.r2dbc.h2.codecs.Codecs;
import io.r2dbc.h2.util.Assert;
import io.r2dbc.spi.Connection;
import io.r2dbc.spi.IsolationLevel;
import io.r2dbc.spi.Option;
import io.r2dbc.spi.TransactionDefinition;
import io.r2dbc.spi.ValidationDepth;
import java.time.Duration;
import java.util.Collections;
import java.util.Iterator;
import java.util.function.Function;
import org.h2.command.CommandInterface;
import org.h2.engine.Constants;
import org.h2.message.DbException;
import org.h2.result.ResultInterface;
import org.reactivestreams.Publisher;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.Logger;
import reactor.util.Loggers;

public final class H2Connection
implements Connection {
    private final Logger logger = Loggers.getLogger(this.getClass());
    private final Client client;
    private final Codecs codecs;
    private final H2ConnectionMetadata metadata;
    private IsolationLevel isolationLevel;

    H2Connection(Client client, Codecs codecs) {
        this.client = Assert.requireNonNull(client, "client must not be null");
        this.codecs = Assert.requireNonNull(codecs, "codecs must not be null");
        this.isolationLevel = IsolationLevel.READ_COMMITTED;
        String version = Constants.VERSION;
        Iterator<CommandInterface> commands = client.prepareCommand("CALL H2VERSION()", Collections.emptyList());
        if (commands.hasNext()) {
            CommandInterface command = commands.next();
            ResultInterface query = client.query(command);
            query.next();
            version = query.currentRow()[0].getString();
            query.close();
        }
        this.metadata = new H2ConnectionMetadata(version);
    }

    public Mono<Void> beginTransaction() {
        return this.beginTransaction(EmptyTransactionDefinition.INSTANCE);
    }

    public Mono<Void> beginTransaction(TransactionDefinition definition) {
        return this.useTransactionStatus(inTransaction -> {
            if (!inTransaction.booleanValue()) {
                IsolationLevel isolationLevel = (IsolationLevel)definition.getAttribute(TransactionDefinition.ISOLATION_LEVEL);
                Boolean readOnly = (Boolean)definition.getAttribute(TransactionDefinition.READ_ONLY);
                Mono startTransaction = Mono.fromRunnable(() -> this.client.disableAutoCommit());
                if (isolationLevel != null) {
                    startTransaction = startTransaction.then(this.setTransactionIsolationLevel(isolationLevel));
                }
                if (readOnly != null) {
                    this.logger.warn(TransactionDefinition.READ_ONLY + " + isn't supported in H2 at the transaction level. You must set it on conenction URL. See http://www.h2database.com/html/features.html#read_only");
                }
                return startTransaction;
            }
            this.logger.debug("Skipping begin transaction because already in one");
            return Mono.empty();
        }).onErrorMap(DbException.class, H2DatabaseExceptionFactory::convert);
    }

    public Mono<Void> close() {
        return this.client.close();
    }

    public Mono<Void> commitTransaction() {
        return this.useTransactionStatus(inTransaction -> {
            if (inTransaction.booleanValue()) {
                this.client.execute("COMMIT");
                this.client.enableAutoCommit();
            } else {
                this.logger.debug("Skipping commit transaction because no transaction in progress.");
            }
            return Mono.empty();
        }).onErrorMap(DbException.class, H2DatabaseExceptionFactory::convert);
    }

    public H2Batch createBatch() {
        return new H2Batch(this.client, this.codecs);
    }

    public Mono<Void> createSavepoint(String name) {
        Assert.requireNonNull(name, "name must not be null");
        return this.beginTransaction().then(Mono.fromRunnable(() -> this.client.execute(String.format("SAVEPOINT %s", name)))).onErrorMap(DbException.class, H2DatabaseExceptionFactory::convert);
    }

    public H2Statement createStatement(String sql) {
        return new H2Statement(this.client, this.codecs, sql);
    }

    public IsolationLevel getTransactionIsolationLevel() {
        return this.isolationLevel;
    }

    public H2ConnectionMetadata getMetadata() {
        return this.metadata;
    }

    public boolean isAutoCommit() {
        return this.client.getSession().getAutoCommit();
    }

    public Mono<Void> releaseSavepoint(String name) {
        Assert.requireNonNull(name, "name must not be null");
        return this.useTransactionStatus(inTransaction -> {
            if (inTransaction.booleanValue()) {
                this.client.execute(String.format("RELEASE SAVEPOINT %s", name));
            } else {
                this.logger.debug("Skipping release savepoint because no transaction in progress.");
            }
            return Mono.empty();
        }).onErrorMap(DbException.class, H2DatabaseExceptionFactory::convert);
    }

    public Mono<Void> rollbackTransaction() {
        return this.useTransactionStatus(inTransaction -> {
            if (inTransaction.booleanValue()) {
                this.client.execute("ROLLBACK");
                this.client.enableAutoCommit();
            } else {
                this.logger.debug("Skipping rollback because no transaction in progress.");
            }
            return Mono.empty();
        }).onErrorMap(DbException.class, H2DatabaseExceptionFactory::convert);
    }

    public Mono<Void> rollbackTransactionToSavepoint(String name) {
        Assert.requireNonNull(name, "name must not be null");
        return this.useTransactionStatus(inTransaction -> {
            if (inTransaction.booleanValue()) {
                this.client.execute(String.format("ROLLBACK TO SAVEPOINT %s", name));
            } else {
                this.logger.debug("Skipping rollback to savepoint because no transaction in progress.");
            }
            return Mono.empty();
        }).onErrorMap(DbException.class, H2DatabaseExceptionFactory::convert);
    }

    public Mono<Void> setAutoCommit(boolean autoCommit) {
        return Mono.fromRunnable(() -> this.client.getSession().setAutoCommit(autoCommit));
    }

    public Mono<Void> setLockWaitTimeout(Duration duration) {
        return Mono.empty();
    }

    public Mono<Void> setStatementTimeout(Duration duration) {
        return Mono.empty();
    }

    public Mono<Void> setTransactionIsolationLevel(IsolationLevel isolationLevel) {
        Assert.requireNonNull(isolationLevel, "isolationLevel must not be null");
        return Mono.fromRunnable(() -> this.client.execute(H2Connection.getTransactionIsolationLevelQuery(isolationLevel))).map(aVoid -> {
            this.isolationLevel = isolationLevel;
            return aVoid;
        }).onErrorMap(DbException.class, H2DatabaseExceptionFactory::convert);
    }

    public Mono<Boolean> validate(ValidationDepth depth) {
        Assert.requireNonNull(depth, "depth must not be null");
        return Mono.fromCallable(() -> {
            if (this.client.getSession().isClosed()) {
                return false;
            }
            this.client.query(this.client.prepareCommand("SELECT CURRENT_TIMESTAMP", Collections.emptyList()).next());
            return true;
        }).switchIfEmpty(Mono.just((Object)false));
    }

    private static String getTransactionIsolationLevelQuery(IsolationLevel isolationLevel) {
        if (IsolationLevel.READ_COMMITTED == isolationLevel) {
            return String.format("SET LOCK_MODE %d", 3);
        }
        if (IsolationLevel.READ_UNCOMMITTED == isolationLevel) {
            return String.format("SET LOCK_MODE %d", 0);
        }
        if (IsolationLevel.REPEATABLE_READ == isolationLevel || IsolationLevel.SERIALIZABLE == isolationLevel) {
            return String.format("SET LOCK_MODE %d", 1);
        }
        throw new IllegalArgumentException(String.format("Invalid isolation level %s", isolationLevel));
    }

    private Mono<Void> useTransactionStatus(Function<Boolean, Publisher<?>> f) {
        return Flux.defer(() -> (Publisher)f.apply(this.client.inTransaction())).onErrorMap(DbException.class, H2DatabaseExceptionFactory::convert).then();
    }

    private static enum EmptyTransactionDefinition implements TransactionDefinition
    {
        INSTANCE;


        public <T> T getAttribute(Option<T> option) {
            return null;
        }
    }
}

