/*
 * Decompiled with CFR 0.152.
 */
package io.r2dbc.mssql;

import io.r2dbc.mssql.ConnectionOptions;
import io.r2dbc.mssql.ExceptionFactory;
import io.r2dbc.mssql.MssqlBatch;
import io.r2dbc.mssql.MssqlConnectionMetadata;
import io.r2dbc.mssql.MssqlResult;
import io.r2dbc.mssql.MssqlStatement;
import io.r2dbc.mssql.ParametrizedMssqlStatement;
import io.r2dbc.mssql.QueryMessageFlow;
import io.r2dbc.mssql.SimpleMssqlStatement;
import io.r2dbc.mssql.api.MssqlTransactionDefinition;
import io.r2dbc.mssql.client.Client;
import io.r2dbc.mssql.client.ConnectionContext;
import io.r2dbc.mssql.client.TransactionStatus;
import io.r2dbc.mssql.util.Assert;
import io.r2dbc.spi.Connection;
import io.r2dbc.spi.IsolationLevel;
import io.r2dbc.spi.Option;
import io.r2dbc.spi.TransactionDefinition;
import io.r2dbc.spi.ValidationDepth;
import java.time.Duration;
import java.util.function.Function;
import java.util.regex.Pattern;
import org.reactivestreams.Publisher;
import org.reactivestreams.Subscription;
import reactor.core.CoreSubscriber;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.util.Logger;
import reactor.util.Loggers;

public final class MssqlConnection
implements Connection {
    private static final Pattern IDENTIFIER_PATTERN = Pattern.compile("[\\d\\w_]{1,32}");
    private static final Pattern IDENTIFIER128_PATTERN = Pattern.compile("[\\d\\w_]{1,128}");
    private static final Logger logger = Loggers.getLogger(MssqlConnection.class);
    private final Client client;
    private final MssqlConnectionMetadata metadata;
    private final ConnectionContext context;
    private final ConnectionOptions connectionOptions;
    private final Flux<Integer> validationQuery;
    private volatile boolean autoCommit;
    private volatile IsolationLevel isolationLevel;
    private volatile IsolationLevel previousIsolationLevel;
    private volatile boolean resetLockWaitTime = false;

    MssqlConnection(Client client, MssqlConnectionMetadata connectionMetadata, ConnectionOptions connectionOptions) {
        this.client = Assert.requireNonNull(client, "Client must not be null");
        this.metadata = connectionMetadata;
        this.context = client.getContext();
        this.connectionOptions = Assert.requireNonNull(connectionOptions, "ConnectionOptions must not be null");
        TransactionStatus transactionStatus = client.getTransactionStatus();
        this.autoCommit = transactionStatus == TransactionStatus.AUTO_COMMIT;
        this.isolationLevel = IsolationLevel.READ_COMMITTED;
        this.validationQuery = new SimpleMssqlStatement(this.client, connectionOptions, "SELECT 1").fetchSize(0).execute().flatMap(MssqlResult::getRowsUpdated);
    }

    public Mono<Void> beginTransaction() {
        return this.beginTransaction(EmptyTransactionDefinition.INSTANCE);
    }

    public Mono<Void> beginTransaction(TransactionDefinition transactionDefinition) {
        return this.useTransactionStatus(tx -> {
            if (tx == TransactionStatus.STARTED) {
                logger.debug(this.context.getMessage("Skipping begin transaction because status is [{}]"), new Object[]{tx});
                return Mono.empty();
            }
            String name = (String)transactionDefinition.getAttribute(MssqlTransactionDefinition.NAME);
            String mark = (String)transactionDefinition.getAttribute(MssqlTransactionDefinition.MARK);
            IsolationLevel isolationLevel = (IsolationLevel)transactionDefinition.getAttribute(MssqlTransactionDefinition.ISOLATION_LEVEL);
            Duration lockWaitTime = (Duration)transactionDefinition.getAttribute(MssqlTransactionDefinition.LOCK_WAIT_TIMEOUT);
            StringBuilder builder = new StringBuilder();
            builder.append("BEGIN TRANSACTION");
            if (name != null) {
                String nameToUse = MssqlConnection.sanitize(name);
                Assert.isTrue(IDENTIFIER_PATTERN.matcher(nameToUse).matches(), "Transaction names must contain only characters and numbers and must not exceed 32 characters");
                builder.append(" ").append(nameToUse);
                if (mark != null) {
                    String markToUse = MssqlConnection.sanitize(mark);
                    Assert.isTrue(IDENTIFIER128_PATTERN.matcher(markToUse.substring(0, Math.min(128, markToUse.length()))).matches(), "Transaction names must contain only characters and numbers and must not exceed 128 characters");
                    builder.append(' ').append("WITH MARK '").append(markToUse).append("'");
                }
            }
            builder.append(';');
            if (isolationLevel != null) {
                builder.append(MssqlConnection.renderSetIsolationLevel(isolationLevel)).append(';');
            }
            if (lockWaitTime != null) {
                this.resetLockWaitTime = true;
                builder.append("SET LOCK_TIMEOUT ").append(lockWaitTime.isNegative() ? "-1" : Long.valueOf(lockWaitTime.toMillis())).append(';');
            }
            logger.debug(this.context.getMessage("Beginning transaction from status [{}]"), new Object[]{tx});
            return this.exchange(builder.toString()).doOnSuccess(unused -> {
                this.previousIsolationLevel = this.isolationLevel;
                if (isolationLevel != null) {
                    this.isolationLevel = isolationLevel;
                }
            });
        });
    }

    Mono<Void> cancel() {
        return this.client.attention();
    }

    public Mono<Void> close() {
        return this.client.close();
    }

    public Mono<Void> commitTransaction() {
        return this.useTransactionStatus(tx -> {
            if (tx != TransactionStatus.STARTED) {
                logger.debug(this.context.getMessage("Skipping commit transaction because status is [{}]"), new Object[]{tx});
                return Mono.empty();
            }
            logger.debug(this.context.getMessage("Committing transaction with status [{}]"), new Object[]{tx});
            return this.exchange("IF @@TRANCOUNT > 0 COMMIT TRANSACTION;" + this.cleanup()).doOnSuccess(v -> {
                if (this.previousIsolationLevel != null) {
                    this.isolationLevel = this.previousIsolationLevel;
                    this.previousIsolationLevel = null;
                }
                this.resetLockWaitTime = false;
            });
        });
    }

    private String cleanup() {
        String cleanupSql = "";
        if (this.previousIsolationLevel != null && this.previousIsolationLevel != this.isolationLevel) {
            cleanupSql = MssqlConnection.renderSetIsolationLevel(this.previousIsolationLevel) + ";";
        }
        if (this.resetLockWaitTime) {
            cleanupSql = cleanupSql + "SET LOCK_TIMEOUT -1;";
        }
        return cleanupSql;
    }

    public MssqlBatch createBatch() {
        return new MssqlBatch(this.client, this.connectionOptions);
    }

    public Mono<Void> createSavepoint(String name) {
        Assert.requireNonNull(name, "Savepoint name must not be null");
        String nameToUse = MssqlConnection.sanitize(name);
        Assert.isTrue(IDENTIFIER_PATTERN.matcher(nameToUse).matches(), "Save point names must contain only characters and numbers and must not exceed 32 characters");
        return this.useTransactionStatus(tx -> {
            logger.debug(this.context.getMessage("Creating savepoint [{}] for transaction with status [{}]"), new Object[]{nameToUse, tx});
            if (this.autoCommit) {
                logger.debug(this.context.getMessage("Setting auto-commit mode to [false]"));
            }
            return this.exchange(String.format("SET IMPLICIT_TRANSACTIONS ON; IF @@TRANCOUNT = 0 BEGIN BEGIN TRAN IF @@TRANCOUNT = 2 COMMIT TRAN END SAVE TRAN %s;", nameToUse)).doOnSuccess(ignore -> {
                this.autoCommit = false;
            });
        });
    }

    public MssqlStatement createStatement(String sql) {
        Assert.requireNonNull(sql, "SQL must not be null");
        logger.debug(this.context.getMessage("Creating statement for SQL: [{}]"), new Object[]{sql});
        if (ParametrizedMssqlStatement.supports(sql)) {
            return new ParametrizedMssqlStatement(this.client, this.connectionOptions, sql);
        }
        return new SimpleMssqlStatement(this.client, this.connectionOptions, sql);
    }

    public Mono<Void> releaseSavepoint(String name) {
        return Mono.empty();
    }

    public Mono<Void> rollbackTransaction() {
        return this.useTransactionStatus(tx -> {
            if (tx != TransactionStatus.STARTED && tx != TransactionStatus.EXPLICIT) {
                logger.debug(this.context.getMessage("Skipping rollback transaction because status is [{}]"), new Object[]{tx});
                return Mono.empty();
            }
            logger.debug(this.context.getMessage("Rolling back transaction with status [{}]"), new Object[]{tx});
            return this.exchange("IF @@TRANCOUNT > 0 ROLLBACK TRANSACTION;" + this.cleanup()).doOnSuccess(v -> {
                if (this.previousIsolationLevel != null) {
                    this.isolationLevel = this.previousIsolationLevel;
                    this.previousIsolationLevel = null;
                }
                this.resetLockWaitTime = false;
            }).doOnSuccess(v -> {
                if (this.previousIsolationLevel != null) {
                    this.isolationLevel = this.previousIsolationLevel;
                    this.previousIsolationLevel = null;
                }
                this.resetLockWaitTime = false;
            });
        });
    }

    public Mono<Void> rollbackTransactionToSavepoint(String name) {
        Assert.requireNonNull(name, "Savepoint name must not be null");
        String nameToUse = MssqlConnection.sanitize(name);
        Assert.isTrue(IDENTIFIER_PATTERN.matcher(nameToUse).matches(), "Save point names must contain only characters and numbers and must not exceed 32 characters");
        return this.useTransactionStatus(tx -> {
            if (tx != TransactionStatus.STARTED) {
                logger.debug(this.context.getMessage("Skipping rollback transaction to savepoint [{}] because status is [{}]"), new Object[]{nameToUse, tx});
                return Mono.empty();
            }
            logger.debug(this.context.getMessage("Rolling back transaction to savepoint [{}] with status [{}]"), new Object[]{nameToUse, tx});
            return this.exchange(String.format("ROLLBACK TRANSACTION %s", nameToUse));
        });
    }

    public boolean isAutoCommit() {
        return this.autoCommit && this.client.getTransactionStatus() != TransactionStatus.STARTED;
    }

    public Mono<Void> setAutoCommit(boolean autoCommit) {
        return Mono.defer(() -> {
            StringBuilder builder = new StringBuilder();
            logger.debug(this.context.getMessage("Setting auto-commit mode to [{}]"), new Object[]{autoCommit});
            if (this.autoCommit != autoCommit) {
                logger.debug(this.context.getMessage("Committing pending transactions"));
                builder.append("IF @@TRANCOUNT > 0 COMMIT TRAN;");
            }
            builder.append(autoCommit ? "SET IMPLICIT_TRANSACTIONS OFF;" : "SET IMPLICIT_TRANSACTIONS ON;");
            return this.exchange(builder.toString()).doOnSuccess(ignore -> {
                this.autoCommit = autoCommit;
            });
        });
    }

    public Mono<Void> setLockWaitTimeout(Duration timeout) {
        Assert.requireNonNull(timeout, "Timeout must not be null");
        return this.exchange("SET LOCK_TIMEOUT " + (timeout.isNegative() ? "-1" : "" + timeout.toMillis()));
    }

    public Mono<Void> setStatementTimeout(Duration timeout) {
        Assert.requireNonNull(timeout, "Timeout must not be null");
        return Mono.fromRunnable(() -> this.connectionOptions.setStatementTimeout(timeout));
    }

    public MssqlConnectionMetadata getMetadata() {
        return this.metadata;
    }

    public IsolationLevel getTransactionIsolationLevel() {
        return this.isolationLevel;
    }

    public Mono<Void> setTransactionIsolationLevel(IsolationLevel isolationLevel) {
        Assert.requireNonNull(isolationLevel, "IsolationLevel must not be null");
        return this.exchange(MssqlConnection.renderSetIsolationLevel(isolationLevel)).doOnSuccess(ignore -> {
            this.isolationLevel = isolationLevel;
        });
    }

    public Mono<Boolean> validate(ValidationDepth depth) {
        if (depth == ValidationDepth.LOCAL) {
            return Mono.fromSupplier(this.client::isConnected);
        }
        return Mono.create(sink -> {
            if (!this.client.isConnected()) {
                sink.success((Object)false);
                return;
            }
            this.validationQuery.subscribe((CoreSubscriber)new CoreSubscriber<Integer>(){

                public void onSubscribe(Subscription s) {
                    s.request(Integer.MAX_VALUE);
                }

                public void onNext(Integer integer) {
                }

                public void onError(Throwable t) {
                    logger.debug("Validation failed", t);
                    sink.success((Object)false);
                }

                public void onComplete() {
                    sink.success((Object)true);
                }
            });
        });
    }

    private static String renderSetIsolationLevel(IsolationLevel isolationLevel) {
        return "SET TRANSACTION ISOLATION LEVEL " + isolationLevel.asSql();
    }

    private static String sanitize(String identifier) {
        return identifier.replace('-', '_').replace('.', '_');
    }

    private Mono<Void> exchange(String sql) {
        ExceptionFactory factory = ExceptionFactory.withSql(sql);
        return QueryMessageFlow.exchange(this.client, sql).handle(factory::handleErrorResponse).then();
    }

    private Mono<Void> useTransactionStatus(Function<TransactionStatus, Publisher<?>> function) {
        return Flux.defer(() -> (Publisher)function.apply(this.client.getTransactionStatus())).then();
    }

    static enum EmptyTransactionDefinition implements TransactionDefinition
    {
        INSTANCE;


        public <T> T getAttribute(Option<T> option) {
            return null;
        }
    }
}

