/*
 * Copyright 2018 Realm Inc.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.realm.transformer.build

import io.realm.transformer.BytecodeModifier
import io.realm.transformer.BytecodeModifier.Companion.isModelField
import io.realm.transformer.ProjectMetaData
import io.realm.transformer.ext.safeSubtypeOf
import io.realm.transformer.logger
import javassist.CtClass
import org.gradle.api.file.ConfigurableFileCollection
import org.gradle.api.file.RegularFile
import java.io.File
import java.nio.file.FileSystem

class FullBuild(
    metadata: ProjectMetaData,
    inputJars: List<RegularFile>,
    output: FileSystem,
    inputDirectories: ConfigurableFileCollection
) : BuildTemplate(
    metadata = metadata,
    allJars = inputJars,
    output = output,
    inputs = inputDirectories
) {
    private lateinit var allModelClasses: List<CtClass>

    override fun prepareOutputClasses() {
        outputClassNames = categorizeClassNames()
        logger.debug("Full build. Number of files being processed: ${outputClassNames.size}.")
    }

    override fun findModelClasses(classNames: Set<String>): List<CtClass> {
        val realmObjectProxyInterface: CtClass = classPool.get("io.realm.internal.RealmObjectProxy")

        // For full builds, we are currently finding model classes by assuming that only
        // the annotation processor is generating files ending with `RealmProxy`. This is
        // a lot faster as we only need to compare the name of the type before we load
        // the CtClass.
        // Find the model classes
        return classNames
            // Quick and loose filter where we assume that classes ending with RealmProxy are
            // a Realm model proxy class generated by the annotation processor. This can
            // produce false positives: https://github.com/realm/realm-java/issues/3709
            .filter { it.endsWith("RealmProxy") }
            .map { classPool.getCtClass(it) }
            .filter { clazz ->
                // Verify the file is in fact a proxy class, in which case the super
                // class is always present and is the real model class.
                clazz.safeSubtypeOf(realmObjectProxyInterface)
            }
            .map { it.superclass }
    }

    // Full builds must categorize all classes.
    override fun File.shouldCategorize(): Boolean = true

    override fun filterForModelClasses(classNames: Set<String>, extraClassNames: Set<String>) {
        val allClassNames: Set<String> = merge(classNames, extraClassNames)

        allModelClasses = findModelClasses(allClassNames)

        outputModelClasses = allModelClasses.filter {
            outputClassNames.contains(it.name)
        }
    }

    override fun transformDirectAccessToModelFields() {
        // Populate a list of the fields that need to be managed with bytecode manipulation
        val allManagedFields = allModelClasses.flatMap { ctClass ->
            ctClass.declaredFields.filter { ctField ->
                ctField.isModelField()
            }
        }

        logger.debug("Managed Fields: ${allManagedFields.joinToString(",") { it.name }}")

        // Use accessors instead of direct field access
        outputClassNames.forEach {
            logger.debug("Modifying accessors in class: $it")
            try {
                val ctClass: CtClass = classPool.getCtClass(it)
                if (ctClass.isFrozen) {
                    ctClass.defrost()
                }
                BytecodeModifier.useRealmAccessors(classPool, ctClass, allManagedFields)
                processedClasses[it] = ctClass
            } catch (e: Exception) {
                throw RuntimeException("Failed to transform $it.", e)
            }
        }
    }

    private fun merge(set1: Set<String>, set2: Set<String>): Set<String>  {
        val merged: MutableSet<String> = hashSetOf()
        merged.addAll(set1)
        merged.addAll(set2)
        return merged
    }
}
