package io.relayr.java;

import com.google.common.base.Strings;

import dagger.ObjectGraph;
import retrofit.RestAdapter;
import retrofit.RestAdapter.LogLevel;

import static retrofit.RestAdapter.LogLevel.BASIC;
import static retrofit.RestAdapter.LogLevel.NONE;

public class RelayrJavaApp {

    private static final String API_DEFAULT_DEV = "https://dev-api.relayr.io";
    private static final String API_DEFAULT_PROD = "https://api.relayr.io";
    private static final String API_DEFAULT_MQTT_DEV = "dev-mqtt.relayr.io";
    private static final String API_DEFAULT_MQTT_PROD = "mqtt.relayr.io";

    private static final RestAdapter.LogLevel PRODUCTION_LEVEL = NONE;
    private static final RestAdapter.LogLevel DEVELOPMENT_LEVEL = BASIC;
    private static final String USER_AGENT = "JavaSdk v1.2.1";

    protected static boolean sProduction = true;

    protected static String mainApi = API_DEFAULT_PROD;
    protected static String mqttApi = API_DEFAULT_MQTT_PROD;

    protected static String mToken;
    protected static String mUserAgent;
    protected static RelayrJavaApp sApp;
    protected static ObjectGraph sObjectGraph;
    protected static LogLevel sLogLevel;
    protected static boolean sCacheModels;

    protected RelayrJavaApp(boolean mockMode) {
        sApp = this;
        buildObjectGraphAndInject(mockMode);
    }

    /**
     * Condition (sApp == null || mockMode) is used when Relayr app is already initialized
     * but you need to recreate it with another set of Dagger modules (e.g. while testing)
     * @param mockMode true for debug mode and tests
     */
    private static void init(String token, boolean mockMode) {
        reset();
        mToken = token;
        if (sApp == null || mockMode) {
            synchronized (new Object()) {
                if (sApp == null || mockMode) new RelayrJavaApp(mockMode);
            }
        }
    }

    /**
     * Initializes Relayr SDK
     * @param token       Relayr's user token
     * @param mockMode    true for debug mode and tests
     * @param production  if true production API is used, if false it uses development environment
     * @param cacheModels true to load and cache all the device models
     * @param level       defines log level for all API calls - {@link LogLevel#NONE} by default for production, {@link LogLevel#BASIC} for development
     */
    public static void init(String token, boolean mockMode, boolean production, boolean cacheModels, LogLevel level) {
        init(token, mockMode, production, cacheModels, level, USER_AGENT);
    }

    /**
     * Initializes Relayr SDK
     * @param token       Relayr's user token
     * @param mockMode    true for debug mode and tests
     * @param production  if true production API is used, if false it uses development environment
     * @param cacheModels true to load and cache all the device models
     * @param level       defines log level for all API calls - {@link LogLevel#NONE} by default for production, {@link LogLevel#BASIC} for development
     * @param userAgent   agent identificator
     */
    public static void init(String token, boolean mockMode, boolean production, boolean cacheModels, LogLevel level, String userAgent) {
        init(token, mockMode, production, cacheModels, level, userAgent,
                production ? API_DEFAULT_PROD : API_DEFAULT_DEV,
                production ? API_DEFAULT_MQTT_PROD : API_DEFAULT_MQTT_DEV);
    }

    /**
     * Initializes Relayr SDK
     * @param token       -  Relayr's user token
     * @param mock        - true for debug mode and tests
     * @param production  - if true production API is used, if false it uses development environment
     * @param cacheModels - true to load and cache all the device models
     * @param level       -  defines log level for all API calls - {@link LogLevel#NONE} by default for production, {@link LogLevel#BASIC} for development
     * @param userAgent   -  agent identificator
     * @param mainApiUrl  - relayr's main api url (default {@link #API_DEFAULT_PROD})
     * @param mqttApiUrl  - relayr's mqtt api url (default {@link #API_DEFAULT_MQTT_PROD})
     */
    public static void init(String token, boolean mock, boolean production, boolean cacheModels, LogLevel level, String userAgent,
                            String mainApiUrl, String mqttApiUrl) {
        sProduction = production;
        mUserAgent = Strings.isNullOrEmpty(userAgent) ? USER_AGENT : userAgent;

        setLogLevel(level);
        setModelsCache(cacheModels);

        mainApi = mainApiUrl;
        mqttApi = mqttApiUrl;

        init(token, mock);
    }

    protected static void buildObjectGraphAndInject(boolean mockMode) {
        sObjectGraph = mockMode ? ObjectGraph.create(DebugModules.list()) : ObjectGraph.create(Modules.list());
        sObjectGraph.injectStatics();
        sObjectGraph.inject(sApp);
    }

    protected static void reset() {
        sApp = null;
        mToken = null;
        sObjectGraph = null;
    }

    public static void inject(Object o) {
        sObjectGraph.inject(o);
    }

    public static String getToken() {
        return mToken;
    }

    public static void clearToken() {
        mToken = null;
    }

    public static String getUserAgent() {
        return mUserAgent;
    }

    public static void setToken(String token) {
        mToken = token;
    }

    public static void setLogLevel(LogLevel logLevel) {
        if (logLevel == null) sLogLevel = sProduction ? PRODUCTION_LEVEL : DEVELOPMENT_LEVEL;
        else sLogLevel = logLevel;
    }

    public static LogLevel getLogLevel() {
        return sLogLevel;
    }

    public static void setModelsCache(boolean caching) {sCacheModels = caching;}

    public static boolean isCachingModels() {return sCacheModels;}

    public static void setProduction(boolean production) {
        sProduction = production;
    }

    public static boolean isProduction() {
        return sProduction;
    }

    public static String getMainApiPoint() {return mainApi;}

    public static String getMqttApiPoint() {return mqttApi;}
}
