package io.relayr.java.api;

import java.util.List;

import io.relayr.java.model.CreateDevice;
import io.relayr.java.model.Device;
import io.relayr.java.model.action.Command;
import io.relayr.java.model.action.Configuration;
import io.relayr.java.model.device.laststate.LastCommands;
import io.relayr.java.model.device.laststate.LastConfigurations;
import io.relayr.java.model.device.laststate.LastReadings;
import io.relayr.java.model.json.JsonListResponse;
import io.relayr.java.model.models.DeviceModel;
import io.relayr.java.model.models.schema.ValueSchema;
import io.relayr.java.model.models.transport.DeviceCommand;
import io.relayr.java.model.models.transport.DeviceConfiguration;
import io.relayr.java.model.models.transport.DeviceReading;
import io.relayr.java.model.state.State;
import io.relayr.java.model.state.StateCommands;
import io.relayr.java.model.state.StateConfigurations;
import io.relayr.java.model.state.StateMetadata;
import io.relayr.java.model.state.StateReadings;
import retrofit.client.Response;
import retrofit.http.Body;
import retrofit.http.DELETE;
import retrofit.http.GET;
import retrofit.http.PATCH;
import retrofit.http.POST;
import retrofit.http.Path;
import retrofit.http.Query;
import rx.Observable;

/** This class incorporates a wrapped version of the relayr device API calls. */
public interface DeviceApi {

    /**
     * Creates device on the platform.
     * @param deviceId of device to fetch
     * @return created device
     */
    @GET("/devices/{id}") Observable<Device> getDevice(@Path("id") String deviceId);

    /**
     * Returns all user devices. Owned and shared by other users.
     * @param pageNumber        - the page number, starting with 0
     * @param pageSize          - how many items are returned per page
     * @param deviceIds         - a comma separated list of device Ids
     * @param modelId           - a model id used by the devices or the keyword 'any' for all devices that have a model
     * @param firmwareVersion   - the firmware version of the searched devices
     * @param deviceName        - a partial name of the devices
     * @param deviceDescription - a partial description of the devices
     */
    @GET("/devices")
    Observable<JsonListResponse<Device>> getAllDevices(@Query("page_number") Integer pageNumber,
                                                       @Query("page_size") Integer pageSize,
                                                       @Query("device_ids") String deviceIds,
                                                       @Query("model_id") String modelId,
                                                       @Query("firmware_version") String firmwareVersion,
                                                       @Query("device_name") String deviceName,
                                                       @Query("device_description") String deviceDescription);

    /**
     * Creates device on the platform.
     * @param device to create
     * @return created device
     */
    @POST("/devices") Observable<Device> createDevice(@Body CreateDevice device);


    /**
     * Updates a device.
     * @param deviceId id of the device to update
     * @param device   updated device with the new details
     * @return an {@link Observable} to the updated Device
     */
    @PATCH("/devices/{id}") Observable<Device> updateDevice(@Path("id") String deviceId, @Body Device device);

    /**
     * Deletes a device from platform.
     * @param deviceId id of device to delete
     * @return 200 OK if successful
     */
    @DELETE("/devices/{id}") Observable<Void> deleteDevice(@Path("id") String deviceId);

    /**
     * Checks device permission
     * @param deviceId - UUID of the device to check
     * @param action   - 'read', 'write' or 'admin'
     * @return 204 - if allowed
     * 400 - Missing or invalid Authorisation Header or Token
     * 401 - Unauthorized
     * 403 - Forbidden
     * 404 - Device is not Found.
     */
    @GET("/devices/{id}/permissions/{action}") Observable<Response> checkDevicePermission(@Path("id") String deviceId,
                                                                                          @Path("action") String action);

    /**
     * A public device is a device which public attribute has been set to 'true' therefore
     * no authorization is required.
     * @param meaning When a meaning is specified, the request returns only
     *                the devices which readings match the meaning.
     * @return an {@link Observable} with a list of all public devices.
     */
    @GET("/devices/public") Observable<List<Device>> getPublicDevices(@Query("meaning") String meaning);

    /**
     * Send command defined by device models {@link DeviceCommand}
     * Before sending a command make sure to validate it by calling {@link Command#validate(ValueSchema)}
     */
    @POST("/devices/{id}/commands") Observable<Void> sendCommand(@Path("id") String deviceId, @Body Command command);

    /**
     * Sets device configuration defined by device models {@link DeviceConfiguration}
     * Before sending a command make sure to validate it by calling {@link Configuration#validate(ValueSchema)}
     */
    @POST("/devices/{id}/configurations") Observable<Void> sendConfiguration(@Path("id") String deviceId,
                                                                             @Body Configuration configuration);

    /**
     * Persists user defined object in device state {@link State} metadata
     * @param deviceId unique device identifier
     * @param key      unique key in metadata dictionary
     * @param metadata user defined object
     */
    @POST("/devices/{id}/metadata") Observable<Void> setMetadata(@Path("id") String deviceId,
                                                                 @Query("key") String key,
                                                                 @Body Object metadata);

    /**
     * Returns device's {@link State}.
     * Device state shows latest state of devices' readings, configuration, commands and metadata.
     * @param deviceId unique device identifier
     */
    @GET("/devices/{id}/state") Observable<State> getState(@Path("id") String deviceId);

    /**
     * Returns device's {@link StateMetadata}
     * @param deviceId unique device identifier
     * @param key      key to filter metadata dictionary
     */
    @GET("/devices/{id}/metadata") Observable<StateMetadata> getMetadata(@Path("id") String deviceId,
                                                                         @Query("key") String key);

    /**
     * Returns device's {@link StateMetadata}
     * @param deviceId unique device identifier
     * @param key      key to filter metadata dictionary
     */
    @DELETE("/devices/{id}/metadata") Observable<Void> deleteMetadata(@Path("id") String deviceId,
                                                                      @Query("key") String key);

    /**
     * Returns device's {@link StateReadings}
     * Path and meaning parameters are optional. If provided used as regular filter fields.
     * @param deviceId unique device identifier
     * @param path     defined by {@link DeviceReading#getPath()} in {@link DeviceModel}
     * @param meaning  defined by {@link DeviceReading#getMeaning()} in {@link DeviceModel}
     */
    @Deprecated
    @GET("/devices/{id}/readings") Observable<StateReadings> getReadings(@Path("id") String deviceId,
                                                                         @Query("path") String path,
                                                                         @Query("meaning") String meaning);

    /**
     * Returns device's {@link StateReadings}
     * Path and meaning parameters are optional. If provided used as regular filter fields.
     * @param deviceIds list of device Ids separated by commas
     * @param path      defined by {@link DeviceReading#getPath()} in {@link DeviceModel}
     * @param meaning   defined by {@link DeviceReading#getMeaning()} in {@link DeviceModel}
     */
    @GET("/devices/readings")
    Observable<JsonListResponse<LastReadings>> getLastReadings(@Path("id") String deviceId,
                                                               @Query("deviceIds") String deviceIds,
                                                               @Query("path") String path,
                                                               @Query("meaning") String meaning);

    /**
     * Returns device's {@link StateCommands}
     * Path and name parameters are optional. If provided used as regular filter fields.
     * @param deviceId unique device identifier
     * @param path     defined by {@link DeviceCommand#getPath()} in {@link DeviceModel}
     * @param name     defined by {@link DeviceCommand#getName()} in {@link DeviceModel}
     */
    @Deprecated
    @GET("/devices/{id}/commands") Observable<StateCommands> getCommands(@Path("id") String deviceId,
                                                                         @Query("path") String path,
                                                                         @Query("name") String name);

    /**
     * Returns device's {@link StateCommands}
     * Path and name parameters are optional. If provided used as regular filter fields.
     * @param deviceIds list of device Ids separated by commas
     * @param path      defined by {@link DeviceCommand#getPath()} in {@link DeviceModel}
     * @param name      defined by {@link DeviceCommand#getName()} in {@link DeviceModel}
     */
    @GET("/devices/commands")
    Observable<JsonListResponse<LastCommands>> getLastCommands(@Path("id") String deviceId,
                                                               @Query("deviceIds") String deviceIds,
                                                               @Query("path") String path,
                                                               @Query("name") String name);

    /**
     * Returns device's {@link StateConfigurations}
     * Path and name parameters are optional. If provided used as regular filter fields.
     * @param deviceId unique device identifier
     * @param path     defined by {@link DeviceConfiguration#getPath()} in {@link DeviceModel}
     * @param name     defined by {@link DeviceConfiguration#getName()} in {@link DeviceModel}
     */
    @Deprecated
    @GET("/devices/{id}/configurations") Observable<StateConfigurations> getConfigurations(@Path("id") String deviceId,
                                                                                           @Query("path") String path,
                                                                                           @Query("name") String name);

    /**
     * Returns device's {@link StateConfigurations}
     * Path and name parameters are optional. If provided used as regular filter fields.
     * @param deviceId unique device identifier
     * @param path     defined by {@link DeviceConfiguration#getPath()} in {@link DeviceModel}
     * @param name     defined by {@link DeviceConfiguration#getName()} in {@link DeviceModel}
     */
    @GET("/devices/configurations")
    Observable<JsonListResponse<LastConfigurations>> getLastConfigurations(@Path("id") String deviceId,
                                                                           @Query("path") String path,
                                                                           @Query("name") String name);
}