package io.relayr.java.api;

import java.util.List;

import io.relayr.java.model.Publisher;
import io.relayr.java.model.User;
import io.relayr.java.model.projects.App;
import io.relayr.java.model.projects.ExtendedApp;
import retrofit.http.Body;
import retrofit.http.DELETE;
import retrofit.http.GET;
import retrofit.http.PATCH;
import retrofit.http.POST;
import retrofit.http.Path;
import rx.Observable;

public interface ProjectsApi {

    /** @return information about the {@link App} initiating the request. */
    @GET("/oauth2/app-info") Observable<App> getAppInfo();

    /** @return list of all {@link App} registered on relayr's platform */
    @GET("/apps") Observable<List<App>> getAllApps();

    /** @return created {@link ExtendedApp} */
    @POST("/apps") Observable<ExtendedApp> createApp(@Body ExtendedApp app);

    /** @return updated {@link ExtendedApp} */
    @PATCH("/apps/{id}") Observable<ExtendedApp> updateApp(@Path("id") String appId);

    /** @return empty {@link Observable} */
    @DELETE("/apps/{id}") Observable<Void> deleteApp(@Path("id") String appId);

    /**
     * @param appId - owned by user
     * @return {@link ExtendedApp} defined by appId
     */
    @GET("/apps/{id}") Observable<ExtendedApp> getApp(@Path("id") String appId);

    /**
     * For more information about {@link Publisher} check {@link PublishersApi}
     * @param publisherId - check {@link PublishersApi#getUserPublishers(String)}
     * @return list of all {@link App} owned by {@link Publisher}.
     */
    @GET("/publishers/{id}/apps")
    Observable<List<App>> getPublisherApps(@Path("id") String publisherId);

    /**
     * For more information about {@link Publisher} check {@link PublishersApi}
     * @param publisherId - check {@link PublishersApi#getUserPublishers(String)}
     * @return list of all {@link ExtendedApp} owned by {@link Publisher}.
     */
    @GET("/publishers/{id}/apps/extended")
    Observable<List<ExtendedApp>> getPublisherExtendedApps(@Path("id") String publisherId);

    /**
     * @param userId
     * @return list of all {@link ExtendedApp} owned by {@link User} defined with userId.
     */
    @GET("/users/{id}/apps") Observable<List<ExtendedApp>> getUserApps(@Path("id") String userId);

    /**
     * @param userId
     * @param appId
     * @param app
     * @return updated {@link ExtendedApp} defined by appId owned by {@link User} defined by userId
     */
    @POST("/users/{id}/apps/{appId}")
    Observable<ExtendedApp> updateUserApp(@Path("id") String userId, @Path("appId") String appId, @Body ExtendedApp app);

    /**
     * @param userId
     * @param appId
     * @return {@link ExtendedApp} defined by appId owned by {@link User} defined by userId
     */
    @GET("/users/{id}/apps/{appId}")
    Observable<ExtendedApp> getUserApp(@Path("id") String userId, @Path("appId") String appId);

    /**
     * Deletes {@link ExtendedApp} defined by appId owned by {@link User} defined by userId
     * @param userId
     * @param appId
     * @return empty {@link Observable}
     */
    @DELETE("/users/{id}/apps/{appId}")
    Observable<Void> deleteUserApp(@Path("id") String userId, @Path("appId") String appId);
}
