package io.relayr.java.api;

import java.util.List;

import io.relayr.java.model.Transmitter;
import io.relayr.java.model.TransmitterDevice;
import retrofit.http.Body;
import retrofit.http.DELETE;
import retrofit.http.GET;
import retrofit.http.PATCH;
import retrofit.http.POST;
import retrofit.http.Path;
import rx.Observable;

/** This class incorporates a wrapped version of the relayr API calls. */
public interface RelayrApi {

    /**
     * Registers the transmitter
     * @param transmitter transmitter object to register
     * @return an {@link Observable} to the registered Transmitter
     */
    @POST("/transmitters")
    Observable<Transmitter> registerTransmitter(@Body Transmitter transmitter);

    /**
     * @param transmitterId
     * @return an {@link Observable} of a specific transmitter
     */
    @GET("/transmitters/{id}")
    Observable<Transmitter> getTransmitter(@Path("id") String transmitterId);

    /**
     * Updates a transmitter.
     * @param transmitterId id of the transmitter to update
     * @param transmitter   updated transmitter with the new details
     * @return an {@link Observable} to the updated Transmitter
     */
    @PATCH("/transmitters/{id}")
    Observable<Transmitter> updateTransmitter(@Path("id") String transmitterId, @Body Transmitter transmitter);

    /**
     * Deletes a transmitter and all of its components (Transmitter and Devices)
     * @param transmitterId id of the transmitter (the Master Module)
     * @return an empty {@link Observable}
     */
    @DELETE("/transmitters/{id}")
    Observable<Void> deleteTransmitter(@Path("id") String transmitterId);

    /**
     * Deletes a WunderBar and all of its components (Transmitter and Devices)
     * @param transmitterId id of the transmitter (the Master Module)
     * @return an empty {@link Observable}
     */
    @DELETE("/wunderbars/{id}") Observable<Void> deleteWunderBar(@Path("id") String transmitterId);

    /**
     * @param transmitter the id of the transmitter to get the devices from
     * @return an {@link Observable} with a list of devices that belong to the specific
     * transmitter.
     */
    @GET("/transmitters/{id}/devices")
    Observable<List<TransmitterDevice>> getTransmitterDevices(@Path("id") String transmitter);
}
