package io.relayr.java.api.helpers;

import com.google.common.base.Strings;

import java.util.concurrent.TimeUnit;

import io.relayr.java.RelayrJavaSdk;
import io.relayr.java.api.helpers.Aggregates.Aggregate;
import io.relayr.java.api.helpers.Aggregates.Interval;
import io.relayr.java.api.services.AggregatedDataService;
import io.relayr.java.helper.Validator;
import io.relayr.java.model.aggregated.AggregatedData;
import rx.Observable;

import static io.relayr.java.helper.observer.TimeZoneUtil.getDate;

/**
 * Wraps History API and provides additional functionalities like
 * 1. fetching latest data by minute, hour or day
 * 2. fetching range of history points
 */
public class AggregatedDataHelper {

    private final String deviceId;

    /**
     * @param deviceId - device identificator
     * @return AggregatedDataHelper for specified device
     */
    public static AggregatedDataHelper init(String deviceId) {
        Validator.requireValidUuid(deviceId, "Device ID can not be NULL");
        Validator.requireNotNullNotEmpty(deviceId, "Device ID can not be NULL");
        return new AggregatedDataHelper(deviceId);
    }

    private AggregatedDataHelper(String deviceId) {this.deviceId = deviceId;}

    /**
     * Returns all aggregated readings for deviceId starting 'time' units ago.
     * For example getLatest(5, TimeUnit.MINUTES) returns all history results in past 5 minutes.
     * Valid time units are: {@link TimeUnit#SECONDS},{@link TimeUnit#MINUTES}, {@link TimeUnit#HOURS}, {@link TimeUnit#DAYS}
     * @param time - number of time units
     * @param unit - time unit
     */
    public Observable<AggregatedData> getLatest(int time, TimeUnit unit) throws IllegalArgumentException {
        return getLatest(null, null, time, unit);
    }

    /**
     * q
     * Return latest results starting 'time' units ago.
     * For example getLatest(5, TimeUnit.MINUTES) returns all history results in past 5 minutes.
     * Valid time units are: {@link TimeUnit#SECONDS},{@link TimeUnit#MINUTES}, {@link TimeUnit#HOURS}, {@link TimeUnit#DAYS}
     * Warning: Sample is not supported with complex data, but only with Integer, Number and Boolean values
     * @param time    - number of time units
     * @param unit    - time unit
     * @param meaning device reading meaning {@link io.relayr.java.model.action.Reading#meaning} - optional
     * @param path    device reading path {@link io.relayr.java.model.action.Reading#path} - optional
     */
    public Observable<AggregatedData> getLatest(
            String path, String meaning,
            int time, TimeUnit unit) throws IllegalArgumentException {

        Validator.requireNotNull(unit, "Time unit can't be NULL");

        final Interval interval = unit == TimeUnit.DAYS ? Interval.i1h :
                unit == TimeUnit.HOURS ? Interval.i5m :
                        unit == TimeUnit.MINUTES ? Interval.i1m :
                                Interval.i10s;

        return getLatest(path, meaning, time, unit,
                interval, new Aggregate[]{Aggregate.AVG});
    }

    /**
     * Return latest results starting 'time' units ago.
     * For example getLatest(5, TimeUnit.MINUTES) returns all history results in past 5 minutes.
     * Valid time units are: {@link TimeUnit#SECONDS},{@link TimeUnit#MINUTES}, {@link TimeUnit#HOURS}, {@link TimeUnit#DAYS}
     * Warning: Sample is not supported with complex data, but only with Integer, Number and Boolean values
     * @param path       - device reading path {@link io.relayr.java.model.action.Reading#path} - optional
     * @param meaning    - device reading meaning {@link io.relayr.java.model.action.Reading#meaning} - optional
     * @param time       - number of time units
     * @param unit       - time unit
     * @param interval   - {@link Interval}
     * @param aggregates - {@link Aggregate}
     */
    public Observable<AggregatedData> getLatest(
            String path, String meaning,
            int time, TimeUnit unit,
            Interval interval, Aggregate[] aggregates) throws IllegalArgumentException {

        Validator.requireNotNull(unit, "Time unit can't be NULL");
        Validator.requireAboveZero(time, "Time can't be less or equal to 0.");
        Validator.requireNotNull(interval, "Interval can't be NULL");
        Validator.requireNotNullNotEmpty(aggregates, "Aggregates can't be NULL or empty. You need to select at least one.");

        if (unit == TimeUnit.NANOSECONDS || unit == TimeUnit.MILLISECONDS || unit == TimeUnit.MICROSECONDS)
            throw new IllegalArgumentException("TimeUnit " + unit.name() + " not supported.");

        return RelayrJavaSdk.getAggregatedDataService().getData(deviceId,
                Strings.emptyToNull(path), Strings.emptyToNull(meaning),
                getDate(System.currentTimeMillis() - unit.toMillis(time)),
                getDate(System.currentTimeMillis()),
                interval.getInterval(),
                Aggregates.toList(aggregates));
    }

    /**
     * Returns average readings aggregated in 5 minute interval for defined range
     * @param start - UTC timestamp in milliseconds
     * @param end   - UTC timestamp in milliseconds
     */
    public Observable<AggregatedData> getForRange(
            long start, long end) throws IllegalArgumentException {
        return getForRange(start, end, null);
    }

    /**
     * Returns average readings aggregated in 5 minute interval for defined range filtered by meaning
     * @param start   - UTC timestamp in milliseconds
     * @param end     - UTC timestamp in milliseconds
     * @param meaning device reading meaning {@link io.relayr.java.model.action.Reading#meaning} - optional
     */
    public Observable<AggregatedData> getForRange(
            long start, long end, String meaning) throws IllegalArgumentException {
        return getForRange(start, end, meaning, Interval.i5m);
    }

    /**
     * Returns average readings aggregated in specified interval for defined range filtered by meaning
     * @param start   - UTC timestamp in milliseconds
     * @param end     - UTC timestamp in milliseconds
     * @param meaning device reading meaning {@link io.relayr.java.model.action.Reading#meaning} - optional
     */
    public Observable<AggregatedData> getForRange(
            long start, long end, String meaning,
            Interval interval) throws IllegalArgumentException {
        return getForRange(start, end, meaning, null, interval, new Aggregate[]{Aggregate.AVG});
    }

    /**
     * Returns aggregated readings, filtered by meaning and path, for defined range and with defined interval
     * @param start   - UTC timestamp in milliseconds
     * @param end     - UTC timestamp in milliseconds
     * @param meaning device reading meaning {@link io.relayr.java.model.action.Reading#meaning} - optional
     * @param path    device reading path {@link io.relayr.java.model.action.Reading#path} - optional
     */
    public Observable<AggregatedData> getForRange(
            long start, long end,
            String meaning, String path,
            Interval interval, Aggregate[] aggregates) throws IllegalArgumentException {

        Validator.requireAboveZero(start, "Start timestamp can't be <= 0");
        Validator.requireAboveZero(end, "End timestamp can't be <= 0");

        if (start >= end) throw new IllegalArgumentException("Error: start ts >= end ts.");

        return RelayrJavaSdk.getAggregatedDataService().getData(
                deviceId, Strings.emptyToNull(path), Strings.emptyToNull(meaning),
                getDate(start), getDate(end),
                interval.getInterval(), Aggregates.toList(aggregates));
    }

    /** @return AggregatedDataService object. */
    public AggregatedDataService getApi() {return RelayrJavaSdk.getAggregatedDataService();}
}
