package io.relayr.java.api.helpers;

import com.google.common.base.Strings;

import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.TimeZone;

import io.relayr.java.RelayrJavaSdk;
import io.relayr.java.api.services.RawDataService;
import io.relayr.java.helper.Validator;
import io.relayr.java.helper.observer.TimeZoneUtil;
import io.relayr.java.model.json.JsonListResponse;
import io.relayr.java.model.raw.RawReading;
import rx.Observable;

import static io.relayr.java.helper.observer.TimeZoneUtil.getDate;

/**
 * Wraps RawDataService API and provides additional functionalities like
 * 2. fetching range of raw data points
 */
public class RawDataHelper {

    private static final int MAX_RANGE = 60 * 60 * 1000; //1 hour
    private final String deviceId;

    /**
     * @param deviceId - device identificator
     * @return AggregatedDataHelper for specified device
     */
    public static RawDataHelper init(String deviceId) {
        Validator.requireValidUuid(deviceId, "Device ID can not be NULL");
        Validator.requireNotNullNotEmpty(deviceId, "Device ID can not be NULL");
        return new RawDataHelper(deviceId);
    }

    private RawDataHelper(String deviceId) {this.deviceId = deviceId;}


    /**
     * Returns raw readings for defined range (maximum 1 hour)
     * @param start - UTC timestamp in milliseconds
     * @param end   - UTC timestamp in milliseconds
     */
    public Observable<JsonListResponse<RawReading>> getForRange(
            long start, long end) throws IllegalArgumentException {

        return getForRange(start, end, null);
    }

    /**
     * Returns raw readings for defined range (maximum 1 hour) filtered by meaning
     * @param start   - UTC timestamp in milliseconds
     * @param end     - UTC timestamp in milliseconds
     * @param meaning device reading meaning {@link io.relayr.java.model.action.Reading#meaning} - optional
     */
    public Observable<JsonListResponse<RawReading>> getForRange(
            long start, long end, String meaning) throws IllegalArgumentException {

        return getForRange(start, end, meaning, null);
    }

    /**
     * Returns raw readings, filtered by meaning and path, for defined range (maximum 1 hour)
     * @param start   - UTC timestamp in milliseconds
     * @param end     - UTC timestamp in milliseconds
     * @param meaning device reading meaning {@link io.relayr.java.model.action.Reading#meaning} - optional
     * @param path    device reading path {@link io.relayr.java.model.action.Reading#path} - optional
     */
    public Observable<JsonListResponse<RawReading>> getForRange(
            long start, long end,
            String meaning, String path) throws IllegalArgumentException {

        Validator.requireAboveZero(start, "Start timestamp can't be <= 0");
        Validator.requireAboveZero(end, "End timestamp can't be <= 0");

        if (start >= end) throw new IllegalArgumentException("Error: start ts >= end ts.");
        if (end >= start + MAX_RANGE)
            throw new IllegalArgumentException("Error: Range can be of maximum 1 hour.");

        return RelayrJavaSdk.getRawDataService().getData(
                deviceId, Strings.emptyToNull(path), Strings.emptyToNull(meaning),
                getDate(start), getDate(end));
    }

    /** @return RawDataService API handler. */
    public RawDataService getApi() {return RelayrJavaSdk.getRawDataService();}
}
