package io.relayr.java.model;

import java.io.Serializable;
import java.util.List;

import io.relayr.java.RelayrJavaSdk;
import io.relayr.java.helper.Validator;
import io.relayr.java.model.account.Account;
import io.relayr.java.model.device.DevicePermission;
import io.relayr.java.model.groups.Group;
import io.relayr.java.model.json.JsonListResponse;
import retrofit.client.Response;
import rx.Observable;

/**
 * The first basic entity in the relayr platform is the user.
 * Every user registers with an email address,
 * a respective name and password and is assigned a unique userId.
 * A user can be both an application owner (a publisher) and an end user.
 * A user is required in order to add other entities to the relayr platform.
 */
public class User implements Serializable {

    private static final long serialVersionUID = 1L;

    private final String id;
    private String name;
    private String email;
    private String firstName;
    private String lastName;
    private String companyName;
    private String industryArea;

    public User(String id, String name, String email) {
        this.id = id;
        this.name = name;
        this.email = email;
    }

    public User(String id, String name, String email, String firstName, String lastName, String company, String industry) {
        this.id = id;
        this.name = name;
        this.email = email;
        this.firstName = firstName;
        this.lastName = lastName;
        this.companyName = company;
        this.industryArea = industry;
    }

    public String getId() {
        return id;
    }

    public String getEmail() {
        return email;
    }

    public String getName() {
        return name;
    }

    public String getFirstName() {
        return firstName;
    }

    public String getLastName() {
        return lastName;
    }

    public String getCompanyName() {
        return companyName;
    }

    public String getIndustryArea() {
        return industryArea;
    }

    public void setName(String name) {
        this.name = name;
    }

    /**
     * @return an {@link Observable} of a list of devices registered under a user.
     * @throws NullPointerException if deviceId is NULL
     */
    public Observable<Device> getDevice(String deviceId) {
        if (deviceId == null) throw new NullPointerException("Device ID can't be null");
        return RelayrJavaSdk.getDeviceApi().getDevice(deviceId);
    }

    /** @return an {@link Observable} of a list of devices owned by user. */
    public Observable<List<Device>> getDevices() {
        return RelayrJavaSdk.getUserApi().getDevices(id);
    }

    /** @return an {@link Observable} of a list of devices OWNED by the user and SHARED with user. */
    public Observable<JsonListResponse<Device>> getAllDevices() {
        return getAll(null, null, null, null, null, null, null);
    }

    /** @return an {@link Observable} of a list of devices owned and shared with user and filtered by provided parameters */
    public Observable<JsonListResponse<Device>> filterDevices(
            String[] deviceIds, String modelId,
            String name, String description) throws IllegalArgumentException, NullPointerException {
        return filterDevices(null, null, deviceIds, modelId, null, name, description);
    }

    /** @return an {@link Observable} of a list of devices owned and shared with user and filtered by provided parameters */
    public Observable<JsonListResponse<Device>> filterDevices(
            Integer pageNumber, Integer pageSize, String[] deviceIds,
            String modelId, String firmwareVersion,
            String name, String description) throws IllegalArgumentException, NullPointerException {

        String concat = null;
        if (deviceIds != null) {
            concat = "";
            for (String x : deviceIds) {
                Validator.requireNotNull(x, "Device ID is used as a filter and can't be null");
                Validator.requireValidUuid(x, "Device ID is used as a filter and can't be null");

                concat += x + ",";
            }
            concat = concat.substring(0, concat.length() - 1);
        }

        if (modelId != null) Validator.requireValidUuid(modelId);
        return getAll(pageNumber, pageSize, concat, modelId, firmwareVersion, name, description);
    }

    /**
     * @return 204 - if allowed
     * 400 - Missing or invalid Authorisation Header or Token
     * 401 - Unauthorized
     * 403 - Forbidden
     * 404 - Device is not Found.
     */
    public Observable<Response> checkDevicePermission(String deviceId, DevicePermission permission) throws IllegalArgumentException, NullPointerException {
        Validator.requireValidUuid(deviceId);
        Validator.requireNotNull(permission, "Permission can't be null");
        return RelayrJavaSdk.getDeviceApi().checkDevicePermission(deviceId, permission.name().toLowerCase());
    }

    /** @return an {@link Observable} with a list all Transmitters listed under a user. */
    public Observable<List<Transmitter>> getTransmitters() {
        return RelayrJavaSdk.getUserApi().getTransmitters(id);
    }

    /**
     * Returns a list of third party accounts that user connected with relayr platform.
     * @return an {@link Observable} with a list of {@link Account}
     */
    public Observable<List<Account>> getAccounts() {
        return RelayrJavaSdk.getUserApi().getAccounts(id);
    }

    /**
     * Returns error if account is not connected.
     * @return an empty {@link Observable}
     * @throws NullPointerException if parameter is NULL
     */
    public Observable<Void> isAccountConnected(String accountName) {
        if (accountName == null) throw new NullPointerException("Account name can't be null");
        return RelayrJavaSdk.getUserApi().isAccountConnected(id, accountName);
    }

    /**
     * Returns error if account is not successfully disconnected.
     * @return an empty {@link Observable}
     * @throws NullPointerException if parameter is NULL
     */
    public Observable<Void> disconnectAccount(String accountName) {
        if (accountName == null) throw new NullPointerException("Account name can't be null");
        return RelayrJavaSdk.getUserApi().disconnectAccount(id, accountName);
    }

    /**
     * Returns list of {@link Group} objects
     * @return an {@link Observable} list
     */
    public Observable<List<Group>> getGroups() {
        return RelayrJavaSdk.getUserApi().getGroups(id);
    }

    /**
     * Updates username.
     * @param username new username
     * @throws NullPointerException if username is NULL
     */
    public Observable<User> update(String username) {
        if (username == null) throw new NullPointerException("Username name can't be null");
        this.setName(username);
        return RelayrJavaSdk.getUserApi().updateUserDetails(this, id);
    }

    private Observable<JsonListResponse<Device>> getAll(Integer pageNumber,
                                                        Integer pageSize,
                                                        String deviceIds,
                                                        String modelId,
                                                        String firmwareVersion,
                                                        String name,
                                                        String description) {
        return RelayrJavaSdk.getDeviceApi()
                .getAllDevices(pageNumber, pageSize, deviceIds, modelId, firmwareVersion, name, description);
    }

    @Override
    public String toString() {
        return "User{" +
                "id='" + id + "\'" +
                ", name='" + name + "\'" +
                ", email='" + email + "\'" +
                "}";
    }
}
