package io.relayr.java.model.action;

import io.relayr.java.model.Device;
import io.relayr.java.model.models.DeviceModel;
import io.relayr.java.model.models.transport.DeviceReading;
import io.relayr.java.storage.DeviceModelCache;

/**
 * A reading is the information gathered by the device.
 * Details about reading are defined in {@link DeviceModel}
 */
public class Reading {

    /** Timestamp - when reading is received on the platform */
    public final long received;

    /** Timestamp - when reading is recorded on the device */
    public final long recorded;

    /** Timestamp - used when publishing through MQTT, later it's saved as #recorded timestamp */
    public final long ts;

    /**
     * Every device has a {@link DeviceModel} which defines all device readings.
     * Every {@link DeviceReading} has a {@link DeviceReading#meaning}.
     * Details for every reading can be found in {@link DeviceReading#getValueSchema()}
     * Details about device readings can be obtained using {@link DeviceModelCache#getModelById(String)}
     * where modelId is {@link Device#getModelId()}.
     */
    public final String meaning;

    /**
     * If device contains multiple levels of readings (more than one component)
     * they will be identified with the path.
     */
    public final String path;

    /** Reading value is determined by {@link DeviceReading#getValueSchema()} */
    public final Object value;

    /**
     * @param received - timestamp written by server
     * @param recorded - timestamp written by device that's publishing the reading object
     * @param meaning  - @see #meaning
     * @param path     - @see #path
     * @param value    - defined by {@link DeviceReading#getValueSchema()}
     */
    public Reading(long received, long recorded, String meaning, String path, Object value) {
        this.received = received;
        this.recorded = recorded;
        this.meaning = meaning;
        this.path = path;
        this.value = value;
        this.ts = recorded;
    }

    @Override public String toString() {
        return "Reading{" +
                "received=" + received +
                ", recorded=" + recorded +
                ", meaning='" + meaning + '\'' +
                ", path='" + path + '\'' +
                ", value=" + value +
                '}';
    }

    @Override public boolean equals(Object o) {
        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;

        Reading reading = (Reading) o;

        if (recorded != reading.recorded) return false;
        return meaning != null ? meaning.equals(reading.meaning) : reading.meaning == null;

    }

    @Override public int hashCode() {
        int result = (int) (recorded ^ (recorded >>> 32));
        result = 31 * result + (meaning != null ? meaning.hashCode() : 0);
        return result;
    }
}
