package io.relayr.java.model.raw;

import java.io.Serializable;

import io.relayr.java.helper.observer.TimeZoneUtil;
import io.relayr.java.model.Device;
import io.relayr.java.model.models.DeviceModel;
import io.relayr.java.model.models.transport.DeviceReading;
import io.relayr.java.storage.DeviceModelCache;

public class RawReading implements Serializable {

    /**
     * Every device has a {@link DeviceModel} which defines all device readings.
     * Every {@link DeviceReading} has a {@link DeviceReading#meaning}.
     * Details for every reading can be found in {@link DeviceReading#getValueSchema()}
     * Details about device readings can be obtained using {@link DeviceModelCache#getModelById(String)}
     * where modelId is {@link Device#getModelId()}.
     */
    public final String meaning;

    /**
     * If device contains multiple levels of readings (more than one component)
     * they will be identified with the path.
     */
    public final String path;

    /** Reading value is determined by {@link DeviceReading#getValueSchema()} */
    public final Object value;

    /** ISO timestamp - when reading is received on the platform */
    public final String received;

    /** ISO timestamp - when reading is recorded on the device */
    public final String recorded;

    public RawReading(String meaning, String path, String received, String recorded, Object value) {
        this.received = received;
        this.recorded = recorded;
        this.meaning = meaning;
        this.path = path;
        this.value = value;
    }

    /** UTC timestamp - when reading is recorded on the platform */
    public long getReceived() {
        return TimeZoneUtil.getUtcTs(received);
    }

    /** UTC timestamp - when reading is recorded on the device */
    public long getRecorded() {
        return TimeZoneUtil.getUtcTs(recorded);
    }

    @Override public String toString() {
        return "RawReading{" +
                "meaning='" + meaning + '\'' +
                ", path='" + path + '\'' +
                ", value=" + value +
                ", received='" + received + '\'' +
                ", recorded='" + recorded + '\'' +
                '}';
    }
}
