/*
 * Copyright 2016 requery.io
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.requery.android.sqlcipher;

import io.requery.android.sqlite.BaseConnection;
import net.sqlcipher.database.SQLiteConstraintException;
import net.sqlcipher.database.SQLiteDatabase;
import net.sqlcipher.database.SQLiteException;

import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLIntegrityConstraintViolationException;
import java.sql.Statement;

/**
 * {@link java.sql.Connection} implementation using SQLCipher SQLite Android API.
 *
 * @author Nikhil Purushe
 */
class SqlCipherConnection extends BaseConnection {

    private final SQLiteDatabase db;
    private final SqlCipherMetaData metaData;
    private boolean enteredTransaction;

    public SqlCipherConnection(SQLiteDatabase db) {
        if(db == null) {
            throw new IllegalArgumentException("null db");
        }
        this.db = db;
        autoCommit = true;
        metaData = new SqlCipherMetaData(this);
    }

    static void throwSQLException(SQLiteException exception) throws SQLException {
        if(exception instanceof SQLiteConstraintException) {
            throw new SQLIntegrityConstraintViolationException(exception);
        }
        throw new SQLException(exception);
    }

    SQLiteDatabase getDatabase() {
        return db;
    }

    @Override
    protected void ensureTransaction() {
        if (!autoCommit) {
            if (!db.inTransaction()) {
                db.beginTransaction();
                enteredTransaction = true;
            }
        }
    }

    @Override
    protected void execSQL(String sql) throws SQLException {
        try {
            db.execSQL(sql);
        } catch (SQLiteException e) {
            throwSQLException(e);
        }
    }

    @Override
    public void commit() throws SQLException {
        if (autoCommit) {
            throw new SQLException("commit called while in autoCommit mode");
        }
        if (db.inTransaction() && enteredTransaction) {
            try {
                db.setTransactionSuccessful();
            } catch (IllegalStateException e) {
                throw new SQLException(e);
            } finally {
                db.endTransaction();
                enteredTransaction = false;
            }
        }
    }

    @Override
    public Statement createStatement() throws SQLException {
        ensureTransaction();
        return new SqlCipherStatement(this);
    }

    @Override
    public Statement createStatement(int resultSetType, int resultSetConcurrency)
            throws SQLException {
        ensureTransaction();
        return createStatement(resultSetType,
                resultSetConcurrency, ResultSet.HOLD_CURSORS_OVER_COMMIT);
    }

    @Override
    public Statement createStatement(int resultSetType, int resultSetConcurrency,
                                     int resultSetHoldability) throws SQLException {
        if (resultSetConcurrency == ResultSet.CONCUR_UPDATABLE) {
            throw new SQLFeatureNotSupportedException("CONCUR_UPDATABLE not supported");
        }
        ensureTransaction();
        return new SqlCipherStatement(this);
    }

    @Override
    public DatabaseMetaData getMetaData() throws SQLException {
        return metaData;
    }

    @Override
    public boolean isClosed() throws SQLException {
        return !db.isOpen();
    }

    @Override
    public boolean isReadOnly() throws SQLException {
        return db.isReadOnly();
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int autoGeneratedKeys)
        throws SQLException {
        ensureTransaction();
        return new SqlCipherPreparedStatement(this, sql, autoGeneratedKeys);
    }

    @Override
    public PreparedStatement prepareStatement(String sql,
                                              int resultSetType,
                                              int resultSetConcurrency,
                                              int resultSetHoldability) throws SQLException {
        ensureTransaction();
        return new SqlCipherPreparedStatement(this, sql, Statement.NO_GENERATED_KEYS);
    }

    @Override
    public PreparedStatement prepareStatement(String sql, String[] columnNames)
            throws SQLException {
        ensureTransaction();
        return new SqlCipherPreparedStatement(this, sql, Statement.RETURN_GENERATED_KEYS);
    }

    @Override
    public void rollback() throws SQLException {
        if (autoCommit) {
            throw new SQLException("commit called while in autoCommit mode");
        }
        db.endTransaction();
    }
}
