/*
 * Copyright 2016 requery.io
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.requery.android.sqlite;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.net.URL;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.NClob;
import java.sql.ParameterMetaData;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSetMetaData;
import java.sql.RowId;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLXML;
import java.sql.Time;
import java.sql.Timestamp;
import java.sql.Types;
import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

/**
 * {@link PreparedStatement} implementation using Android's local SQLite database.
 */
public abstract class BasePreparedStatement extends BaseStatement implements PreparedStatement {

    private static final char[] hex = "0123456789ABCDEF".toCharArray();

    protected final String sql;
    protected final int autoGeneratedKeys;
    protected List<Object> bindings;

    protected BasePreparedStatement(BaseConnection connection, String sql, int autoGeneratedKeys)
        throws SQLException {
        super(connection);
        if (sql == null) {
            throw new SQLException("null sql");
        }
        if (autoGeneratedKeys != RETURN_GENERATED_KEYS) {
            bindings = new ArrayList<>(4);
        }
        this.sql = sql;
        this.autoGeneratedKeys = autoGeneratedKeys;
    }

    protected abstract void bindNullOrString(int index, Object value);

    protected abstract void bindLong(int index, long value);

    protected abstract void bindDouble(int index, double value);

    protected abstract void bindBlob(int index, byte[] value);

    protected String[] bindingsToArray() {
        String[] args = new String[bindings.size()];
        for (int i = 0; i < bindings.size() ; i++) {
            Object value = bindings.get(i);
            if (value != null) {
                args[i] = value.toString();
            }
        }
        return args;
    }

    protected static String byteToHexString(byte[] bytes) {
        StringBuilder sb = new StringBuilder(bytes.length * 2);
        for (byte b : bytes) {
            sb.append(hex[(b >> 4) & 0xF]);
            sb.append(hex[(b & 0xF)]);
        }
        return sb.toString();
    }

    @Override
    public void addBatch() throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public ResultSetMetaData getMetaData() throws SQLException {
        return null;
    }

    @Override
    public ParameterMetaData getParameterMetaData() throws SQLException {
        return null;
    }

    @Override
    public void setArray(int parameterIndex, Array theArray) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setAsciiStream(int parameterIndex, InputStream theInputStream, int length)
            throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setBigDecimal(int parameterIndex, BigDecimal theBigDecimal) throws SQLException {
        bindNullOrString(parameterIndex, theBigDecimal);
    }

    @Override
    public void setBinaryStream(int parameterIndex, InputStream theInputStream, int length)
            throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setBlob(int parameterIndex, Blob theBlob) throws SQLException {
        setBytes(parameterIndex, theBlob.getBytes(0, (int) theBlob.length()));
    }

    @Override
    public void setBoolean(int parameterIndex, boolean theBoolean) throws SQLException {
        long value = theBoolean ? 1 : 0;
        bindLong(parameterIndex, value);
    }

    @Override
    public void setByte(int parameterIndex, byte theByte) throws SQLException {
        bindLong(parameterIndex, (long) theByte);
    }

    @Override
    public void setBytes(int parameterIndex, byte[] theBytes) throws SQLException {
        bindBlob(parameterIndex, theBytes);
    }

    @Override
    public void setCharacterStream(int parameterIndex, Reader reader, int length) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setClob(int parameterIndex, Clob theClob) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setDate(int parameterIndex, Date theDate) throws SQLException {
        setDate(parameterIndex, theDate, null);
    }

    @Override
    public void setDate(int parameterIndex, Date theDate, Calendar cal) throws SQLException {
        if (theDate == null) {
            bindNullOrString(parameterIndex, null);
        } else {
            bindLong(parameterIndex, theDate.getTime());
        }
    }

    @Override
    public void setDouble(int parameterIndex, double theDouble) throws SQLException {
        bindDouble(parameterIndex, theDouble);
    }

    @Override
    public void setFloat(int parameterIndex, float theFloat) throws SQLException {
        bindDouble(parameterIndex, (double) theFloat);
    }

    @Override
    public void setInt(int parameterIndex, int theInt) throws SQLException {
        bindLong(parameterIndex, (long) theInt);
    }

    @Override
    public void setLong(int parameterIndex, long theLong) throws SQLException {
        bindLong(parameterIndex, theLong);
    }

    @Override
    public void setNull(int parameterIndex, int sqlType) throws SQLException {
        bindNullOrString(parameterIndex, null);
    }

    @Override
    public void setNull(int paramIndex, int sqlType, String typeName) throws SQLException {
        bindNullOrString(paramIndex, null);
    }

    @Override
    public void setObject(int parameterIndex, Object theObject) throws SQLException {
        if (theObject == null) {
            setNull(parameterIndex, Types.NULL);
        } else {
            if (theObject instanceof String) {
                setString(parameterIndex, theObject.toString());
            } else if (theObject instanceof Byte) {
                setByte(parameterIndex, (Byte) theObject);
            } else if (theObject instanceof Short) {
                setShort(parameterIndex, (Short) theObject);
            } else if (theObject instanceof Integer) {
                setInt(parameterIndex, (Integer) theObject);
            } else if (theObject instanceof Long) {
                setLong(parameterIndex, (Long) theObject);
            } else if (theObject instanceof Double) {
                setDouble(parameterIndex, (Double) theObject);
            } else if (theObject instanceof Float) {
                setFloat(parameterIndex, (Float) theObject);
            } else if (theObject instanceof Boolean) {
                setLong(parameterIndex, (Boolean) theObject ? 1 : 0);
            } else if (theObject instanceof byte[]) {
                setBytes(parameterIndex, (byte[]) theObject);
            } else if (theObject instanceof Date) {
                setDate(parameterIndex, (Date) theObject);
            } else if (theObject instanceof java.util.Date) {
                java.util.Date date = (java.util.Date) theObject;
                setDate(parameterIndex, new Date(date.getTime()));
            } else {
                throw new SQLException("unhandled type " + theObject.getClass().getCanonicalName());
            }
        }
    }

    @Override
    public void setObject(int parameterIndex, Object theObject, int targetSqlType)
            throws SQLException {
        if (theObject == null || targetSqlType == Types.NULL) {
            setNull(parameterIndex, Types.NULL);
            return;
        }
        switch (targetSqlType) {
            case Types.INTEGER:
            case Types.SMALLINT:
                if (theObject instanceof Integer) {
                    bindLong(parameterIndex, ((Integer) theObject).longValue());
                } else if (theObject instanceof Long) {
                    bindLong(parameterIndex, (Long) theObject);
                } else if (theObject instanceof Short) {
                    bindLong(parameterIndex, ((Short) theObject).longValue());
                }
                break;
            case Types.TINYINT:
                if (theObject instanceof Byte) {
                    bindLong(parameterIndex, ((Byte) theObject).longValue());
                }
                break;
            case Types.DOUBLE:
            case Types.FLOAT:
            case Types.REAL:
                if (theObject instanceof Double) {
                    setDouble(parameterIndex, (Double) theObject);
                } else if (theObject instanceof Float) {
                    setFloat(parameterIndex, (Float) theObject);
                }
                break;
            case Types.BLOB:
            case Types.BINARY:
            case Types.VARBINARY:
                setBytes(parameterIndex, (byte[]) theObject);
                break;
            case Types.BOOLEAN:
                Boolean value = (Boolean) theObject;
                setBoolean(parameterIndex, value);
                break;
            case Types.VARCHAR:
            case Types.NVARCHAR:
                String string = theObject instanceof String ?
                        (String) theObject : theObject.toString();
                setString(parameterIndex, string);
                break;
            case Types.DATE:
                if (theObject instanceof Date) {
                    Date date = (Date) theObject;
                    setLong(parameterIndex, date.getTime());
                } else if (theObject instanceof java.util.Date) {
                    java.util.Date date = (java.util.Date) theObject;
                    setLong(parameterIndex, date.getTime());
                }
                break;
            case Types.TIMESTAMP:
                if (theObject instanceof Timestamp) {
                    Timestamp timestamp = (Timestamp) theObject;
                    setLong(parameterIndex, timestamp.getTime());
                }
                break;
            case Types.BIGINT:
                if (theObject instanceof BigInteger) {
                    BigInteger bigInteger = (BigInteger) theObject;
                    bindNullOrString(parameterIndex, bigInteger.toString());
                }
                break;
            default:
                throw new SQLException("unhandled type " + targetSqlType);
        }
    }

    @Override
    public void setObject(int parameterIndex, Object theObject, int targetSqlType, int scale)
            throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setRef(int parameterIndex, Ref theRef) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setShort(int parameterIndex, short theShort) throws SQLException {
        bindLong(parameterIndex, (long) theShort);
    }

    @Override
    public void setString(int parameterIndex, String theString) throws SQLException {
        bindNullOrString(parameterIndex, theString);
    }

    @Override
    public void setTime(int parameterIndex, Time theTime) throws SQLException {
        if (theTime == null) {
            bindNullOrString(parameterIndex, null);
        } else {
            bindLong(parameterIndex, theTime.getTime());
        }
    }

    @Override
    public void setTime(int parameterIndex, Time theTime, Calendar cal) throws SQLException {
        if (theTime == null) {
            bindNullOrString(parameterIndex, null);
        } else {
            bindLong(parameterIndex, theTime.getTime());
        }
    }

    @Override
    public void setTimestamp(int parameterIndex, Timestamp theTimestamp) throws SQLException {
        if (theTimestamp == null) {
            bindNullOrString(parameterIndex, null);
        } else {
            bindLong(parameterIndex, theTimestamp.getTime());
        }
    }

    @Override
    public void setTimestamp(int parameterIndex, Timestamp theTimestamp, Calendar cal)
        throws SQLException {
        if (theTimestamp == null) {
            bindNullOrString(parameterIndex, null);
        } else {
            bindLong(parameterIndex, theTimestamp.getTime());
        }
    }

    @Override
    public void setUnicodeStream(int parameterIndex, InputStream theInputStream, int length)
            throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setURL(int parameterIndex, URL theURL) throws SQLException {
        bindNullOrString(parameterIndex, theURL);
    }

    @Override
    public void setRowId(int parameterIndex, RowId theRowId) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setNString(int parameterIndex, String theString) throws SQLException {
        bindNullOrString(parameterIndex, theString);
    }

    @Override
    public void setNCharacterStream(int parameterIndex, Reader reader, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setNClob(int parameterIndex, NClob value) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setClob(int parameterIndex, Reader reader, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setBlob(int parameterIndex, InputStream inputStream, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setNClob(int parameterIndex, Reader reader, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setSQLXML(int parameterIndex, SQLXML xmlObject) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setAsciiStream(int parameterIndex, InputStream inputStream, long length)
            throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setBinaryStream(int parameterIndex, InputStream inputStream, long length)
            throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setCharacterStream(int parameterIndex, Reader reader, long length) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setAsciiStream(int parameterIndex, InputStream inputStream) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setBinaryStream(int parameterIndex, InputStream inputStream) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setCharacterStream(int parameterIndex, Reader reader) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setNCharacterStream(int parameterIndex, Reader reader) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setClob(int parameterIndex, Reader reader) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setBlob(int parameterIndex, InputStream inputStream) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public void setNClob(int parameterIndex, Reader reader) throws SQLException {
        throw new SQLFeatureNotSupportedException();
    }

    @Override
    public String toString() {
        return sql;
    }
}
