/*
 * Copyright 2016 requery.io
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.requery.android.sqlitex;

import android.database.sqlite.SQLiteConstraintException;
import android.database.sqlite.SQLiteException;
import io.requery.android.database.sqlite.SQLiteDatabase;
import io.requery.android.sqlite.BaseConnection;

import java.sql.DatabaseMetaData;
import java.sql.PreparedStatement;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.sql.SQLFeatureNotSupportedException;
import java.sql.SQLIntegrityConstraintViolationException;
import java.sql.Statement;

class SqlitexConnection extends BaseConnection {

    private final SQLiteDatabase db;
    private final SqlitexMetaData metaData;
    private boolean enteredTransaction;

    SqlitexConnection(SQLiteDatabase db) {
        if(db == null) {
            throw new IllegalArgumentException("null db");
        }
        this.db = db;
        autoCommit = true;
        metaData = new SqlitexMetaData(this);
    }

    static void throwSQLException(SQLiteException exception) throws SQLException {
        if(exception instanceof SQLiteConstraintException) {
            throw new SQLIntegrityConstraintViolationException(exception);
        }
        throw new SQLException(exception);
    }

    SQLiteDatabase getDatabase() {
        return db;
    }

    @Override
    protected void ensureTransaction() {
        if (!autoCommit) {
            if (!db.inTransaction()) {
                db.beginTransaction();
                enteredTransaction = true;
            }
        }
    }

    @Override
    protected void execSQL(String sql) throws SQLException {
        try {
            db.execSQL(sql);
        } catch (SQLiteException e) {
            throwSQLException(e);
        }
    }

    @Override
    public void commit() throws SQLException {
        if (autoCommit) {
            throw new SQLException("commit called while in autoCommit mode");
        }
        if (db.inTransaction() && enteredTransaction) {
            try {
                db.setTransactionSuccessful();
            } catch (IllegalStateException e) {
                throw new SQLException(e);
            } finally {
                db.endTransaction();
                enteredTransaction = false;
            }
        }
    }

    @Override
    public Statement createStatement() throws SQLException {
        ensureTransaction();
        return new SqlitexStatement(this);
    }

    @Override
    public Statement createStatement(int resultSetType, int resultSetConcurrency)
            throws SQLException {
        ensureTransaction();
        return createStatement(resultSetType,
                resultSetConcurrency, ResultSet.HOLD_CURSORS_OVER_COMMIT);
    }

    @Override
    public Statement createStatement(int resultSetType, int resultSetConcurrency,
                                     int resultSetHoldability) throws SQLException {
        if (resultSetConcurrency == ResultSet.CONCUR_UPDATABLE) {
            throw new SQLFeatureNotSupportedException("CONCUR_UPDATABLE not supported");
        }
        ensureTransaction();
        return new SqlitexStatement(this);
    }

    @Override
    public DatabaseMetaData getMetaData() throws SQLException {
        return metaData;
    }

    @Override
    public boolean isClosed() throws SQLException {
        return !db.isOpen();
    }

    @Override
    public boolean isReadOnly() throws SQLException {
        return db.isReadOnly();
    }

    @Override
    public PreparedStatement prepareStatement(String sql, int autoGeneratedKeys)
        throws SQLException {
        ensureTransaction();
        return new SqlitexPreparedStatement(this, sql, autoGeneratedKeys);
    }

    @Override
    public PreparedStatement prepareStatement(String sql,
                                              int resultSetType,
                                              int resultSetConcurrency,
                                              int resultSetHoldability) throws SQLException {
        ensureTransaction();
        return new SqlitexPreparedStatement(this, sql, Statement.NO_GENERATED_KEYS);
    }

    @Override
    public PreparedStatement prepareStatement(String sql, String[] columnNames)
            throws SQLException {
        ensureTransaction();
        return new SqlitexPreparedStatement(this, sql, Statement.RETURN_GENERATED_KEYS);
    }

    @Override
    public void rollback() throws SQLException {
        if (autoCommit) {
            throw new SQLException("commit called while in autoCommit mode");
        }
        db.endTransaction();
    }
}
