/*
 * Copyright 2016 requery.io
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package io.requery.android.sqlite;

import android.annotation.SuppressLint;
import android.database.Cursor;
import android.database.sqlite.SQLiteStatement;

import java.sql.ResultSet;
import java.sql.SQLException;

/**
 * {@link java.sql.Statement} implementation using Android's local SQLite database.
 *
 * @author Nikhil Purushe
 */
class SqliteStatement extends BaseStatement {

    protected final SqliteConnection connection;

    SqliteStatement(SqliteConnection connection) {
        super(connection);
        this.connection = connection;
    }

    @Override
    public boolean execute(String sql, int autoGeneratedKeys) throws SQLException {
        SQLiteStatement statement = null;
        try {
            statement = connection.getDatabase().compileStatement(sql);
            if (autoGeneratedKeys == RETURN_GENERATED_KEYS) {
                long rowId = statement.executeInsert();
                insertResult = new SingleResultSet(this, rowId);
                return true;
            } else {
                statement.execute();
            }
        } catch (android.database.SQLException e) {
            SqliteConnection.throwSQLException(e);
        } finally {
            if (statement != null) {
                statement.close();
            }
        }
        return false;
    }

    @Override
    public ResultSet executeQuery(String sql) throws SQLException {
        throwIfClosed();
        try {
            @SuppressLint("Recycle") // released with the queryResult
            Cursor cursor = connection.getDatabase().rawQuery(sql, null);
            return queryResult = new CursorResultSet(this, cursor, true);
        } catch (android.database.SQLException e) {
            SqliteConnection.throwSQLException(e);
        }
        return null;
    }

    @Override
    public int executeUpdate(String sql, int autoGeneratedKeys) throws SQLException {
        SQLiteStatement statement = null;
        try {
            statement = connection.getDatabase().compileStatement(sql);
            if (autoGeneratedKeys == RETURN_GENERATED_KEYS) {
                long rowId = statement.executeInsert();
                insertResult = new SingleResultSet(this, rowId);
                updateCount = 1;
            } else {
                updateCount = statement.executeUpdateDelete();
            }
        } catch (android.database.SQLException e) {
            SqliteConnection.throwSQLException(e);
        } finally {
            if (statement != null) {
                statement.close();
            }
        }
        return updateCount;
    }
}
