/**
 * $Id$
 * 
 * SARL is an general-purpose agent programming language.
 * More details on http://www.sarl.io
 * 
 * Copyright (C) 2014-2021 the original authors or authors.
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package io.sarl.core;

import io.sarl.lang.annotation.SarlElementType;
import io.sarl.lang.annotation.SarlSpecification;
import io.sarl.lang.annotation.SyntheticMember;
import io.sarl.lang.core.Address;
import io.sarl.lang.core.Event;
import java.util.Objects;
import org.eclipse.xtext.xbase.lib.Pure;
import org.eclipse.xtext.xbase.lib.util.ToStringBuilder;

/**
 * Informs an agent has been killed.
 * The event is fires within all the external contexts (including the default context) of the killed agent.
 * The killed agent does not receive this event.
 * The source of the event is the killed agent.
 * The context id can be retrieved via {@code ev.source.spaceID.contextID}.
 * 
 * @author <a href="http://www.sebastianrodriguez.com.ar/">Sebastian Rodriguez</a>
 * @author <a href="http://www.ciad-lab.fr/author-10836/">St&eacute;phane Galland</a>
 * @version io.sarl.core 0.12.0 20210527-171007
 * @mavengroupid io.sarl
 * @mavenartifactid io.sarl.core
 */
@SarlSpecification("0.12")
@SarlElementType(15)
@SuppressWarnings("all")
public final class AgentKilled extends Event {
  /**
   * The fully qualified name of the type of the killed agent.
   */
  public final String agentType;
  
  /**
   * The object that is the cause of the killing of the agent or explains the cause
   * of the "abnormal" killing.
   * If it is {@code null}, the agent has committed a suicide in "normal" situation.
   * 
   * @since 0.12
   */
  public final Object terminationCause;
  
  /**
   * Construct the event.
   * 
   * @param source the address of the emitter.
   * @param agentType fully qualified name of the type of the agent.
   * @param terminationCause cause of the termination of the agent, or {@code null} if there is
   *      no specific termination cause.
   */
  public AgentKilled(final Address source, final String agentType, final Object terminationCause) {
    this.setSource(source);
    this.agentType = agentType;
    this.terminationCause = terminationCause;
  }
  
  @Override
  @Pure
  @SyntheticMember
  public boolean equals(final Object obj) {
    if (this == obj)
      return true;
    if (obj == null)
      return false;
    if (getClass() != obj.getClass())
      return false;
    AgentKilled other = (AgentKilled) obj;
    if (!Objects.equals(this.agentType, other.agentType))
      return false;
    return super.equals(obj);
  }
  
  @Override
  @Pure
  @SyntheticMember
  public int hashCode() {
    int result = super.hashCode();
    final int prime = 31;
    result = prime * result + Objects.hashCode(this.agentType);
    return result;
  }
  
  /**
   * Returns a String representation of the AgentKilled event's attributes only.
   */
  @SyntheticMember
  @Pure
  protected void toString(final ToStringBuilder builder) {
    super.toString(builder);
    builder.add("agentType", this.agentType);
    builder.add("terminationCause", this.terminationCause);
  }
  
  @SyntheticMember
  private static final long serialVersionUID = -885477460L;
}
